<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\Item;

use Blackbit\PimBundle\lib\Pim\Import\Javascript;
use Blackbit\PimBundle\lib\Pim\Shop\Helper;
use Blackbit\PimBundle\model\Dataport;
use Blackbit\PimBundle\model\Groupmapping;
use Blackbit\PimBundle\model\GroupmappingField;
use Blackbit\PimBundle\model\GroupmappingTarget;
use Blackbit\PimBundle\model\ImportStatus;
use Blackbit\PimBundle\model\RawItem;
use Pimcore\Config;
use Pimcore\Mail;
use Pimcore\Model\Document\Page;
use Pimcore\Tool;
use Pimcore\Logger;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\GenericEvent;

/**
 * Steuert den Import von Rohdaten in Pim-Artikel und -Produkte
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Medien GmbH, http://www.blackbit.de/
 */
class Importmanager {

	public static $statusKey = null;

	/** @var ImporterInterface */
    private $importer;

    /** @var LoggerInterface  */
    private $logger;

    /**
     * Importmanager constructor.
     *
     * @param ImporterInterface $importer
     * @param LoggerInterface   $logger
     */
    public function __construct(ImporterInterface $importer, LoggerInterface $logger, $ignoreHashCheck = false)
    {
        $this->importer = $importer;
        if(\method_exists($this->importer, 'setIgnoreHashCheck')) {
            $this->importer->setIgnoreHashCheck($ignoreHashCheck);
        }
        $this->logger = $logger;
    }

    public function importDataport($dataportId, $importType = ImportStatus::TYPE_PIM, $rawItemId = null) {
        $cacheCoreHandler = \Pimcore::getContainer()->get('pimcore.cache.core.handler');
        $cacheCoreHandler->setHandleCli(true);
        if(\method_exists($cacheCoreHandler, 'setMaxWriteToCacheItems')) {
            $cacheCoreHandler->setMaxWriteToCacheItems(PHP_INT_MAX);
        }
		$table = new Dataport();
		$dataport = $table->get($dataportId);

		if (!$dataport) {
			return false;
		}

		$statusModel = new Importstatus();
		$statusKey = uniqid();

		self::$statusKey = $statusKey;

        $startTime = new \DateTimeImmutable();
		$statusModel->create(array(
			'key' => $statusKey,
			'dataportId' => $dataportId,
			'startDate' => $startTime,
			'lastUpdate' => $startTime,
			'importType' => $importType,
		));

		$targetconfig = unserialize($dataport['targetconfig']);
		$javascriptEngine = $targetconfig['javascriptEngine'];

		if (empty($javascriptEngine) || !Javascript::isEngineAvailable($javascriptEngine)) {
            $this->logger->info('Importing data of dataport ' . $dataportId . ' without javascript support as no engine has been selected');
		}

		// Warengruppenzuordnung
		$groupMappingTable = new Groupmapping();
		$mappings = $groupMappingTable->find(array(
			'dataportId = ?' => $dataportId,
		));
		$groupMappings = array();

		$purgedDocuments = array();

		/**
		 * Mappings vorbereiten. Trifft ein Datensatz mit allen Rohdatenwerten auf ein Mapping zu,
		 * wird der Artikel an das angegebene Dokument angehängt.
		 * Ist das Flag "purgemappings" gesetzt, werden vor dem Import alle Artikelzuordnungen entfernt
		 */
		foreach ($mappings as $mapping) {
		    $groupMappingFieldRepository = new GroupmappingField();
			$fields = $groupMappingFieldRepository->find(['groupmappingId = ?' => $mapping['id']]);

			$groupMappingTargetFieldRepository = new GroupmappingTarget();
			$mappingTargets = $groupMappingTargetFieldRepository->find(['groupmappingId = ?' => $mapping['id']]);

			if (count($mappingTargets) === 0) {
				continue;
			}

			$targets = array();
			foreach ($mappingTargets as $mappingTarget) {
				$targets[] = array(
					'objectId' => $mappingTarget->documentId,
					'editable' => $mappingTarget->editable,
					'purge' => $mappingTarget->purgemappings,
				);
			}


			$condition = array();
			foreach ($fields as $field) {
				$fieldNo = $field['fieldNo'];
				$value = $field['value'];

				$condition['field_'.$fieldNo] = $value;
			}

			if (!empty($condition)) {
				$groupMappings[] = array(
					'conditions' => $condition,
					'target' => $targets,
				);
			}

		}

		// IDs der geänderten Objekte merken
		$changedObjectIds = [];

		// Rohdaten auslesen und aufteilen
		$rawItemTable = new RawItem();
		$itemCondition = array(
			'dataportId = ?' => $dataport['id'],
		);

		if (!empty($rawItemId)) {
			$itemCondition['id = ?'] = $rawItemId;
		}

		$count = 100;
		$total = $rawItemTable->countRows($itemCondition);

        $event = new GenericEvent($this, [
            'dataportId' => $dataportId,
            'rawitemCount' => $total,
        ]);
        \Pimcore::getEventDispatcher()->dispatch('pim.preImport', $event);

		$statusModel->update(array(
			'lastUpdate' => new \DateTime(),
			'totalItems' => $total,
		), ['key' => $statusKey]);

		$chunks = ceil($total / $count);
		for ($i = 0; $i < $chunks; $i++) {
			$offset = $i * $count;
			$rawItems = $rawItemTable->find($itemCondition, 'priority', $count, $offset);

			$this->importer->import($dataport, $rawItems, $groupMappings);
            
            $changedObjectIds = [];
            if(method_exists($this->importer, 'getChangedObjectIds')) {
			    $changedObjectIds = array_unique(array_merge($changedObjectIds, $this->importer->getChangedObjectIds()));
            }

			$done = ($i + 1) * $count;
			if ($done > $total) {
				$done = $total;
			}

			/**
			 * Ebenfalls Status und Endzeitpunkt zurücksetzen, falls dies zwischenzeitlich durch den Maintenance-Job als
			 * abgebrochener Prozess markiert wurde
			 */
			$statusModel->update(array(
				'status' => ImportStatus::STATUS_RUNNING,
				'endDate' => null,
				'lastUpdate' => new \DateTime(),
				'doneItems' => $done,
			), ['key' => $statusKey]);

            // Fire status event
            $event = new GenericEvent($this, [
                'dataportId' => $dataportId,
                'total' => $total,
                'done' => $done,
                'startTime' => $startTime,
                'changedObjects' => count($changedObjectIds)
            ]);

            \Pimcore::getEventDispatcher()->dispatch('pim.importstatus', $event);

            $cacheCoreHandler->writeSaveQueue();
		}

		// Workaround: Geleerte Dokumente noch einmal speichern, da sonst die Artikelzuzordnungen evt. unvollständig sind
		// Nach Speicherung des Dokuments im Backend ist die Zuordnung wieder okay, dieshalb wird hier diese Speicherung simuliert
		foreach ($purgedDocuments as $purgedDocumentId) {
			$document = Page::getById($purgedDocumentId);

			if ($document) {
                $document->setUserModification(0);
				$document->save();
			}
		}

		Helper::afterImport($dataportId);

        $dbNow = new \DateTime();
		$statusModel->update(array(
			'status' => ImportStatus::STATUS_FINISHED,
			'lastUpdate' => $dbNow,
			'endDate' => $dbNow,
			'doneItems' => $total,
		), ['key' => $statusKey]);


        $event = new GenericEvent($this, [
            'dataportId' => $dataportId,
            'changedObjectIds' => $changedObjectIds,
        ]);

        \Pimcore::getEventDispatcher()->dispatch('pim.importFinished', $event);

		return true;
	}
}
