<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\Import;


class Javascript
{
    const ENGINE_SPIDERMONKEY_LEGACY = 'spidermonkeyLegacy';
    const ENGINE_SPIDERMONKEY = 'spidermonkey';
    const ENGINE_V8JS = 'v8Js';
    const ENGINE_PHP = 'php';

    private static $preJS;

    public static function isEngineAvailable($engine)
    {
        switch ($engine) {
            case self::ENGINE_SPIDERMONKEY:
            case self::ENGINE_SPIDERMONKEY_LEGACY:
                return class_exists('JSContext');

            case self::ENGINE_V8JS:
                return class_exists('V8Js');
            case self::ENGINE_PHP:
                return true;

            default:
                return false;
        }
    }

    private static function getPreJS() {
        if(self::$preJS === null) {
            self::$preJS = file_get_contents(__DIR__ . '/../Item/libraries.js');
        }
        return self::$preJS;
    }

    /**
     * @param $code string Javascript code to execute
     * @param $engine string Javascirpt engine name
     * @param $params array
     *
     * @return mixed
     */
    public static function evaluateScript($code, $engine, $params)
    {
        switch ($engine) {
            case self::ENGINE_SPIDERMONKEY_LEGACY:
                return self::evaluateScriptSpidermonkeyLegacy(self::getPreJS()."\n\n".$code, $params);

            case self::ENGINE_SPIDERMONKEY:
                return self::evaluateScriptSpidermonkey(self::getPreJS()."\n\n".$code, $params);

            case self::ENGINE_V8JS:
                return self::evaluateScriptV8(self::getPreJS()."\n\n".$code, $params);

            case self::ENGINE_PHP:
                return self::evaluateScriptPHP($code, $params);

            default:
                return null;
        }
    }

    private static function evaluateScriptSpidermonkeyLegacy($js, $params)
    {
        $jsCtx = new \JSContext();

        foreach ($params as $key => $param) {
            $jsCtx->assign($key, $param);
        }

        return $jsCtx->evaluateScript($js);
    }

    private static function evaluateScriptSpidermonkey($js, $params)
    {

        $jsCtx = new \JSContext();
        $jsCtx->assign('input', $params);

        $code = [
            ';(function(params) {',
            $js,
            '})(input);'
        ];

        return $jsCtx->evaluateScript(implode(PHP_EOL, $code));
    }

    private static function evaluateScriptV8($js, $params)
    {

        $v8 = new \V8Js();
        $v8->input = $params;

        $code = [
            ';(function(params) {',
            $js,
            '})(PHP.input);'
        ];

        return $v8->executeString(implode(PHP_EOL, $code), 'V8Js::pimimport()', \V8Js::FLAG_FORCE_ARRAY);
    }

    private static function evaluateScriptPHP($code, $params)
    {
        $function = function($code, $params) {
            return eval($code);
        };

        return $function($code, $params);
    }

}