<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\Parser;

use Psr\Log\LoggerInterface;

/**
 * Parses CSV-Files according to given configuration
 */
class CsvParser implements Parser {
    use ResourceBasedParser;

    /** @var array */
	private $config;

	/** @var LoggerInterface */
	private $logger;

    public function __construct(array $config, LoggerInterface $logger)
    {
        $this->config = $config;
        $this->logger = $logger;
    }

    /**
     * Verarbeitet eine CSV-Datei anhand der übergebenen Konfiguration
     *
     * @param int|null $limit Maximale Anzahl an Einträgen, die zurückgeliefert werden sollen
     *
     * @return array Aus der Datei ausgelesene Einträge; max $limit, wenn $limit != null
     */
	public function parse($limit = null) {
	    $source = $this->config['file'];
        try {
            $filePathOrUrl = $this->getFileOrUrl($source);

            $handle = fopen($filePathOrUrl, 'rb');
            if(!\is_resource($handle)) {
                $this->logger->error('Could not find any data in "'.$source.'" (or it is not readable)');
            }
        } catch (\InvalidArgumentException $e) {
            $this->logger->warning($e->getMessage());
            return array();
        }

		if (empty($this->config['fields'])) {
			$this->logger->error("No field definition given in config");
			return array();
		}

		$fieldsToIndexes = array();

        $items = array();
		try {
			if ($handle !== false) {
			    if($this->config['separator'] === "\\t") {
                    $this->config['separator'] = "\t";
                }
				$first = true;
				while (($data = fgetcsv($handle, 0, $this->config['separator'], $this->config['quote'])) !== false) {
					if ($first) {
						$first = false;

						if ($this->config['hasHeader']) {
							foreach ($this->config['fields'] as $field => $values) {
								foreach ($data as $index => $name) {
									if (trim($name) === trim($values['column']) || (string)$index === $values['column']) {
										$fieldsToIndexes[$index] = $field;
									}
								}
							}

							continue;
						}

                        foreach ($this->config['fields'] as $field => $values) {
                            $colIndex = (int)$values['column'];
                            $fieldsToIndexes[$colIndex] = $field;
                        }
					}

					$item = array();
					foreach ($fieldsToIndexes as $index => $field) {
						$item[$field] = $data[$index];
					}

					$items[] = $item;

					if ($limit !== null && \count($items) >= $limit) {
						break;
					}
				}
			}
		} catch (\Exception $e) {
			$this->logger->error("Unable to parse CSV file \"{$source}\": " . $e);
			return array();
		}

        if(empty($limit)) {
            $this->archive($filePathOrUrl, $source);
        }

		return $items;
	}
}
