<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\model;

use Doctrine\DBAL\Query\QueryBuilder;
use Pimcore\Db;
use Pimcore\Db\Connection;

abstract class PimcoreDbRepository implements Repository
{
    /** @var Connection */
    protected $connection;

    /**
     * PimcoreDbRepository constructor.
     *
     * @param Connection $connection
     */
    public function __construct(Connection $connection = null)
    {
        $this->connection = $connection ?? Db::get();
    }

    public function find(array $where = [], string $order = null, int $count = null, int $offset = 0): array
    {
        $queryBuilder = new QueryBuilder($this->connection);
        $queryBuilder = $queryBuilder->select('*')->from($this->getTableName());
        if(!empty($where)) {
            $conditions = [];
            foreach($where as $condition => $value) {
                if(is_array($value)) {
                    $value = array_map(function ($item) {
                        return $this->connection->quote($item);
                    }, $value);
                    $conditions[] = str_replace('?', implode(',', $value), $condition);
                } else {
                    $conditions[] = str_replace('?', $this->connection->quote($value), $condition);
                }
            }
            $queryBuilder = $queryBuilder->where(implode(' AND ', $conditions));
        }
        if(!empty($order)) {
            $queryBuilder = $queryBuilder->add('orderBy', $order);
        }
        if($count > 0) {
            $queryBuilder = $queryBuilder->setMaxResults($count);
        }
        $queryBuilder = $queryBuilder->setFirstResult($offset);

        return $this->connection->fetchAll($queryBuilder->getSQL());
    }

    public function findOne(array $where = [], string $order = null, int $offset = 0): array {
        $result = $this->find($where, $order, 1, $offset);

        if(count($result) < 1) {
            return [];
        }

        return $result[0];
    }

    public function countRows(array $where = []) {
        $queryBuilder = new QueryBuilder($this->connection);
        $queryBuilder = $queryBuilder->select('COUNT(*)')->from($this->getTableName());
        if(!empty($where)) {
            $conditions = [];
            foreach($where as $condition => $value) {
                $conditions[] = str_replace('?', $this->connection->quote($value), $condition);
            }
            $queryBuilder = $queryBuilder->where(implode(' AND ', $conditions));
        }
        return $this->connection->fetchOne($queryBuilder->getSQL());
    }

    abstract protected function getTableName(): string;

    public function create(array $data)
    {
        $dataTypes = $this->getDataTypes($data);

        $insertedRows = $this->connection->insert($this->getTableName(), $data, $dataTypes);

        if ($insertedRows > 0) {
            $data['id'] = $this->connection->lastInsertId();
            return $data;
        }

        return false;
    }

    public function update($data, $where)
    {
        if(!is_array($where)) {
            $where = ['id' => $where];
        }

        $dataTypes = $this->getDataTypes($data);
        $dataTypes = array_merge($dataTypes, $this->getDataTypes($where));

        if (count($where) > 0) {
            $this->connection->update($this->getTableName(), $data, $where, $dataTypes);

            $quotedConditions = [];
            foreach($where as $column => $value) {
                $quotedConditions['`'.$column.'` = ?'] = $value;
            }

            return $this->find($quotedConditions);
        }

        return false;
    }

    public function delete($id)
    {
        if ((int)$id > 0) {
            return $this->deleteWhere(['id' => $id]);
        }

        return 0;
    }

    public function deleteWhere(array $where = [])
    {
        return $this->connection->delete($this->getTableName(), $where);
    }

    public function get($id): array {
        return $this->findOne(['id = ?' => (int)$id]);
    }

    public function beginTransaction() {
        $this->connection->beginTransaction();
    }

    public function commit() {
        $this->connection->commit();
    }

    public function rollback() {
        $this->connection->rollBack();
    }

    /**
     * @param array $data
     *
     * @return array
     */
    private function getDataTypes(array $data): array
    {
        $dataTypes = array_values(
            array_map(
                function ($item) {
                    if (is_int($item)) {
                        return \PDO::PARAM_INT;
                    }

                    if ($item instanceof \DateTimeInterface) {
                        return 'datetime';
                    }

                    return \PDO::PARAM_STR;
                }, $data
            )
        );

        return $dataTypes;
    }
}