<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\Controller;

use Blackbit\PimBundle\lib\Pim\Helper;
use Blackbit\PimBundle\model\Dataport;
use Blackbit\PimBundle\model\Fieldmapping;
use Blackbit\PimBundle\model\RawItemField;
use Pimcore\Controller\FrontendController;
use Pimcore\Tool;
use Pimcore\Translation\Translator;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/**
 * Class MappingconfigController
 *
 * @package BlackbitPimBundle\Controller
 * @Route("/mappingconfig")
 */
class MappingconfigController extends FrontendController {
    
    /** @var Translator */
    private $pimTranslator;
    
    protected function getTranslator(Request $request) {
        if($this->pimTranslator === null) {
            $locale = $request->getLocale();
            
            if (!$locale) {
                $locale = 'de';
            }
            
            $this->pimTranslator = $this->get('translator');
            $this->pimTranslator->setLocale($locale);
        }
        
        return $this->pimTranslator;
    }
    
    /**
     * @Route("/get-dataports")
     */
	public function getDataportsAction() {
		$dataportList = new Dataport();
		$dataports = $dataportList->find();
		$result = array();

		foreach ($dataports as $dataport) {
			$result[] = $this->getDataportData($dataport);
		}

		return new JsonResponse($result);
	}

	public function getDataportAction(Request $request) {
		$id = $request->get('id');
		$dataportList = new Dataport();
		$dataport = $dataportList->get($id);

        return new JsonResponse($dataport);
	}

    /**
     * @Route("/get-config/{dataportId}")
     */
	public function getConfigAction(Request $request, int $dataportId) {
		$config = array(
			'success' => false,
		);

		$table = new Dataport();
		$dataport = $table->get($dataportId);

		if ($dataport) {
			$config['success'] = true;
			$config['dataport'] = array(
				'id' => $dataport['id'],
				'name' => $dataport['name'],
			);

			$config['mapping'] = Helper::createFieldMappings($dataport, $this->getTranslator($request));
		}

        return new JsonResponse($config);
	}

    /**
     * @Route("/save-config")
     */
	public function saveConfigAction(Request $request) {
        $dataportId = (int)$request->get('dataportId');
		$mappings = json_decode($request->get('mapping'));
		$omitReturnMapping = $request->get('omitReturnMapping') == '1';

		$config = array(
			'success' => false,
		);

		$table = new Dataport();
		$dataport = $table->get($dataportId);

		if ($dataport && $mappings) {
			if (!is_array($mappings)) {
				$mappings = array($mappings);
			}

			foreach ($mappings as $mapping) {
				$key = $mapping->{'attribute.key'};
				$field = $mapping->field;
				$type = $mapping->type;
				$settings = $mapping->settings;
				$localized = !empty($mapping->locale);

				$keyMapping = $settings->keyMapping === true;
				$calculation = $settings->calculation;

				$defaults = Helper::getMappingDefaults();
                $format = [];
                $settings->format = (array)$settings->format;
				if (array_key_exists($type, $defaults)) {
                    $format = $defaults[$type];

                    foreach ($settings->format as $attr => $value) {
                        $format[$attr] = $value;
                    }
				}
                $format['writeProtected'] = $settings->format['writeProtected'];

				$attributeName = $key;
				$attributeLanguage = null;

				if ($localized === true && strpos($key, '#') !== false) {
					$parts = explode('#', $key);
					$attributeName = $parts[0];
					$attributeLanguage = $parts[1];
				}

				$validLocales = Tool::getValidLanguages();
				if (!in_array($attributeLanguage, $validLocales)) {
					$attributeLanguage = null;
				}

                $brickName = $mapping->brickName ?? null;
                $targetBrickField = $mapping->targetBrickField ?? null;

				$condition = array(
					'dataportId = ?' => $dataport['id'],
					'fieldName = ?' => $attributeName,
				);

				if ($attributeLanguage != null) {
					$condition['locale = ?'] = $attributeLanguage;
				}

				$mappingTable = new Fieldmapping();
				$row = $mappingTable->findOne($condition);

                if(!empty($field) || !empty($calculation)) {
                    if ($row) {
                        $mappingTable->update(array(
                            'fieldNo' => $field,
                            'format' => serialize($format),
                            'calculation' => $calculation,
                            'keyMapping' => $keyMapping && $attributeLanguage == null, // Sprachabhängige Attribute können nicht Schlüssel sein
                            'brickName' => $brickName,
                            'targetBrickField' => $targetBrickField
                        ), array('dataportId' => $dataport['id'], 'fieldName' => $attributeName, 'locale' => $attributeLanguage == null ? '' : $attributeLanguage));
                    } else {
                        $mappingTable->create(array(
                            'dataportId' => $dataport['id'],
                            'fieldName' => $attributeName,
                            'fieldNo' => $field,
                            'format' => serialize($format),
                            'calculation' => $calculation,
                            'keyMapping' => $keyMapping && $attributeLanguage == null, // Sprachabhängige Attribute können nicht Schlüssel sein
                            'locale' => $attributeLanguage == null ? '' : $attributeLanguage,
                            'brickName' => $brickName,
                            'targetBrickField' => $targetBrickField
                        ));
                    }
                } else {
                    $mappingTable->deleteWhere([
                        'dataportId' => $dataport['id'],
                        'fieldName' => $attributeName,
                        'locale' => $attributeLanguage == null ? '' : $attributeLanguage,
                    ]);
                }
			}

			$config['success'] = true;
			if (!$omitReturnMapping) {
				$config['mapping'] = Helper::createFieldMappings($dataport, $this->getTranslator($request));
			}
		}

		return new JsonResponse($config);
	}

    /**
     * @Route("/rawdata-fields/{id}")
     */
	public function rawdataFieldsAction(int $id) {
		$data = array();

		$rawItemField = new RawItemField();

		$rows = $rawItemField->find(array(
			'dataportId = ?' => $id
		));

		foreach ($rows as $row) {
			$data[$row['fieldNo']] = $row['name'];
		}

		return new JsonResponse(array(
			'success' => true,
			'fields' => $data,
		));
	}

	/**
	 * @param array $dataport
	 * @return array
	 */
	private function getDataportData($dataport) {
		$rawItemField = new RawItemField();

		$rows = $rawItemField->find(array(
			'dataportId = ?' => $dataport['id']
		));

		$fields = array();

		foreach ($rows as $row) {
			$fields[] = array(
				'id' => 'd' . $dataport['id'] . '-field' . $row['fieldNo'],
				'text' => $row['fieldNo'] . ': ' . $row['name'],
				'leaf' => true,
				'iconCls' => 'pim_icon_textfield_rename'
			);
		}

		return array(
            'id'       => $dataport['id'],
            'text'     => $dataport['id'].': '.$dataport['name'],
            'icon'     =>'',
            'children' => $fields,
            'leaf'     => empty($fields)
		);
	}
}
