<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Blackbit\PimBundle\Controller;

use Blackbit\PimBundle\lib\Pim\Variant\VariantGenerator;
use Blackbit\PimBundle\lib\Pim\Variant\VariantHelper;
use Blackbit\PimBundle\model\ResourceFactory;
use Pimcore\Controller\FrontendController;
use Pimcore\Tool\Console;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Command\LockableTrait;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Lock\Store\FlockStore;
use Symfony\Component\Lock\Factory as LockFactory;
use Symfony\Component\Routing\Annotation\Route;

/**
 * Class VariantgeneratorController
 *
 * @package Blackbit\PimBundle\Controller
 *
 * @Route("/variantgenerator")
 */
class VariantgeneratorController extends FrontendController
{
    const LOCK_ID = 'variantGeneratorLock';

    use LockableTrait;

    /**
     * @var ResourceFactory
     */
    private $resourceFactory;

    /**
     * @var VariantHelper
     */
    private $variantHelper;

    /**
     * @var VariantGenerator
     */
    private $variantGenerator;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * VariantconfigController constructor.
     *
     * @param ResourceFactory  $resourceFactory
     * @param VariantHelper    $variantHelper
     * @param VariantGenerator $variantGenerator
     * @param LoggerInterface  $logger
     */
    public function __construct(
        ResourceFactory $resourceFactory,
        VariantHelper $variantHelper,
        VariantGenerator $variantGenerator,
        LoggerInterface $logger
    ) {
        $this->resourceFactory = $resourceFactory;
        $this->variantHelper = $variantHelper;
        $this->variantGenerator = $variantGenerator;
        $this->logger = $logger;
    }

    /**
     * @Route("/run")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function runAction(Request $request): JsonResponse
    {
        $response = ['success' => false];

        try {
            if (true === (bool) Console::getPhpCli()) {
                Console::execInBackground(
                    escapeshellcmd(
                        '"' . Console::getPhpCli() . '" '
                        . realpath(PIMCORE_PROJECT_ROOT . DIRECTORY_SEPARATOR . 'bin' . DIRECTORY_SEPARATOR  . 'console')
                        . ' import:variants'
                    )
                );

                $response['success'] = true;
            }
        } catch (\Exception $e) {
            $response['msg'] = 'pim.manual.error.noPhpCli';
        }

        return new JsonResponse($response);
    }

    /**
     * @Route("/add")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function addAction(Request $request): JsonResponse
    {
        $variantId = (int)$request->get('variantId');
        $sourceType = $request->get('type');

        try {
            $children = $this->variantHelper->getChildIds($variantId, $sourceType, []);

            foreach ($children as $childId) {
                if ($this->resourceFactory->variantGenerator()->countRows(['variantId = ?' => $childId]) > 0) {
                    $this->resourceFactory->variantGenerator()->update(
                        [
                            'state' => \Blackbit\PimBundle\model\VariantGenerator::STATE_PENDING,
                        ],
                        ['variantId' => $childId]
                    );
                } else {
                    $this->resourceFactory->variantGenerator()->create([
                        'variantId' => $childId,
                        'state' => \Blackbit\PimBundle\model\VariantGenerator::STATE_PENDING,
                    ]);
                }
            }
        } catch (\Exception $exception) {
            dump($exception);
        }

        return new JsonResponse(['success' => true]);
    }

    /**
     * @Route("/state")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function stateAction(Request $request): JsonResponse
    {
        $running = true;
        if ($this->lock(self::LOCK_ID)) {
            $running = false;
            $this->release();
        }
        return new JsonResponse([
            'success' => true,
            'running' => $running,
        ]);
    }

    /**
     * @Route("/stategrid")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function stategridAction(Request $request): JsonResponse
    {
        $items = $this->resourceFactory->variantGenerator()->find([]);

        return new JsonResponse($items);
    }

    /**
     * @Route("/generate")
     * @param Request $request
     *
     * @return JsonResponse
     * @throws \Exception
     */
    public function generateAction(Request $request): JsonResponse
    {
        $response = ['success' => true];

        try {
            $this->variantGenerator->generate((int)$request->get('id'));
        } catch (\Exception $exception) {
            $response['success'] = false;
        }

        return new JsonResponse($response);
    }
}
