/*
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * GUI für FlexibleTable im Objekt-Editor
 */

pimcore.registerNS("pimcore.object.tags.flexibleTable");
pimcore.object.tags.flexibleTable = Class.create(pimcore.object.tags.abstract, {

    type: "flexibleTable",
    dataChanged:false,

    initialize: function (data, fieldConfig) {
        this.data = [];
        this.fieldConfig = fieldConfig;

        if (!data) {
            data = [];
        }
        this.data = data;

        var fields = [];

        for(var i = 0; i < fieldConfig.cols.length; i++) {
            var field = {name:fieldConfig.cols[i].key};
            if(fieldConfig.cols[i].type == "number") {
                field.type = "float";
            }
            if(fieldConfig.cols[i].type == "bool") {
                field.type = "bool";
            }
            fields.push(field);
        }

        this.store = new Ext.data.JsonStore({
            fields: fields
        });

        this.store.loadData(this.data);
    },

    getGridColumnConfig: function(field) {
        return {
			header: ts(field.label),
			width: 150,
			sortable: false,
			dataIndex: field.key,
			renderer: function (key, field, value, metaData, record) {
				if(record.data.inheritedFields[key] && record.data.inheritedFields[key].inherited == true) {
					metaData.css += " grid_value_inherited";
				}

				var rows = Object.keys(value);

				if (rows && rows.length > 0) {
					var keys = [];
					var table = '<table cellpadding="2" cellspacing="0" border="1">';

					//column headlines
					table += '<tr>';
					for (var c = 0; c < field.layout.cols.length; c++) {
						table += '<td style="font-weight: bold;">' + ts(field.layout.cols[c].label) + '</td>';
						keys.push(field.layout.cols[c].key);
					}
					table += '</tr>';

					//row values
					for (var i = 0; i < rows.length; i++) {
						var row = value[rows[i]];
						var rowKeys = Object.keys(row);
						var diff = rowKeys.filter(function(i) {
							return !(keys.indexOf(i) > -1);
						});

						if (diff.length == 0) {
							table += '<tr>';
	//						table += '<td>' + ts(field.layout.rows[i].label) + '</td>';
							for (var c = 0; c < rowKeys.length; c++) {
								table += '<td>' + row[rowKeys[c]] + '</td>';
							}
							table += '</tr>';
						}
					}
					table += '</table>';
					return table;
				}
				return "";
			}.bind(this, field.key, field)
		};
    },

    getLayoutEdit: function () {

        var autoHeight = false;
        if (intval(this.fieldConfig.height) < 15) {
            autoHeight = true;
        }

        var columns = [];

        for(var i = 0; i < this.fieldConfig.cols.length; i++) {

            var editor = null;
            var renderer = null;
            var listeners = null;
            if(this.fieldConfig.cols[i].type == "number") {
                editor = new Ext.form.NumberField({});
            } else if(this.fieldConfig.cols[i].type == "text") {
                editor = new Ext.form.TextField({});
            } else if(this.fieldConfig.cols[i].type == "bool") {
                editor = new Ext.form.Checkbox();
                renderer = function (value, metaData, record, rowIndex, colIndex, store) {
                    metaData.css += ' x-grid3-check-col-td';
                    return String.format('<div class="x-grid3-check-col{0}" style="background-position:10px center;">&#160;</div>', value ? '-on' : '');
                };
                listeners = {
                    "mousedown": function (col, grid, rowIndex, event) {
                        var store = grid.getStore();
                        var record = store.getAt(rowIndex);
                        record.set(col.dataIndex, !record.data[col.dataIndex]);
                        this.dataChanged = true;
                    }.bind(this)
                }
            }

            columns.push({
                header: ts(this.fieldConfig.cols[i].label),
                width: this.fieldConfig.cols[i].width,
                sortable: false,
                dataIndex: this.fieldConfig.cols[i].key,
                editor: editor,
                listeners: listeners,
                renderer: renderer
            });
        }
        this.component = new Ext.grid.GridPanel({
            store: this.store,
            colModel: new Ext.grid.ColumnModel({
                defaults: {
                    sortable: false
                },
                columns: columns
            }),
            cls: 'object_field',
            width: this.fieldConfig.width,
            height: this.fieldConfig.height,
            tbar: [
                {
                    xtype: "tbtext",
                    text: "<b>" + this.fieldConfig.title + "</b>"
                },
				{
					iconCls: "pimcore_icon_table_row",
					handler: this.addRow.bind(this)
				},
//                "->",
                {
                    xtype: "button",
                    iconCls: "pimcore_icon_delete",
                    handler: this.removeRow.bind(this)
                }
            ],
            autoHeight: autoHeight,
            bodyCssClass: "pimcore_object_tag_objects"
        });

        this.component.on("afteredit", function() {
            this.dataChanged = true;
        }.bind(this));

        this.component.reference = this;

        return this.component;
    },

    getLayoutShow: function () {

        var autoHeight = false;
        if (intval(this.fieldConfig.height) < 15) {
            autoHeight = true;
        }

        var cls = 'object_field';

        var columns = [];

        for(var i = 0; i < this.fieldConfig.cols.length; i++) {
            columns.push({header: ts(this.fieldConfig.cols[i].label), width: 120, sortable: false, dataIndex: this.fieldConfig.cols[i].key, editor: null});
        }
        this.component = new Ext.grid.GridPanel({
            store: this.store,
            columns: columns,
            cls: cls,
            width: this.fieldConfig.width,
            height: this.fieldConfig.height,
            tbar: [
                {
                    xtype: "tbspacer",
                    width: 20,
                    height: 16
                },
                {
                    xtype: "tbtext",
                    text: "<b>" + this.fieldConfig.title + "</b>"
                }
            ],
            autoHeight: autoHeight,
            bodyCssClass: "pimcore_object_tag_objects"
        });

        return this.component;
    },

    empty: function () {
        for(var i = 0; i < this.data.length; i++) {
            for(var j = 0; j < this.fieldConfig.cols.length; j++) {
                this.data[i][this.fieldConfig.cols[j].key] = null;
            }
        }
        this.store.loadData(this.data);
        this.dataChanged = true;
    },

    isInvalidMandatory: function () {
        var empty = true;

        this.store.each(function(record) {
            for(var j = 0; j < this.fieldConfig.cols.length; j++) {
                if(record.data[this.fieldConfig.cols[j].key] != null && record.data[this.fieldConfig.cols[j].key] != "") {
                    empty = false;
                }
            }
        }.bind(this));

        return empty;
    },

    getValue: function () {
        var tmData = [];

        var data = this.store.queryBy(function(record, id) {
            record.commit();
            return true;
        });

        for (var i = 0; i < data.items.length; i++) {
            tmData.push(data.items[i].data);
        }

		return tmData;
    },

    getName: function () {
        return this.fieldConfig.name;
    },


    isDirty: function() {
        if(!this.isRendered()) {
            return false;
        }
        
        return this.dataChanged;
    },

	addRow: function  () {
		var initData = {};

		for(var j = 0; j < this.fieldConfig.cols.length; j++) {
			initData[this.fieldConfig.cols[j].key] = null;
		}

		this.store.add(new this.store.recordType(initData, this.store.getCount() + 1));
		this.dataChanged = true;
	},

	removeRow : function  () {
		var selected = this.component.getSelectionModel();
		if (selected.selection) {
			this.store.remove(selected.selection.record);
			this.dataChanged = true;
		}
	}
});