<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Blackbit\PimBundle\lib\Pim\Variant;

use Blackbit\PimBundle\model\ResourceFactory;
use Pimcore\Model\DataObject\Service;

/**
 * Class VariantHelper
 *
 * @package Blackbit\PimBundle\lib\Pim\Variant
 */
class VariantHelper
{
    /**
     * @var ResourceFactory
     */
    private $resourceFactory;

    /**
     * VariantHelper constructor.
     *
     * @param ResourceFactory $resourceFactory
     */
    public function __construct(ResourceFactory $resourceFactory)
    {
        $this->resourceFactory = $resourceFactory;
    }

    /**
     * @param int $id
     *
     * @return array
     */
    public function getRawitemfieldConfig(int $id): array
    {
        $data = [];

        $variant = $this->resourceFactory->variant()->get($id);
        $sourcetype = $variant['sourcetype'];
        $fieldConfig = unserialize($variant['sourceconfig']);
        $fields = $this->resourceFactory->variantItemField()->find(['dataportId = ?' => $id], 'priority, fieldNo');

        $isMultivalue = [];
        if (is_array($fieldConfig['fields'])) {
            foreach ($fieldConfig['fields'] as $key => $values) {
                if (isset($values['multiValues']) && $values['multiValues'] === true) {
                    $isMultivalue[] = $key;
                }
            }
        }

        foreach ($fields as $field) {
            $key = 'field_' . $field['fieldNo'];
            $fieldData = [
                'fieldNo' => $field['fieldNo'],
                'name' => $field['name'],
            ];

            foreach ($this->getSourceConfigFields($sourcetype) as $config => $defaultValue) {
                $fieldData[$config] = $defaultValue;
                if (is_array($fieldConfig['fields'][$key]) && array_key_exists($config, $fieldConfig['fields'][$key])) {
                    $fieldData[$config] = $fieldConfig['fields'][$key][$config];
                }
            }

            $data[] = $fieldData;
        }

        return $data;
    }

    /**
     * @param $sourceType
     *
     * @return array
     */
    public function getSourceConfigFields($sourceType): array
    {
        $result = [];
        $fields = $this->getSourceConfig($sourceType);
        foreach ($fields as $field) {
            $result[$field['name']] = '';
        }

        return $result;
    }

    /**
     * @param $sourceType
     *
     * @return array
     */
    public function getSourceConfig($sourceType): array
    {
        $result = [];

        $variantConfigs = $this->resourceFactory->dataport()->find(['name LIKE ?' => $sourceType . '%']);
        $fieldConfig = unserialize($variantConfigs[0]['sourceconfig']);
        $fields = $this->resourceFactory->rawItemField()->find(['dataportId = ?' => $variantConfigs[0]['id']]);

        foreach ($fields as $field) {
            $key = 'field_' . $field['fieldNo'];

            $fieldData = [
                'fieldNo' => $field['fieldNo'],
                'name' => $field['name'],
                'type' => $fieldConfig['fields'][$key]['type'],
            ];

            $result[] = $fieldData;
        }

        return $result;
    }

    /**
     * @param $name
     *
     * @return mixed|string
     */
    public function sanitizeName($name): string
    {
        return Service::getValidKey(trim($name), 'object');
    }

    /**
     * Checks if the given name is still available.
     *
     * @param $name string Name to check
     * @param $id int (Optional) ID of a dataport to exclude from check
     *
     * @return bool true if it is, false otherwise
     */
    public function checkDataportName($name, $id = null): bool
    {
        $condition = ['name = ?' => $name];
        if ($id !== null) {
            $condition['id != ?'] = $id;
        }

        return !($this->resourceFactory->variant()->countRows($condition) > 0);
    }

    /**
     * @param string $sourceType
     * @param string $mode
     *
     * @return int
     * @throws \Exception
     */
    public function getMapperId(string $sourceType, string $mode): int
    {
        return $this->getMapperConfig($sourceType, $mode)['id'];
    }

    /**
     * @param string $sourceType
     * @param string $mode
     *
     * @return array
     * @throws \Exception
     */
    public function getMapperConfig(string $sourceType, string $mode): array
    {
        $mapperDataport = $this->resourceFactory->dataport()->findOne(['name = ?' => $sourceType . '-' . $mode]);
        if (null === $mapperDataport || !isset($mapperDataport['id'])) {
            throw new \Exception('Mapping Error: Could not find mapper for: ' . $sourceType . '-' . $mode);
        }

        return $mapperDataport;
    }

    /**
     * @param int $variantId
     *
     * @return bool
     */
    public function hasChildren(int $variantId): bool
    {
        return ($this->resourceFactory->variant()->countRows(['parent = ?' => $variantId]) > 0);
    }

    /**
     * @param int    $variantId
     * @param string $sourceType
     * @param array  $childIds
     *
     * @return array
     */
    public function getChildIds(int $variantId, string $sourceType, array $childIds): array
    {
        if ($sourceType != 'folder' && !$this->hasChildren($variantId)) {
            $childIds[] = $variantId;
        }

        foreach ($this->resourceFactory->variant()->find(['parent = ?' => $variantId]) as $child) {
            $childIds = $this->getChildIds($child['id'], $child['sourcetype'], $childIds);
        }

        return $childIds;
    }
}
