<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\Controller;

use Blackbit\PimBundle\model\RawItemData;
use Blackbit\PimBundle\Tools\Installer;
use Blackbit\PimBundle\lib\Pim\Feature;
use Blackbit\PimBundle\model\Categorymapping;
use Blackbit\PimBundle\model\CategorymappingField;
use Blackbit\PimBundle\model\Dataport;
use Blackbit\PimBundle\model\Fieldmapping;
use Blackbit\PimBundle\model\Groupmapping;
use Blackbit\PimBundle\model\GroupmappingField;
use Blackbit\PimBundle\model\GroupmappingTarget;
use Blackbit\PimBundle\model\RawItem;
use Blackbit\PimBundle\model\RawItemField;
use Blackbit\PimBundle\model\Repository;
use Doctrine\DBAL\DBALException;
use Blackbit\PimBundle\lib\Pim\Import\Javascript;
use Pimcore\Controller\FrontendController;
use Pimcore\Db;
use Pimcore\Model\Asset;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\ClassDefinition;
use Pimcore\Translation\Translator;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Pimcore\Model\DataObject\Service;

/**
 * Class ImportconfigController
 *
 * @package BlackbitPimBundle\Controller
 * @Route("/importconfig")
 */
class ImportconfigController extends FrontendController {
	const DEFAULT_SOURCETYPE = 'xml';
	const MODE_CREATE = 1<<0;
	const MODE_EDIT = 1<<1;

	/** @var Translator */
	private $pimTranslator;

	/** @var LoggerInterface */
	private $logger;

	/**
	 * Default values for dataport
	 * @var array
	 */
	public static $sourceconfigDefaults = array(
		'xml' => array(
			'file' => null,
            'assetSource' => null,
			'itemxpath' => '/',
			'keyfield' => 1,
			'fields' => array(),
		),
		'csv' => array(
			'file' => null,
			'assetSource' => null,
			'hasHeader' => true,
			'separator' => ',',
			'quote' => '"',
			'keyfield' => 1,
			'fields' => array(),
		),
        'excel' => array(
            'file' => null,
            'assetSource' => null,
            'hasHeader' => true,
            'sheet' => 1,
            'fields' => array(),
        ),
        'pimcore' => array(
            'sourceClass' => null,
        ),
        'variant' => [],
        'bmecat' => array(
			'file' => null,
			'version' => '1.2',
			'assetSource' => null,
			'fields' => array(),
			'categoryClassId' => null,
			'categoryProductElement' => null,
			'categoryAttributeElement' => null,
			'categoryNameElement' => null,
			'itemAttributeElement' => null,
			'categoryfolder' => '/',
		)
	);

	/**
	 * Default values for each field in a dataport
	 * @var array
	 */
	public static $sourceconfigFields = array(
		'xml' => array(
			'xpath' => '',
			'multiValues' => false,
		),
		'csv' => array(
			'column' => '',
        ),
        'excel' => array(
            'column' => '',
        ),
        'pimcore' => array(
            'method' => '',
            'parameters' => ''
        ),
        'variant' => [
            'type' => '',
        ],
	);

	public static $targetconfigDefaults = array(
	    'mode' => self::MODE_EDIT | self::MODE_CREATE,
		'itemClass' => null,
		'itemFolder' => '/',
		'javascriptEngine' => Javascript::ENGINE_V8JS,
		'assetFolder' => '/',
		'idPrefix' => '',
		'categoryClass' => '',
		'fieldnameProducts' => '',
        'importerClass' => ''
	);

    public function __construct(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    private function getTranslator(Request $request) {
        if($this->pimTranslator === null) {
            $locale = $request->getLocale();

            if (!$locale) {
                $locale = 'de';
            }

            $this->pimTranslator = $this->get('translator');
            $this->pimTranslator->setLocale($locale);
        }

        return $this->pimTranslator;
    }

    /**
     *  @Route("/get-dataports")
     */
    public function getDataportsAction() {
        $dataportList = new Dataport();
        $dataports = $dataportList->find();
        $result = array();

        foreach ($dataports as $dataport) {
	        $result[] = array(
                'id'   => $dataport['id'],
                'text' => $dataport['id'] . ': ' .$dataport['name'],
                'icon' => '',
            );

        }

        return new JsonResponse($result);
    }

    /**
     * @Route("/get")
     */
	public function getAction(Request $request) {
		$id = (int)$request->get('id');

		$table = new Dataport();
		$dataport = $table->get($id);

		$response = array(
			'success' => false
		);
		
		if ($dataport) {
			$response['data'] = $dataport;
			$targetConfig = unserialize($response['data']['targetconfig']);

			$sourceConfig = unserialize($response['data']['sourceconfig']);
			if (!is_array($sourceConfig)) {
				$sourceConfig = array();
			}

			$response['data']['sourceconfig'] = array_merge(self::$sourceconfigDefaults[$response['data']['sourcetype']], $sourceConfig);
            $response['data']['mode'] = $targetConfig['mode'];
			$response['data']['targetclass'] = $targetConfig['itemClass'];
			$response['data']['targetfolder'] = $targetConfig['itemFolder'];
			$response['data']['assetfolder'] = $targetConfig['assetFolder'];
            $response['data']['importerClass'] = $targetConfig['importerClass'];
			$response['data']['javascriptEngine'] = $targetConfig['javascriptEngine'];
			$response['data']['idPrefix'] = $targetConfig['idPrefix'];
			$response['data']['categoryClass'] = $targetConfig['categoryClass'];
			$response['data']['fieldnameProducts'] = $targetConfig['fieldnameProducts'];

			$response['success'] = true;
		} else {
			$response['errorMessage'] = 'pim_dataport_notfound';
		}

		return new JsonResponse($response);
	}

    /**
     *  @Route("/add")
     */
	public function addAction(Request $request) {
		$name = $this->sanitizeName($request->get('name'));

		if (!$this->checkDataportName($name)) {
			$response = array(
				'success' => false,
				'errorMessage' => 'pim_dataport_name_taken',
			);
			return new JsonResponse($response);
		}


		$targetConfig = self::$targetconfigDefaults;

		$table = new Dataport();
		$dataport = $table->create(array(
			'name' => $name,
			'sourcetype' => self::DEFAULT_SOURCETYPE,
			'sourceconfig' => serialize(self::$sourceconfigDefaults[self::DEFAULT_SOURCETYPE]),
			'targetconfig' => serialize(self::$targetconfigDefaults),
		));

//		$targetConfig['idPrefix'] = $dataport['id'] . '-';
		$dataport = $table->update(array(
			'targetconfig' => serialize($targetConfig),
		), $dataport['id']);

		$response = array(
			'success' => true,
			'dataport' => $dataport,
		);

		return new JsonResponse($response);
	}

    /**
     * @Route("/clone")
     */
	public function cloneAction(Request $request) {
		$name = $this->sanitizeName($request->get('name'));
		$dataportId = (int)$request->get('dataportId');

		if (empty($name)) {
			$response = array(
				'success' => false,
				'errorMessage' => 'pim.dataport.noname',
			);
			return new JsonResponse($response);
		}

		if (!$this->checkDataportName($name)) {
			$response = array(
				'success' => false,
				'errorMessage' => 'pim.dataport.nametaken',
			);
            return new JsonResponse($response);
		}

		$table = new Dataport();
        $data = $table->get($dataportId);

		if (empty($data)) {
			$response = array(
				'success' => false,
				'errorMessage' => 'pim.dataport.invalidid',
			);
			return new JsonResponse($response);
		}

		unset($data['id']);
		$data['name'] = $name;
		$clone = $table->create($data);
		$cloneDataportId = $clone['id'];

		/**
		 * fieldmapping
		 * groupmapping
		 * rawitemfield
		 *
		 */


		$fieldMappingModel = new Fieldmapping();
		$this->cloneModel($fieldMappingModel, $dataportId, $cloneDataportId);

		$rawitemFieldModel = new RawItemField();
		$this->cloneModel($rawitemFieldModel, $dataportId, $cloneDataportId);


		$groupMappingModel = new Groupmapping();
		$groupMappingFieldModel = new GroupmappingField();
		$groupMappingTargetModel = new GroupmappingTarget();

		$groupMappingRows = $groupMappingModel->find(array(
			'dataportId' => $dataportId,
		));
		foreach ($groupMappingRows as $rowData) {
			$oldId = $rowData['id'];
			unset($rowData['id']);
			$rowData['dataportId'] = $cloneDataportId;
			$newMapping = $groupMappingModel->create($rowData);

			// groupmappingfield
            $fieldData = $groupMappingFieldModel->find(array('groupmappingId = ?' => $oldId));
			foreach ($fieldData as $f) {
				$f['groupmappingId'] = $newMapping['id'];
				$groupMappingFieldModel->create($f);
			}

			// groupmappingtarget
            $targetData = $groupMappingTargetModel->find(array('groupmappingId = ?' => $oldId));
			foreach ($targetData as $t) {
				$t['groupmappingId'] = $newMapping['id'];
				$groupMappingTargetModel->create($t);
			}
		}


		$categoryMappingModel = new Categorymapping();
		$categoryMappingfieldModel = new CategorymappingField();

		$categoryMapping = $categoryMappingModel->find(array('dataportId = ?' => $dataportId));
		foreach ($categoryMapping as $c) {
			$oldId = $c['id'];
			unset($c['id']);
			$c['dataportId'] = $cloneDataportId;
			$newMapping = $categoryMappingModel->create($c);

            $mappingFieldData = $categoryMappingfieldModel->find(array('id = ?' => $oldId));
			foreach ($mappingFieldData as $f) {
				$f['id'] = $newMapping['id'];
				$categoryMappingfieldModel->create($f);
			}
		}




		$response = array(
			'success' => true,
			'dataport' => $clone,
		);

		return new JsonResponse($response);
	}

	/**
	 * @param $model
	 * @param $oldDataportId integer
	 * @param $newDataportId integer
	 */
	private function cloneModel(Repository $model, $oldDataportId, $newDataportId) {
        $existingFields = $model->find(array(
			'dataportId = ?' => $oldDataportId,
		));

		foreach ($existingFields as $f) {
			$f['dataportId'] = $newDataportId;
			$model->create($f);
		}
	}

    /**
     * @Route("/delete/{id}")
     */
	public function deleteAction(int $id) {
	    // delete raw items, otherwise the foreign key restricts deleting the dataport
        $rawItems = new RawItem();
        $rawItems->delete($id);

		$table = new Dataport();
		$table->delete($id);

		return new Response();
	}

    /**
     * @Route("/update")
     */
	public function updateAction(Request $request) {
		$id = (int)$request->get('id');
		$table = new Dataport();
		$dataport = $table->get($id);


		$response = array(
			'success' => false
		);

		if (!$dataport) {
			$response['errorMessage'] = 'pim_dataport_notfound';
			return new JsonResponse($response);
		}

		$name = $this->sanitizeName($request->get("name"));

		if (!$this->checkDataportName($name, $id)) {
			$response['errorMessage'] = 'pim_dataport_name_taken';
			return new JsonResponse($response);
		}

		$sourcetype = $request->get("sourcetype");

		if (!array_key_exists($sourcetype, self::$sourceconfigDefaults)) {
			$sourcetype = self::DEFAULT_SOURCETYPE;
		}

		$data = array(
			'name' => $name,
			'description' => $request->get("description"),
			'sourcetype' => $sourcetype,
		);

		$sourceData = (array)json_decode($request->get('sourceData'));

		$targetClassId = $request->get("targetclass");
		$targetClass = ClassDefinition::getById($targetClassId);
		if (!($targetClass instanceof ClassDefinition)) {
            $targetClassId = null;
		}

		// Object folder
		$targetFolderPath = $request->get("targetfolder");
		$targetFolder = AbstractObject::getByPath($targetFolderPath);

		if (!($targetFolder instanceof AbstractObject)) {
			$targetFolder = AbstractObject::getById(1);
		}

		// Asset folder
		$assetFolderPath = $request->get("assetfolder");
		$assetFolder = Asset::getByPath($assetFolderPath);

		if (!($assetFolder instanceof Asset\Folder)) {
			$assetFolder = Asset::getById(1);
		}

		$targetData = array(
		    'mode' => $request->get("mode"),
			'itemClass' => $targetClassId,
			'itemFolder' => $targetFolder->getFullPath(),
			'assetFolder' => $assetFolder->getFullPath(),
			'idPrefix' => \Pimcore\File::getValidFilename($request->get('idPrefix')),
			'categoryClass' => $request->get('categoryClass'),
			'fieldnameProducts' => $request->get('fieldnameProducts'),
            'importerClass' => $request->get('importerClass'),
		);


		if ($request->get('javascriptEngine', null) !== null) {
			$targetData['javascriptEngine'] = $request->get('javascriptEngine');

			if (empty($targetData['javascriptEngine'])) {
				$targetData['javascriptEngine'] = null;
			}
		}

		$rawitemData = json_decode($request->get("rawitemData"));

		// Defaults: Initialize config
		$sourceConfig = array_intersect_key($sourceData, self::$sourceconfigDefaults[$sourcetype]);
		$sourceConfig = array_merge(self::$sourceconfigDefaults[$sourcetype], $sourceConfig);
		$sourceConfig['fields'] = array();

		if (!empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);
			if (!empty($config['fields'])) {
				$sourceConfig['fields'] = $config['fields'];
			}
		}
  
		// Default: Targetconfig
		$targetConfig = array_intersect_key($targetData, self::$targetconfigDefaults);
		$targetConfig = array_merge(self::$targetconfigDefaults, $targetConfig);
		$data['targetconfig'] = serialize($targetConfig);

		// Find existing fields
		$fieldTable = new RawItemField();
		$rawItemFields = array();
		$toBeDeleted = array();
		$existingFields = $fieldTable->find(array('dataportId = ?' => $id));
		foreach ($existingFields as $field) {
			$rawItemFields['field_' . $field['fieldNo']] = $field;
			$toBeDeleted['field_' . $field['fieldNo']] = $field;
		}

		// Create new fields and update existing ones
		if ($sourcetype === 'bmecat') {
			$sourceConfig['fields'] = array();
		} else if (\is_array($rawitemData)) {
			$sourceConfig['fields'] = [];
			foreach ($rawitemData as $index => $setting) {
				$fieldNo = (int)$setting->fieldNo;
				$name = $setting->name;

				if ($fieldNo > 0) {
					$key = 'field_' . $fieldNo;

					if (empty($name)) {
						continue;
					}

					if (array_key_exists($key, $rawItemFields)) {
                        $rawItemFields[$key]['name'] = $name;
                        $rawItemFields[$key]['priority'] = $index;
					} else {
						$rawItemFields[$key] = array(
							'dataportId' => $id,
							'fieldNo' => $fieldNo,
							'name' => $name,
                            'priority' => $index
						);
					}

					unset($toBeDeleted[$key]);
					
					// Custom field config
					foreach (self::$sourceconfigFields[$sourcetype] as $config => $defaultValue) {
						if (property_exists($setting, $config)) {
							// A value for the config key has been submitted
                            $sourceConfig['fields'][$key][$config] = $setting->$config;
						}
					}
				}
			}
		}

		$data['sourceconfig'] = serialize($sourceConfig);
  
		try {
            $fieldTable->beginTransaction();
			if ($table->update($data, ['id' => $id]) !== false) {
				$response['success'] = true;
			}

			foreach ($rawItemFields as $key => $field) {
				try {
					$fieldTable->create($field);
				} catch(DBALException $e) {
                    $fieldTable->update($field, ['dataportId' => $field['dataportId'],
                                                'fieldNo' => $field['fieldNo']]);
				}
			}

			foreach ($toBeDeleted as $field) {
                $fieldTable->deleteWhere(['dataportId' => $field['dataportId'],
                                          'fieldNo' => $field['fieldNo']]);
			}

            $fieldTable->commit();
		} catch (\Exception $e) {
            $fieldTable->rollback();
			$response['success'] = false;
			$response['errorMessage'] = 'Unable to save configuration: ' . $e;
		} catch (\Error $e) {
            $fieldTable->rollback();
            $response['success'] = false;
            $response['errorMessage'] = 'Unable to save configuration: ' . $e;
        }

		return new JsonResponse($response);
	}

    /**
     * @Route("/get-rawitemfield-config/{id}")
     */
	public function getRawitemfieldConfigAction(Request $request, int $id) {
		$data = array();

		$table = new Dataport();
		$dataport = $table->get($id);

		if (!$dataport) {
			return new JsonResponse(array(
				'success' => false,
				'total' => 0,
				'fields' => array(),
			));
		}
		
		$sourcetype = $dataport['sourcetype'];
		$fieldConfig = unserialize($dataport['sourceconfig']);

		$fieldTable = new RawItemField();
		$fields = $fieldTable->find(array('dataportId = ?' => $id), 'priority, fieldNo');

		$demoItems = array();

		try {
			$oldSetting = ini_get('display_errors');
			ini_set('display_errors', 0);
			$parser = $table->getParser($id);
			$demoItems = $parser->parse(2);
			ini_set('display_errors', $oldSetting);
		} catch (\Exception $e) {
			$this->logger->error("Unable to retrieve demo items: " . $e);
		}

		$isMultivalue = array();
		if (is_array($fieldConfig['fields'])) {
			foreach ($fieldConfig['fields'] as $key => $values) {
				if (isset($values['multiValues']) && $values['multiValues'] === true) {
					$isMultivalue[] = $key;
				}
			}
		}
		
		foreach ($fields as $field) {
			$key = 'field_' . $field['fieldNo'];

			$fieldData = array(
				'fieldNo' => $field['fieldNo'],
				'name' => $field['name'],
			);

			foreach (self::$sourceconfigFields[$sourcetype] as $config => $defaultValue) {
				if (is_array($fieldConfig['fields'][$key]) && array_key_exists($config, $fieldConfig['fields'][$key])) {
					$fieldData[$config] = $fieldConfig['fields'][$key][$config];
				} else {
					$fieldData[$config] = $defaultValue;
				}
			}

			// Testdata
			foreach($demoItems as $i => $demoItem) {
			 
				$value = array_key_exists($key, $demoItem) ? $demoItem[$key] : '';
				if (in_array('field_' . $field['fieldNo'], $isMultivalue) && !empty($value)) {
					$fieldData['data' . ($i + 1)] = unserialize($value);
				} else {
					$fieldData['data' . ($i + 1)] = $value;
				}
			}

			$data[] = $fieldData;
		}

		return new JsonResponse(array(
			'success' => true,
			'total' => count($data),
			'fields' => $data,
		));
	}

    /**
     * @Route("/get-previewgrid-config")
     */
	public function getPreviewgridConfigAction(Request $request) {
		$response = array(
			'success' => false
		);

		$id = (int)$request->get("dataportId");
		$table = new Dataport();
		$dataport = $table->get($id);

		if (!$dataport) {
			$response['errorMessage'] = 'pim_dataport_notfound';
			return new JsonResponse($response);
		}

		$columns = array();

		$fieldTable = new RawItemField();
		$rawItemFields = $fieldTable->find(array('dataportId = ?' => $dataport['id']));
		foreach ($rawItemFields as $field) {
			$columns[] = array(
				"name" => $field['name'],
				"fieldNo" => $field['fieldNo']
			);
		}

		$response['success'] = true;
		$response['columns'] = $columns;

		return new JsonResponse($response);
	}

    /**
     * @Route("/get-rawdata/{id}")
     */
	public function getRawdataAction(Request $request, int $id) {
		$data = array();

		$table = new Dataport();
		$dataport = $table->get($id);

		if (!$dataport) {
			return new JsonResponse(array(
				'success' => false,
				'total' => 0,
				'fields' => array(),
			));
		}

		$isMultivalue = array();
		$sourceconfig = unserialize($dataport['sourceconfig']);
		if (is_array($sourceconfig['fields'])) {
			foreach ($sourceconfig['fields'] as $key => $values) {
				if (isset($values['multiValues']) && $values['multiValues'] === true) {
					$isMultivalue[] = $key;
				}
			}
		}

		$start = (int)$request->get('start');
        if (!$start || $start < 0) {
            $start = 0;
        }

		$limit = (int)$request->get('limit');
        if (!$limit || $limit < 1) {
            $limit = 25;
        }

		$condition = 'FROM plugin_pim_rawitem i, plugin_pim_rawitemData d WHERE i.id = d.rawItemId AND i.dataportId = ?';

		$variables = array($id);

		$query = $request->get('query');
		if (!empty($query)) {
			$condition .= ' AND (i.id=? OR d.value LIKE ?)';
            $variables[] = $query;
            $variables[] = '%' . $query . '%';
		}

		$db = Db::get();

        $total = (int)$db->fetchOne('SELECT COUNT(DISTINCT i.id) ' . $condition, $variables);

		$rawItemDataRepository = new RawItemData();
		if ($total > 0) {
            $items = $db->fetchAll('SELECT i.id, i.inserted, i.updated ' . $condition.' GROUP BY i.id ORDER BY priority LIMIT '.$start.','.$limit, $variables);

            foreach ($items as $item) {
                $itemData = [
                    'id' => $item['id'],
                    'inserted' => $item['inserted'],
                    'updated' => $item['updated'],
                ];

                $rawItemData = $rawItemDataRepository->find(['rawItemId = ?' => $item['id']]);
                foreach ($rawItemData as $row) {
                    $value = $row['value'];
                    if (in_array('field_' . $row['fieldNo'], $isMultivalue)) {
                        $unserialized = unserialize($value);
                        if(\is_array($unserialized)) {
                            $itemData['field_' . $row['fieldNo']] = json_encode($unserialized, JSON_PRETTY_PRINT);
                        } elseif(\is_object($unserialized)) {
                            $itemData['field_' . $row['fieldNo']] = json_encode(\get_object_vars($unserialized), JSON_PRETTY_PRINT);
                        } else {
                            $itemData['field_' . $row['fieldNo']] = $value;
                        }
                    } else {
                        $itemData['field_' . $row['fieldNo']] = $value;
                    }
                }

                $data[] = $itemData;
            }
		}

		return new JsonResponse(array(
			'success' => true,
			'total' => $total,
			'fields' => $data,
		));
	}

    /**
     * @Route("/delete-rawdata")
     */
	public function deleteRawdataAction(Request $request) {
		$id = (int)$request->get('id');

		if (!$id) {
			return new JsonResponse(array(
				'success' => false,
			));
		}

		$table = new RawItem();
		$numDeleted = $table->delete($id);

		return new JsonResponse(array(
			'success' => $numDeleted > 0,
		));
	}

    /**
     * @Route("/get-pimcore-classes")
     */
	public function getPimcoreClassesAction() {
		$result = array();

		$list = new ClassDefinition\Listing();
		$classes = $list->load();

		/**
		 * @var $class ClassDefinition
		 */
		foreach ($classes as $class) {
			$result[] = array(
				'id' => $class->getId(),
				'name' => $class->getName(),
			);
		}

		return new JsonResponse(array(
			'success' => true,
			'classes' => $result
		));
	}

    /**
     * @Route("/get-multihrefs-in-object")
     */
	public function getMultihrefsInObjectAction(Request $request) {
		$result = array();
		$classId = $request->get('classId');

		try {
			$clazz = ClassDefinition::getById($classId);
			$fieldDefs = $clazz->getFieldDefinitions();

			foreach ($fieldDefs as $def) {
				if (($def instanceof ClassDefinition\Data\Multihref && $def->getObjectsAllowed()) ||
					($def instanceof ClassDefinition\Data\ObjectsMetadata)) {
					$result[] = array(
						'id' => $def->getName(),
						'name' => $def->getTitle(),
					);
				}
			}

		} catch (\Exception $e) {
			$this->logger->error('Unable to load class "' . $classId . '"');
		}

		return new JsonResponse(array(
			'success' => true,
			'elements' => $result
		));
	}

    /**
     * @Route("/get-fieldcollections-in-object")
     */
	public function getFieldcollectionsInObjectAction(Request $request) {
		$result = array();
		$classId = $request->get('classId');

		try {
			$clazz = ClassDefinition::getById($classId);
			$fieldDefs = $clazz->getFieldDefinitions();

			foreach ($fieldDefs as $def) {
				if ($def instanceof ClassDefinition\Data\Fieldcollections) {
					$result[] = array(
						'id' => $def->getName(),
						'name' => $def->getTitle(),
					);
				}
			}

		} catch (\Exception $e) {
			$this->logger->error('Unable to load class "' . $classId . '"');
		}

		return new JsonResponse(array(
			'success' => true,
			'elements' => $result
		));
	}

    /**
     * @Route("/get-inputfields-in-object")
     */
	public function getInputfieldsInObjectAction(Request $request) {
		$result = array();
		$classId = $request->get('classId');

		try {
			$clazz = ClassDefinition::getById($classId);
			$fieldDefs = $clazz->getFieldDefinitions();

			foreach ($fieldDefs as $def) {
				if ($def instanceof ClassDefinition\Data\Input) {
					$result[] = array(
						'id' => $def->getName(),
						'name' => $def->getTitle(),
					);
				}
			}

		} catch (\Exception $e) {
			$this->logger->error('Unable to load class "' . $classId . '"');
		}

        return new JsonResponse(array(
			'success' => true,
			'elements' => $result
		));
	}

	private function sanitizeName($name) {
		$name = trim($name);
		$name = Service::getValidKey($name, 'object');

		return $name;
	}

	/**
	 * Checks if the given name is still available.
	 * @param $name string Name to check
	 * @param $id int (Optional) ID of a dataport to exclude from check
	 * @return bool true if it is, false otherwise
	 */
	private function checkDataportName($name, $id = null) {
		$table = new Dataport();

		$condition = array(
			'name = ?' => $name
		);

		if ($id !== null) {
			$condition['id != ?'] = $id;
		}

		$existing = $table->countRows($condition);

		if ($existing > 0) {
			return false;
		}

		return true;
	}

    /**
     * @Route("/get-javascript-engines")
     */
	public function getJavascriptEnginesAction(Request $request)
	{
		$data = [
			Javascript::ENGINE_V8JS => Javascript::isEngineAvailable(Javascript::ENGINE_V8JS),
			Javascript::ENGINE_SPIDERMONKEY=> Javascript::isEngineAvailable(Javascript::ENGINE_SPIDERMONKEY),
            Javascript::ENGINE_SPIDERMONKEY_LEGACY => Javascript::isEngineAvailable(Javascript::ENGINE_SPIDERMONKEY_LEGACY),
            Javascript::ENGINE_PHP => Javascript::isEngineAvailable(Javascript::ENGINE_PHP),
		];


		$result = [];

		$result[] = [
			'id' => 0,
			'available' => true,
			'name' => $this->getTranslator($request)->trans('pim.jsengine.none', [], 'admin')
		];

		foreach ($data as $key => $available) {

			$nameSuffix = '';
			if (!$available) {
				$nameSuffix = ' (' . $this->getTranslator($request)->trans('pim.jsengine.unavailable', [], 'admin') . ')';
			}

			$result[] = [
				'id' => $key,
				'available' => $available,
				'name' => $this->getTranslator($request)->trans('pim.jsengine.' . $key, [], 'admin') . $nameSuffix
			];
		}


        return new JsonResponse(array(
			'success' => true,
			'data' => $result
		));
	}

    /**
     * @Route("/get-source-class-methods/{classId}")
     */
    public function getSourceClassMethodsAction($classId) {
        $class = \Pimcore\Model\Object\ClassDefinition::getById($classId);
        if (!($class instanceof \Pimcore\Model\Object\ClassDefinition)) {
            \Pimcore\Logger::error("No class found for id " . $classId);
            return new JsonResponse(array(
                'success' => false,
                'methods' => []
            ));
            return;
        }

        $className = $class->getName();
        try {
            $object = \Pimcore::getContainer()->get('pimcore.model.factory')->build('Pimcore\\Model\\DataObject\\' . ucfirst($className));
        } catch(Pimcore\Loader\ImplementationLoader\Exception\UnsupportedException $e) {
            $object = \Pimcore::getContainer()->get('pimcore.model.factory')->build('Pimcore\\Model\\Object\\' . ucfirst($className));
        }

        $classReflection = new \ReflectionClass($object);
        $methods = [];
        foreach ($classReflection->getMethods(\ReflectionMethod::IS_PUBLIC) as $method) {
            if(0 === strpos($method->getName(), 'get')) {
                $methods[] = ['name' => $method->getName()];
            }
        }

        usort($methods, function($method1, $method2) {
            return strcmp($method1['name'], $method2['name']);
        });

        return new JsonResponse(array(
            'success' => true,
            'methods' => $methods
        ));
    }
}
