<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\RawData;

use Blackbit\PimBundle\model\Dataport;
use Blackbit\PimBundle\model\Fieldmapping;
use Blackbit\PimBundle\model\ImportStatus;
use Blackbit\PimBundle\model\RawItem;
use Psr\Log\LoggerInterface;

/**
 * Ausgangspunkt für den Rohdatenimport. Wählt den korrekten Importer für einen Importvorgang (von Datenquelle zu
 * Rohdaten) und startet diesen.
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Medien GmbH, http://www.blackbit.de/
 */
class Importmanager {

	private $_overrideFile = null;
	private $_removeFileAfterImport = false;

	/**
	 * Importiert alle Rohdaten aus der Datenquelle des Datenports
	 * @param $id integer ID des Datenports
	 * @param $importType integer Importtyp
	 */
	public function importDataport($id, $importType = ImportStatus::TYPE_RAWDATA) {
		$table = new Dataport();
		$dataport = $table->get($id);

		if (!$dataport) {
			return;
		}

		$statusModel = new ImportStatus();
        $dbNow = new \DateTime();
		$statusKey = uniqid();
		$statusModel->create(array(
			'key' => $statusKey,
			'dataportId' => $id,
			'startDate' => $dbNow,
			'lastUpdate' => $dbNow,
			'importType' => $importType,
		));

		$existingRawItems = new RawItem();
		$existingRawItems->deleteWhere(['dataportId' => $id]);

		$parser = $table->getParser($id);
		if(\method_exists($parser, 'setSourceFile')) {
            $parser->setSourceFile($this->getOverrideFile());
        }

        if(\method_exists($parser, 'removeFileAfterImport')) {
            $parser->removeFileAfterImport($this->_removeFileAfterImport === true);
        }

        @ini_set('memory_limit', -1);
        @ini_set('max_execution_time', -1);
        @ini_set('max_input_time', -1);

		$items = $parser->parse(null);
		$total = \count($items);
		$statusModel->update(array(
			'lastUpdate' => $dbNow,
			'totalItems' => $total,
		), ['key' => $statusKey]);

		$chunkSize = 100;
		$chunks = array_chunk($items, $chunkSize);

        $keyMappingFieldNumber = null;

        $fieldMappingTable = new Fieldmapping();
        $mappings = $fieldMappingTable->find(array('dataportId = ?' => $id));
        foreach ($mappings as $row) {
            if ($row['keyMapping'] == true) {
                $keyMappingFieldNumber = $row['fieldNo'];
                break;
            }
        }

		$i = 0;
		foreach ($chunks as $chunk) {
			$this->process($id, $keyMappingFieldNumber, $chunk);

			$done = ($i + 1) * $chunkSize;
			if ($done > $total) {
				$done = $total;
			}

			/**
			 * Ebenfalls Status und Endzeitpunkt zurücksetzen, falls dies zwischenzeitlich durch den Maintenance-Job als
			 * abgebrochener Prozess markiert wurde
			 */
			$statusModel->update(array(
				'status' => ImportStatus::STATUS_RUNNING,
				'endDate' => null,
				'lastUpdate' => $dbNow,
				'doneItems' => $done,
			), ['key' => $statusKey]);

			$i++;
		}

		$statusModel->update(array(
			'status' => ImportStatus::STATUS_FINISHED,
			'lastUpdate' => $dbNow,
			'endDate' => $dbNow,
			'doneItems' => $total,
		), ['key' => $statusKey]);
	}

	/**
	 * Fügt die Items in die Rohdatentabelle ein. Momentan synchrone Implementierung.
	 * @param integer $dataportId ID des Datenports
	 * @param integer $keyFieldNo Nummer des Feldes mit dem Schlüsselwert
	 * @param array $items Items zum einfügen
	 */
	private function process($dataportId, $keyFieldNo, $items) {
		$importer = new Importer($dataportId, $keyFieldNo, \Pimcore::getContainer()->get('pim.logger'));
		$importer->insert($items);
	}

	/**
	 * @return null
	 */
	public function getOverrideFile() {
		return $this->_overrideFile;
	}

	/**
	 * @param string $overrideFile
	 */
	public function setOverrideFile($overrideFile) {
	    if(!empty($overrideFile)) {
            $this->_overrideFile = $overrideFile;
        }
	}

    /**
     * @param bool $removeFileAfterImport
     */
    public function setRemoveFileAfterImport($removeFileAfterImport)
    {
        $this->_removeFileAfterImport = (bool)$removeFileAfterImport;
    }
}
