<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\RawData;

use Blackbit\PimBundle\model\RawItem;
use Blackbit\PimBundle\model\RawItemData;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\GenericEvent;

/**
 * Fügt Rohdaten in die Rohdatentabellen ein
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Medien GmbH, http://www.blackbit.de/
 */
class Importer {
    private $dataportId;
    private $keyFieldNo;

    /** @var LoggerInterface */
    private $logger;

    public function __construct($dataportId, $keyFieldNo, LoggerInterface $logger) {
        $this->dataportId = (int)$dataportId;
        $this->keyFieldNo = (int)$keyFieldNo;
        $this->logger = $logger;
    }

    /**
     * Startet den Einfügeprozess
     */
    public function insert(array $items)
    {
        $rawItem = new RawItem();
        $rawItemData = new RawItemData();

        $errors = 0;
        $successes = 0;

        foreach ($items as $item) {
            try {
                $hashFromItemInFile = md5(implode('_', $item));

                $rawItemId = $rawItem->findByHash($this->dataportId, $hashFromItemInFile);

                $rawItem->beginTransaction();

                $date = new \DateTime();
                if(empty($rawItemId)) {
                    $newRawItem = $rawItem->create([
                        'dataportId' => $this->dataportId,
                        'inserted'   => $date,
                        'updated'    => $date,
                        'hash' => $hashFromItemInFile
                    ]);
                    $rawItemId = $newRawItem['id'];
                } else {
                    $rawItem->update([
                        'dataportId' => $this->dataportId,
                        'inserted'   => $date,
                        'updated'    => $date
                    ], [
                        'id' => $rawItemId
                    ]);
                }

                $priority = [];
                foreach ($item as $field => $value) {
                    $fieldNo = (int)substr($field, strrpos($field, '_') + 1);
                    if ($fieldNo > 0) {
                        $rawItemData->create([
                            'rawItemId' => $rawItemId,
                            'fieldNo'   => $fieldNo,
                            'value'     => $value,
                        ]);

                        $value = trim($value);
                        if($value !== '') {
                            $priority[] = $value;
                        }
                    }
                }
                
                $priority = implode(' ', $priority);

                $rawItem->update([
                    'priority' => \mb_substr($priority, 0, 255)
                ], [
                    'id' => $rawItemId
                ]);

                $rawItem->commit();
                $successes++;

                \Pimcore::getEventDispatcher()->dispatch('pim.rawItemImported', new GenericEvent(
                    $rawItem, [
                        'dataportId' => $this->dataportId,
                        'data' => $item
                    ]
                ));
            } catch (\Exception $e) {
                $this->logger->error("Unable to import raw data (raw item {$item['id']}): {$e}");
                try {
                    $rawItem->rollBack();
                } catch(Doctrine\DBAL\ConnectionException $e) {
                }
            } catch (\Error $e) {
                $this->logger->error("Unable to import raw data (raw item {$item['id']}): {$e}");
                try {
                    $rawItem->rollBack();
                } catch(Doctrine\DBAL\ConnectionException $e) {
                }
            }
        }

        return [
            'successes' => $successes,
            'errors'    => $errors,
        ];
    }
}