<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
class Object_Class_Data_FlexibleTable extends \Pimcore\Model\DataObject\ClassDefinition\Data {

	/**
	 * Static type of this element
	 *
	 * @var string
	 */
	public $fieldtype = "flexibleTable";

	/**
	 * @var integer
	 */
	public $width;

	/**
	 * @var integer
	 */
	public $height;

	/**
	 * @var integer
	 */
	public $labelWidth;

	/**
	 * @var object
	 */
	public $cols;

	/**
	 * Type for the column to query
	 *
	 * @var string
	 */
	public $queryColumnType = "text";

	/**
	 * Type for the column
	 *
	 * @var string
	 */
	public $columnType = "text";

	/**
	 * Type for the generated phpdoc
	 *
	 * @var string
	 */
	public $phpdocType = "array";

	/**
	 * @return integer
	 */
	public function getWidth() {
		return $this->width;
	}

	/**
	 * @param integer $width
	 * @return void
	 */
	public function setWidth($width) {
		$this->width = $width;
	}

	/**
	 * @return integer
	 */
	public function getHeight() {
		return $this->height;
	}

	/**
	 * @param integer $height
	 * @return void
	 */
	public function setHeight($height) {
		$this->height = $height;
	}

	/**
	 * @return integer
	 */
	public function getLabelWidth() {
		return $this->labelWidth;
	}

	/**
	 * @param integer $labelWidth
	 * @return void
	 */
	public function setLabelWidth($labelWidth) {
		$this->labelWidth = $labelWidth;
	}

	/**
	 * @param string $labelFirstCell
	 */
	public function setLabelFirstCell($labelFirstCell) {
		$this->labelFirstCell = $labelFirstCell;
	}

	/**
	 * @return object
	 */
	public function getCols() {
		return $this->cols;
	}

	/**
	 * @param object $cols
	 * @return void
	 */
	public function setCols($cols) {
		if(isset($cols['key'])) {
			$cols = array($cols);
		}
		usort($cols, array($this, 'sort'));

		$this->cols = array();;
		foreach($cols as $c) {
			$c['key'] = strtolower($c['key']);
			$this->cols[] = $c;
		}
	}

	public function sort($a, $b) {
		if(is_array($a) && is_array($b)) {
			return $a['position'] - $b['position']; // strcmp($a['position'], $b['position']);
		}
		return strcmp($a, $b);
	}


	/**
	 * @see Object_Class_Data::getDataForResource
	 * @param string $data
	 * @param null|Object_Abstract $object
	 * @return string
	 */
	public function getDataForResource($data, $object = null) {
		$resourceData = array();

		if(!empty($data)) {
			$resourceData = Pimcore_Tool_Serialize::serialize($data);
		}

		return $resourceData;
	}

	/**
	 * @see Object_Class_Data::getDataFromResource
	 * @param string $data
	 * @return Object_Data_FlexibleTable
	 */
	public function getDataFromResource($data) {
		return Pimcore_Tool_Serialize::unserialize((string) $data);
	}

	/**
	 * @see Object_Class_Data::getDataForQueryResource
	 * @param string $data
	 * @param null|Object_Abstract $object
	 * @return string
	 */
	public function getDataForQueryResource($data, $object = null) {
		return $this->getDataForResource($data, $object);
	}

	/**
	 * @see Object_Class_Data::getDataForEditmode
	 * @param string $data
	 * @param null|Object_Abstract $object
	 * @return string
	 */
	public function getDataForEditmode($data, $object = null, $params = array()) {
		$editArray = array();

		if($data instanceof Object_Data_FlexibleTable) {
			if($data->isEmpty()) {
				return array();
			} else {
				$dataArray = $data->getData();

				foreach($dataArray as $d) {
					// Convert stdClass to array
					if (is_object($d)) {
						$d = (array)$d;
					}

					$editArrayItem = array();
					foreach($this->getCols() as $c) {
						$editArrayItem[$c['key']] = $d[$c['key']];
					}
					$editArray[] = $editArrayItem;
				}
			}
		}

		return $editArray;
	}

	/**
	 * @see Object_Class_Data::getDataFromEditmode
	 * @param string $data
	 * @param null|Object_Abstract $object
	 * @return string
	 */
	public function getDataFromEditmode($data, $object = null, $params = array()) {
		return new Object_Data_FlexibleTable($data);
	}

	public function getDataForGrid($data, $object = null) {
		if($data instanceof Object_Data_FlexibleTable) {
			if(!$data->isEmpty()) {
				return $data->getData();
			}
		}
		return null;
	}

	/**
	 * @see Object_Class_Data::getVersionPreview
	 * @param string $data
	 * @return string
	 */
	public function getVersionPreview($data) {
		if ($data){
			return $data->getHtmlTable($this->cols);
		}
		else {
			return null;
		}
	}

	/**
	 * Checks if data is valid for current data field
	 *
	 * @param mixed $data
	 * @param boolean $omitMandatoryCheck
	 * @throws Exception
	 */
	public function checkValidity($data, $omitMandatoryCheck = false){
		if(!$omitMandatoryCheck and $this->getMandatory()){

			$empty = true;
			if(!empty($data)) {
				$dataArray = $data->getData();
				foreach($this->getCols() as $c) {
					if(!empty($dataArray[$c['key']])) {
						$empty = false;
					}
				}
			}
			if($empty) {
				throw new Exception("Empty mandatory field [ ".$this->getName()." ]");
			}
		}

		if(!empty($data) and !$data instanceof Object_Data_FlexibleTable){
			throw new Exception("invalid table data");
		}
	}

	/**
	 * converts object data to a simple string value or CSV Export
	 * @abstract
	 * @param Object_Abstract $object
	 * @return string
	 */
	public function getForCsvExport($object) {
		$key = $this->getName();
		$getter = "get".ucfirst($key);
		$value = $object->$getter();
		if ($value instanceof Object_Data_FlexibleTable) {
			$string = "";
			$dataArray = $value->getData();

			foreach ($dataArray as $data) {
				$values = array();
				foreach($this->getCols() as $c) {
					$values[] = $data[$c['key']];
				}

				$string .= implode("##", $values) . "§§";
			}

			return rtrim($string, "§");
		} else {
			return null;
		}
	}

	/**
	 * fills object field data values from CSV Import String
	 * @abstract
	 * @param string $importValue
	 * @param Object_Abstract $abstract
	 * @return Object_Class_Data
	 */
	public function getFromCsvImport($importValue) {
		$dataRows = explode("§§", $importValue);

		$dataTable = array();

		foreach($dataRows as $row) {
			$dataArray = explode("##", $row);
			$i = 0;
			foreach($this->getCols() as $c) {
				$dataTable[$c['key']] = $dataArray[$i];
				$i++;
			}
		}


		$value = new Object_Data_FlexibleTable($dataTable);
		return $value;
	}

	/**
	 * converts data to be exposed via webservices
	 * @param string $object
	 * @return mixed
	 */
	public function getForWebserviceExport ($object) {
		$key = $this->getName();
		$getter = "get".ucfirst($key);

		$table = $object->$getter();
		if ($table instanceof Object_Data_FlexibleTable) {
			return $table->getData();
		} else {
			return null;
		}
	}

	/**
	 * converts data to be imported via webservices
	 * @param mixed $value
	 * @return mixed
	 */
	public function getFromWebserviceImport ($value) {
		if(empty($value)){
			return null;
		} else if(is_array($value)){
			return new Object_Data_FlexibleTable($value);
		} else {
			throw new Exception("cannot get values from web service import - invalid data");
		}
	}

	public function getGetterCode ($class) {
		// getter

		$key = $this->getName();
		$code = "";

		$code .= '/**' . "\n";
		$code .= '* @return ' . $this->getPhpdocType() . "\n";
		$code .= '*/' . "\n";
		$code .= "public function get" . ucfirst($key) . " () {\n";

		// adds a hook preGetValue which can be defined in an extended class
		$code .= "\t" . '$preValue = $this->preGetValue("' . $key . '");' . " \n";
		$code .= "\t" . 'if($preValue !== null && !Pimcore::inAdmin()) { return $preValue;}' . "\n";

		if(method_exists($this,"preGetData")) {
			$code .= "\t" . '$data = $this->getClass()->getFieldDefinition("' . $key . '")->preGetData($this);' . "\n";
		} else {
			$code .= "\t" . '$data = $this->' . $key . ";\n";
		}

		// insert this line if inheritance from parent objects is allowed
		if ($class->getAllowInherit()) {
			$code .= "\t" . 'if((!$data || $data->isEmpty()) && Object_Abstract::doGetInheritedValues()) { ' . "\n";
			$code .= "\t\t" . '$parentData  = $this->getValueFromParent("' . $key . '");' . "\n";
			$code .= "\t\t" . 'if($parentData) { return $parentData; }' . "\n";
			$code .= "\t\t" . 'else { return $data; }' . "\n";
			$code .= "\t" . '}' . "\n";
		}

		$code .= "\t return " . '$data' . ";\n";
		$code .= "}\n\n";

		return $code;
	}


	/**
	 * Creates getter code which is used for generation of php file for object brick classes using this data type
	 * @param $brickClass
	 * @return string
	 */
	public function getGetterCodeObjectbrick ($brickClass) {
		$key = $this->getName();
		$code = '/**' . "\n";
		$code .= '* @return ' . $this->getPhpdocType() . "\n";
		$code .= '*/' . "\n";
		$code .= "public function get" . ucfirst($key) . " () {\n";

		$code .= "\t" . 'if((!$this->' . $key . '|| $this->' . $key . '->isEmpty()) && Object_Abstract::doGetInheritedValues($this->getObject())) {' . "\n";
		$code .= "\t\t" . 'return $this->getValueFromParent("' . $key . '");' . "\n";
		$code .= "\t" . '}' . "\n";

		$code .= "\t return " . '$this->' . $key . ";\n";
		$code .= "}\n\n";

		return $code;

	}
}
