<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\Import\Worker;

use Blackbit\PimBundle\lib\Pim\Helper;
use Blackbit\PimBundle\lib\Pim\Item\Bmecat\Importer;
use Blackbit\PimBundle\model\BmeProduct;
use Blackbit\PimBundle\model\BmeProductCategory;
use Pimcore\Cache;
use Pimcore\File;
use Pimcore\Logger;
use Pimcore\Model\Asset;
use Pimcore\Model\Asset\Folder;
use Pimcore\Model\Asset\Image;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\ClassDefinition\Data;
use Pimcore\Model\DataObject\ClassDefinition\Data\Multihref;
use Pimcore\Model\DataObject\ClassDefinition\Data\ObjectsMetadata;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\DataObject\Data\ObjectMetadata;
use Pimcore\Model\DataObject\Fieldcollection;
use Pimcore\Model\DataObject\Listing;
use Pimcore\Model\DataObject\Service;
use Pimcore\Model\User;
use Pimcore\Tool;

/**
 *
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Werbung neue Medien GmbH, http://www.blackbit.de/
 */
class Bmecat {
	private $debug = true;

	private $categoryDummy;

	/**
	 * @var \DOMXPath
	 */
	private $xp;

	/**
	 * @var array
	 */
	private $xpathExpr;

	/**
	 * ID des laufenden Imports
	 * @var string
	 */
	private $importKey;

	/**
	 * Prefix für Produkt-IDs. Am Datenport konfigurierbar
	 * @var string
	 */
	private $idPrefix;

	/**
	 * @var int
	 */
	private $dataportId;

	/**
	 * @var string
	 */
	private $file;

	/**
	 * @var string
	 */
	private $preJs;

	/**
	 * @var Folder
	 */
	private $assetFolder;
	private $assetSource;

	/**
	 * @var array
	 */
	private $categoryMappings = array();

	/**
	 * @var array
	 */
	private $referencesList;

	public function resetContext() {
		$this->categoryDummy = null;
		$this->xp = null;
		$this->xpathExpr = null;
		$this->importKey = null;
		$this->idPrefix = null;
		$this->dataportId = null;
		$this->file = null;
		$this->preJs = '';
		$this->assetFolder = null;
		$this->assetSource = null;
		$this->referencesList = array();
		$this->categoryMappings = array();
	}

	/**
	 * @param bool $shallow settings this flag to true will prevent this instance from acting as a gearman worker
	 */
	public function __construct($shallow = false) {
		if (!$shallow) {
			$worker = new \GearmanWorker();
			$worker->addServer();

			$worker->addFunction('ImportItem', array($this, 'importItem'));
			$worker->addFunction('AssignCategory', array($this, 'assignCategory'));
			$worker->addFunction('UpdateReferences', array($this, 'updateReferences'));

			if ($this->debug) {
				$this->debug('Worker started.');
			}

			while ($worker->work());
		}
	}

	/**
	 * Importjob für ein einzelnes Item
	 * @param \GearmanJob $job
	 * @return string serialisiertes Ergebnisarray
	 */
	public function importItem(\GearmanJob $job) {
		// Reset
		$this->resetContext();

		$importResult = array(
			'success' => false,
			'bmeToPimId' => array(),
		);

		try {
			if ($this->debug) {
				$this->debug('Executing ' . $job->unique());
			}

			// Kontext herstellen
			$workload = Cache::load($job->workload());
			$key = $workload['sourceProductId'];
//			$sourceProduct = dom_import_simplexml(simplexml_load_string($workload['product']));
			$itemClassName = $workload['itemClassName'];
			$itemClassId = $workload['itemClassId'];
			$itemFolder = $workload['itemFolder'];
			$categoryClassName = $workload['categoryClassName'];
			$categoryProductElement = $workload['categoryProductElement'];
			$categoryAttributeElementName = $workload['categoryAttributeElementName'];
			$itemStatusFormula = $workload['itemStatusFormula'];
			$itemAttributeElement = $workload['itemAttributeElement'];

			$this->idPrefix = $workload['idPrefix'];
			$this->file = $workload['file'];
			$this->dataportId = $workload['dataportId'];
			$this->assetSource = $workload['assetSource'];
			$this->assetFolder = Folder::getById($workload['assetFolderId']);
			$this->importKey = $workload['importKey'];

			if ($this->debug) {
				$start = microtime(true);
				$timings = array(
					'id' => null,
					'xmlToArray' => null,
					'collectData' => null,
					'load' => null,
					'findCategories' => null,
					'addData' => null,
					'save' => null,
					'total' => null,
				);
			}

			$dom = Helper::getDomDocument($workload['file']);
			$this->xp = new \DOMXPath($dom);
			$this->xpathExpr = Helper::getBmecatXpathForVersion($workload['version']);

			$sourceProduct = $this->xp->query(vsprintf($this->xpathExpr['product']['byId'], array($key)))->item(0);
			$sourceProductXML = Helper::nodeToArray($sourceProduct);

			if ($this->debug) {
				$timings['xmlToArray'] = microtime(true) - $start;
				$startCollect = microtime(true);
			}

			$BMEData = $this->collectDataFromBMECAT($this->xp, $this->xpathExpr, $sourceProduct);
			if ($this->debug) {
				$timings['collectData'] = microtime(true) - $startCollect;
			}

			// Produkte ohne Schlüssel Attribute überspringen
			if (count($BMEData['keyValues']) == 0) {
				if ($this->debug) {
					$this->debugTimings('Artikel:', $timings);
				}
				$importResult['message'] = 'No keyValues defined for product "' . $key . '"';
				return serialize($importResult);
			}

			if ($this->debug) {
				$startLoad = microtime(true);
			}
			$item = $this->getItem($itemClassName, $itemClassId, $itemFolder, $BMEData['keyValues'], $key);

			$importResult['bmeToPimId'][$key] = $item->getId();

			if ($this->debug) {
				$timings['id'] = $item->getId();
				$timings['load'] = microtime(true) - $startLoad;
				$startFindCat = microtime(true);
			}

			$productHasCategory = $this->setCategories($item, $key, $categoryClassName, $categoryProductElement);
			if ($this->debug) {
				$timings['findCategories'] = microtime(true) - $startFindCat;
			}

			if (!$productHasCategory) {
				if ($this->debug) {
					$this->debugTimings('Artikel:', $timings);
				}
				$importResult['message'] = 'Skipping item with ID "' . $item->getId() . '" as it is not mapped to a category';
				return serialize($importResult);
			}

			if ($this->debug) {
				$startAddData = microtime(true);
			}

			$this->updateStatus($item, $itemStatusFormula, $BMEData);
			$this->addData($item, $BMEData["fieldValues"], $itemAttributeElement, $sourceProductXML);

			if ($this->debug) {
				$timings['addData'] = microtime(true) - $startAddData;
				$startSave = microtime(true);
			}

			$item->setOmitMandatoryCheck(true);
			$item->save();

			if ($this->debug) {
				$timings['save'] = microtime(true) - $startSave;
				$timings['total'] = microtime(true) - $start;
				$this->debugTimings('Artikel:', $timings);
			}

			$importResult['success'] = true;
		} catch (\Exception $e) {
			$importResult['message'] = 'Exception while importing an item: ' . $e;
		}

		$importResult['referencesList'] = $this->referencesList;
		$importResult['categoryMappings'] = $this->categoryMappings;
		return serialize($importResult);
	}

	public function assignCategory(\GearmanJob $job) {
		// Reset
		$this->resetContext();

		$result = array(
			'success' => false
		);


		$catId = 'unknown';

		try {
			// Kontext herstellen
			$workload = Cache::load($job->workload());
			$catId = $workload['catId'];
			$mappings = $workload['mappings'];

			$category = Concrete::getById($catId);

			if ($this->debug) {
				$startCat = microtime(true);
				$this->debug('Kategoriezuordnung für ' . $catId . ' ...');
			}

			if ($category instanceof Concrete) {
				// Handle each field seperately (should be only one per import anyways)
				foreach ($mappings as $field => $itemIds) {
					$productListElementGetter = 'get'. ucfirst($field);
					$productListElementSetter = 'set'. ucfirst($field);
					$productListElement = $category->$productListElementGetter();

					// Load items for each field and assign them
					foreach ($itemIds as $itemId) {
						$item = Concrete::getById($itemId);

						$fieldDefinition = $category->getClass()->getFieldDefinition($field);

						if ($fieldDefinition instanceof ObjectsMetadata) {
							$insertElement = new ObjectMetadata($field, array(), $item);
						} else if ($fieldDefinition instanceof Multihref) {
							$insertElement = $item;
						} else {
							$insertElement = null;
						}

						if ($insertElement !== null){
							if (!is_array($productListElement)) {
								$productListElement = array();
							}
                            $productListElement[] = $insertElement;
						}
					}

					// Set updated reference container back to category
					$category->$productListElementSetter($productListElement);
				}

				$category->save();
			}

			if ($this->debug) {
				$this->debug('Zeit für ' . $catId . ': ' . (microtime(true) - $startCat));
			}
			$result['success'] = true;
		} catch (\Exception $e) {
			$result['message'] = 'Exception while assigning items to category "' . $catId . '": ' . $e;
		}

		return serialize($result);
	}

	public function updateReferences(\GearmanJob $job) {
		// Reset
		$this->resetContext();

		$result = array(
			'success' => false
		);

		$itemId = 'unknown';

		try {
			// Kontext herstellen
			$workload = Cache::load($job->workload());
			$itemId = $workload['data']['id'];
			$itemAttributeElement = $workload['itemAttributeElement'];
			$itemClassName = $workload['itemClassName'];
			$this->importKey = $workload['importKey'];
			$this->dataportId = $workload['dataportId'];
			$this->file = $workload['file'];
			$this->idPrefix = $workload['idPrefix'];

			// ID-Mapping
			$bmeToPimId = Cache::load('bme2pim_' . $this->importKey);

			$dom = Helper::getDomDocument($this->file);
			$xp = new \DOMXPath($dom);
			$xpathExpr = Helper::getBmecatXpathForVersion($workload['version']);

			if ($this->debug) {
				$timings = array(
					'id' => null,
					'collectData' => null,
					'getItem' => null,
					'findCollection' => null,
					'findReferences' => null,
					'save' => null,
					'total' => null,
				);
				$start = microtime(true);
			}
			$itemNode = $xp->query(sprintf($xpathExpr['product']['byId'], $itemId))->item(0);

			if (!($itemNode instanceof \DOMElement)) {
				$result['message'] = 'No item found for id "' . $itemId . '"';
				return serialize($result);
			}

			if ($this->debug) {
				$startCollect = microtime(true);
			}
			$itemKeys = $this->collectDataFromBMECAT($xp, $xpathExpr, $itemNode, true);

			if ($this->debug) {
				$timings['collectData'] = microtime(true) - $startCollect;
				$startGetItem = microtime(true);
			}

			$item = $this->getItem($itemClassName, null, null, $itemKeys['keyValues'], null);
			// Wenn Item nicht class ist -> suche die Feldsamlung um dort die Daten einzufügen
			if ($this->debug) {
				$timings['getItem'] = microtime(true) - $startGetItem;
			}

			if (!($item instanceof Concrete)) {
				$result['message'] = 'No PIM item found for keyValues ' . json_encode($itemKeys['keyValues']);
				Logger::warn($result['message']);
				return serialize($result);
			}

			if ($this->debug) {
				$timings['id'] = $item->getId();
				$startFindCollection = microtime(true);
			}

			foreach ($workload['data']['references'] as $element) {
				if ($element['fieldCollection'] === "class") {
					$addToObject = $item;
				} else {
					$getter = 'get' . ucfirst($itemAttributeElement);
					$setter = 'set' . ucfirst($itemAttributeElement);

					if (!method_exists($item, $getter) || !method_exists($item, $setter)) {
						$result['message'] = 'Getter or setter not found for attribute ' . $itemAttributeElement;
						Logger::warn($result['message']);
						return serialize($result);
					}

					$realCollectionName = 'Pimcore\\Model\\Object\\Fieldcollection\Data\\'. ucfirst($element['fieldCollection']);

					// if collection element exists check if specific collection exists, if not -> create it
					// if no collection element exists -> create it and the collection
                    /** @var Fieldcollection $itemCollectionsElement */
					$itemCollectionsElement = $item->$getter();
					if ($itemCollectionsElement != null) {
						$currentCollections = $itemCollectionsElement->getItems();
						$theCollection = null;
						foreach ($currentCollections as $key => $collection) {
							if ($collection instanceof $realCollectionName) {
								$theCollection = $collection;
							}
						}

						if ($theCollection == NULL) {
							$theCollection = new $realCollectionName();
							$itemCollectionsElement->add($theCollection);
							$item->setOmitMandatoryCheck(true);
							$item->save();
						}
					} else {
						$itemCollectionsElement = new Fieldcollection();
						$theCollection = new $realCollectionName();
						$itemCollectionsElement->add($theCollection);
						$item->$setter($itemCollectionsElement);
						$item->setOmitMandatoryCheck(true);
						$item->save();
					}
					// end if -> $theCollection contains a valid collection
					$addToObject = $theCollection;
				}

				if ($this->debug) {
					$timings['findCollection'] = microtime(true) - $startFindCollection;
					$startReferences = microtime(true);

					$this->debug('Ermittele ' . count($element['items']) . ' Referenzen für Artikel ' . $item->getId());

					$itemCounter = 1;
				}

				$referenceObjects = array();
				foreach ($element['items'] as $referenceId) {
					if ($this->debug) {
						$debug = 'Referenz ' . $itemCounter . ' (auf AID ' . $referenceId . ') ... ';
						$startRef = microtime(true);
					}
					$referenceObject = null;

					// Wenn die BMEcat AID in 'bmeToPimId' vorhanden ist, verwende diese Pim-ID
					if (array_key_exists($referenceId, $bmeToPimId)){
						if ($bmeToPimId[$referenceId] >= 0){
							$objectId = $bmeToPimId[$referenceId];
							$referenceObject = Concrete::getById($objectId);
						} else {
							// 'bmeToPimId' enthält Wert -1, das heißt es gibt für diese BMEcat AID kein passendes Object -> überspringen
							if ($this->debug) {
								$debug .= 'Kein passendes PIM-Objekt gefunden';
								$this->debug($debug . ' ' . (microtime(true) - $startRef));
							}
							continue;
						}
					}
					// Wenn die BMEcat AID in 'bmeToPimId' NICHT vorhanden ist, versuche das Object über die Zuordnung in dem XML-File zu finden
					if ($referenceObject == null) {
						$referenceNodes = $xp->query(sprintf($xpathExpr['product']['byId'], $referenceId));
						foreach ($referenceNodes as $referenceNode) {
							$referenceKeys = $this->collectDataFromBMECAT($xp, $xpathExpr, $referenceNode, true);
							$referenceObject = $this->getItem($itemClassName, null, null, $referenceKeys['keyValues'], null);
							// Pim-Object gefunden -> Zuordnung in 'bmeToPimId' eintragen
							if ($referenceObject != null) {
								$bmeToPimId[$referenceId] = $referenceObject->getId();
							}
						}

						// Pim-Object nicht vorhanden -> entsprechenden Eintrag in 'bmeToPimId' erstellen
						if ($referenceObject == null) {
							$bmeToPimId[$referenceId] = -1;
						}
					}
					if ($referenceObject != null) {
						$referenceObjects[] = $referenceObject;
					}
					if ($this->debug) {
						$this->debug($debug . ' ' . (microtime(true) - $startRef));
						$itemCounter++;
					}
				}

				if ($this->debug) {
					$timings['findReferences'] = microtime(true) - $startReferences;
					$startSave = microtime(true);
				}

				$setter = 'set' . ucfirst($element['attributeName']);
				if (method_exists($addToObject, $setter)) {
					$addToObject->$setter($referenceObjects);
				} else {
					Logger::warn('Setter not found for collection ' . $element['attributeName']);
				}
			}

			try {
				$item->save();
			} catch (\Exception $e) {
				Logger::error('Unable to save product relations for object "' . $itemId . '"');
			}

			if ($this->debug) {
				$timings['save'] = microtime(true) - $startSave;
				$timings['total'] = microtime(true) - $start;
				$this->debugTimings('Referenz:', $timings);
			}
		} catch (\Exception $e) {
			$result['message'] = 'Exception while assigning items to category "' . $itemId . '": ' . $e;
		}

		return serialize($result);
	}

	/**
	 * Stellt alle benötigten Daten aus dem XML File zusammen und gibt sie als Array zurück
	 * Dieses Array wird später zum setzen der Daten in dem Pimcore Objekt benutzt
	 * @param \DOMXPath $xp array
	 * @param array $xpathExpr
	 * @param \DOMElement $domNode
	 * @param bool $keysOnly
	 * @return array
	 */
	public function collectDataFromBMECAT($xp, $xpathExpr, \DOMElement $domNode, $keysOnly = false) {
		$data = array();
		$data['keyValues'] = array();
		$data['fieldValues'] = array();

		$keyList = $xp->query($xpathExpr['globalAttributes']['aid']['id'], $domNode);
		if ($keyList->length == 0) {
			return $data;
		}
		$artId = $keyList->item(0)->nodeValue;

		$bmeProductCategoryRepository = new BmeProductCategory();

		$categories = $bmeProductCategoryRepository->find([
		    'productId = ?' => $artId
        ]);

		if (is_array($categories)) {
			foreach ($categories as $category) {
				$BMEcategoryId = $category['categoryId'];
				// $BMEcategoryId kann Zeichen enthalten, die als Cache-Key nicht gültig sind. Deshalb Hash bilden
				$cacheKey = 'blackbit_categorymapping_' . md5($BMEcategoryId) .  '_' . $this->importKey;

				if (($mappingarray = Cache::load($cacheKey)) === false) {
					$mappingarray = $this->getMappingForCategory($BMEcategoryId);
					Cache::save($mappingarray, $cacheKey, array('bmecat_import_' . $this->importKey));
				}

				foreach ($mappingarray as $PIMcategory => $mappingGroups) { // Iterate over all PIMCategories for this BMECategory
					foreach ($mappingGroups as $mappingGroupName => $mappingGroupData) { // Iterate over all mappingGroups

						if ($keysOnly) {
							$data['keyValues'] = $this->collectKeyMappings($xp, $xpathExpr, $domNode, $mappingGroupData, $data['keyValues']);

							// Return first match
							if (!empty($data['keyValues'])) {
								return $data;
							}
						} else {
							if ($mappingGroupName === 'class') { // Set class Attributes
								$data['fieldValues']['class'] = $this->collectAttributeData($xp, $xpathExpr, $domNode, $mappingGroupData, $data['fieldValues']['class'], "class");

								// Use first set of keyValues found
								if (empty($data['keyValues'])) {
									$data['keyValues'] = $this->collectKeyMappings($xp, $xpathExpr, $domNode, $mappingGroupData, $data['keyValues']);
								}
							} else { // Set fieldcollections and Attributes
								$data['fieldValues'][$mappingGroupName] = $this->collectAttributeData($xp, $xpathExpr, $domNode, $mappingGroupData, $data['fieldValues'][$mappingGroupName], $mappingGroupName);
							}
						}
					}
				}
			}
		}

		return $data;
	}

	/*
	 * Versucht ein Object zu finden
	 * Falls es nicht vorhanden ist, wird versucht ein neues zu erzeugen
	 */
	public function getItem($itemClassName, $itemClassId, $itemFolder, $keyValues, $key) {
		// Try to find Item
		$item = null;

		if (!empty($keyValues)) {
            $itemListClassName = 'Pimcore\\Model\\Object\\' . ucfirst($itemClassName).'\\Listing';

			/**
			 * @var $list Listing
			 */
			$list = new $itemListClassName();
			$list->setUnpublished(true);

			foreach ($keyValues as $keyName => $keyValue) {
				$conditionValue = Importer::processJS($keyValue["mappings"]["calculation"], $keyValue["value"], $keyValue["valueArray"], $this->preJs);
				$conditionValue = $this->getIdPrefix() . $conditionValue;
				$list->setCondition("`$keyName` = ?", array($conditionValue));
			}
			if (count($list) == 1) {
				$item = $list->current();
			} else if (count($list) > 1) {
				$item = $list->current();
				Logger::warning('Found more than one current Item for KeyMapping '. print_r($keyValues, true));
			}
		}

		if ($item == null && !empty($itemClassName) && !empty($itemClassId) && !empty($itemFolder)) {
			$item = $this->createObject($itemClassName, $itemClassId, $itemFolder, $key);
		}

		return $item;
	}

	/**
	 * Erstellt ein neues Object
	 */
	private function createObject($className, $classId, $parentId, $key) {
		$realClassName = "Object_" . ucfirst($className);
		$key = File::getValidFilename($key);
		$key = $this->getIdPrefix() . $key;

		$parent = Concrete::getById($parentId);
		$intendedPath = $parent->getFullPath() . "/" . $key;


		if (Service::pathExists($intendedPath)) {
			$i = 1;
			$intendedPath = $parent->getFullPath() . "/" . $key . "-" . $i;
			while (Service::pathExists($intendedPath) && $i < 1000) {
				$i++;
				$intendedPath = $parent->getFullPath() . "/" . $key . "-" . $i;
			}
			$key .= '-'. $i;
		}


		if (!Service::pathExists($intendedPath)) {
			$object = new $realClassName();
			if ($object instanceof Concrete) {
				$object->setOmitMandatoryCheck(true); // allow to save the object although there are mandatory fields
			}
			$object->setClassId($classId);
			$object->setClassName($className);
			$object->setParentId($parentId);
			$object->setKey($key);
			$object->setCreationDate(time());
			$admin = User::getByName('admin');
			$object->setUserOwner($admin->getId());
			$object->setUserModification($admin->getId());
			$object->setPublished(false);
			$object->setType(AbstractObject::OBJECT_TYPE_OBJECT);

			return $object;
		}

		Logger::error("prevented creating PIM object because object with same path+key [ $intendedPath ] already exists");
        return null;
	}

	private function getIdPrefix() {
		return $this->idPrefix;
	}

	/**
	 * Liefert die Mappingvorschriften zurück, die für Produktzuordnung von der angegeben Quell- zur Zielkategorie
	 * angewendet werden müssen
	 */
	private function getMappingForCategory($sourceCategoryId) {
		return Helper::findMappings($this->dataportId, $sourceCategoryId, md5_file($this->file));
	}

	/**
	 * @param $xp \DOMXPath
	 * @param $xp array
	 * Hilfsfunktion für collectDataFromBMECAT()
	 * Baut ein Array für die Schlüssel Attribute zusammen
	 */
	private function collectKeyMappings($xp, $xpathExpr, \DOMElement $domNode, $mapping, $currentData) {
		$data = $currentData;

		foreach ($mapping as $attributeName => $mappingData) {
			if ($mappingData['keyMapping']) {
				$data[$attributeName] = $this->generateValueArray($xp, $xpathExpr, $domNode, $mappingData["xpath"], $attributeName);
				$data[$attributeName]["mappings"] = $mappingData;
			}
		}
		return $data;
	}

	/**
	 * @param $xp \DOMXPath
	 * @param $xp array
	 * Hilfsfunktion für collectDataFromBMECAT()
	 * Baut ein Array für die Attribute von class oder einer fieldCollection zusammen
	 */
	private function collectAttributeData($xp, $xpathExpr, \DOMElement $domNode, $mapping, $currentData, $mappingGroupName) {
		$data = $currentData;
		foreach ($mapping as $attributeName => $mappingData) {
			$values = $this->generateValueArray($xp, $xpathExpr, $domNode, $mappingData["xpath"], $attributeName, $mappingGroupName);
			if ($values != null) {
				$data[$attributeName] = $values;
				$data[$attributeName]["mappings"] = $mappingData;
			}
		}
		return $data;
	}

	/**
	 * @param $xp \DOMXPath
	 * @param $xp array
	 * Hilfsfunktion für collectDataFromBMECAT()
	 * Baut ein Array für ein generisches Attribut zusammen
	 */
	private function generateValueArray($xp, $xpathExpr, \DOMElement $domNode, $xPath, $attributeName, $mappingGroupName = "") {
		$data = array();
		if (strpos($xPath, "price:") !== false) {
			$priceData = explode(":", $xPath);
			return $this->generateValueArrayPrice($xp, $xpathExpr, $domNode, $priceData[1]);
		}
		if (strpos($xPath, "mime:") !== false) {
			$mimeData = explode(":", $xPath);
			return $this->generateValueArrayMime($xp, $xpathExpr, $domNode, $mimeData[1]);
		}
		if (strpos($xPath, "reference:") !== false) {
			$referenceData = explode(":", $xPath);
			$this->generateArrayReferences($xp, $xpathExpr, $domNode, $referenceData[1], $attributeName, $mappingGroupName);
			return null;
		}


		try {
			$node = null;
			if (!empty($xPath)) {
				$node = $xp->query($xPath, $domNode)->item(0);
			}
		} catch (\Exception $e) {
			$data["value"] = "No value. Match error!";
			$data["valueArray"] = array();
			return $data;
		}
		if ($node == null) {
			$data["value"] = '';
			$data["valueArray"] = array();
			return $data;
		}

		if ($node->localName === "FEATURE") {
			if ($node->getElementsByTagName("FVALUE")->length == 1 && $node->getElementsByTagName("FUNIT")->length == 0) {
				$fvalue = $node->getElementsByTagName("FVALUE")->item(0);
				$value = $fvalue != null ? $fvalue->nodeValue : '';
				$valueArray = null;
			} else {
				$valueArray = array("value" => array(), "unit" => array());
				foreach ($node->getElementsByTagName("FUNIT") as $funit) {
					$value = $funit->nodeValue;
					array_push($valueArray["unit"], $funit->nodeValue);
				}
				foreach ($node->getElementsByTagName("FVALUE") as $fvalue) {
					$value = $fvalue->nodeValue;
					array_push($valueArray["value"], $fvalue->nodeValue);
				}
			}
		} else {
			$value = $node->nodeValue;
			$valueArray = array();
		}
		$data["value"] = $value;
		$data["valueArray"] = $valueArray;
		return $data;
	}

	/**
	 * @param $xp \DOMXPath
	 * @param $xp array
	 * Hilfsfunktion für collectDataFromBMECAT()
	 * Baut ein Array für einen Preis zusammen
	 */
	private function generateValueArrayPrice($xp, $xpathExpr, \DOMElement $domNode, $priceType) {
		$xPath = sprintf($xpathExpr['product']['priceDetails']['byType'], $priceType);
		$priceDetailsNodeList = $xp->query($xPath, $domNode);

		$value = "";
		$valueArray = array();

		foreach ($priceDetailsNodeList as $priceDetailsNode) {
			$priceDetailArray = array();
			$startdateNode = $xp->query($xpathExpr['product']['priceDetails']['startDate'], $priceDetailsNode);
			$priceDetailArray["startdate"] = $startdateNode->item(0) != null ? $startdateNode->item(0)->nodeValue : '';
			$enddateNode = $xp->query($xpathExpr['product']['priceDetails']['endDate'], $priceDetailsNode);
			$priceDetailArray["enddate"] = $enddateNode->item(0) != null ? $enddateNode->item(0)->nodeValue : '';

			$priceNodeList = $xp->query(sprintf($xpathExpr['product']['priceDetails']['priceByType'], $priceType), $priceDetailsNode);
			$priceDetailArray["prices"] = array();
			foreach ($priceNodeList as $priceNode) {
				$priceArray = array();

				$amountNode = $xp->query($xpathExpr['product']['priceAmount'], $priceNode);
				$priceArray["amount"] = $amountNode->item(0) != null ? $amountNode->item(0)->nodeValue : '';
				$value = $amountNode->item(0) != null ? $amountNode->item(0)->nodeValue : '';
				$currencyNode = $xp->query($xpathExpr['product']['priceCurrency'], $priceNode);
				$priceArray["currency"] = $currencyNode->item(0) != null ? $currencyNode->item(0)->nodeValue : '';
				$taxNode = $xp->query("TAX", $priceNode);
				$priceArray["tax"] = $taxNode->item(0) != null ? $taxNode->item(0)->nodeValue : '';
				$factorNode = $xp->query($xpathExpr['product']['priceFactor'], $priceNode);
				$priceArray["factor"] = $factorNode->item(0) != null ? $factorNode->item(0)->nodeValue : '';
				$lowerBoundNode = $xp->query($xpathExpr['product']['priceLowerBound'], $priceNode);
				$priceArray["lowerbound"] = $lowerBoundNode->item(0) != null ? $lowerBoundNode->item(0)->nodeValue : '';
				$teritoryNodeList = $xp->query($xpathExpr['product']['priceTerritory'], $priceNode);
				$priceArray["teritories"] = array();
				foreach ($teritoryNodeList as $teritoryNode) {
					array_push($priceArray["teritories"], $teritoryNode->nodeValue);
				}

				array_push($priceDetailArray["prices"], $priceArray);
			}
			array_push($valueArray, $priceDetailArray);
		}
		$data["value"] = $value;
		$data["valueArray"] = $valueArray;
		return $data;
	}

	/**
	 * @param $xp \DOMXPath
	 * @param $xp array
	 * Hilfsfunktion für collectDataFromBMECAT()
	* Baut ein Array für einen Preis zusammen
	*/
	private function generateValueArrayMime($xp, $xpathExpr, \DOMElement $domNode, $mimeType) {
		$xPath = $xpathExpr['product']['mimeBase'];
		try {
			$nodeList = $xp->query($xPath, $domNode);
		} catch (\Exception $e) {
			$data["value"] = "No value. Match error!";
			$data["valueArray"] = array();
			return $data;
		}
		$value = "";
		$valueArray = array();
		foreach ($nodeList as $node) {
			$nodeArray = array();

			$sourceNode = $xp->query($xpathExpr['product']['mimeSource'], $node);
			$nodeArray["source"] = $sourceNode->item(0) != null ? $sourceNode->item(0)->nodeValue : '';
			if (Helper::getAssetType($nodeArray["source"]) != $mimeType) {
				continue;
			}
			$tpyeNode = $xp->query($xpathExpr['product']['mimeType'], $node);
			$nodeArray["type"] = $tpyeNode->item(0) != null ? $tpyeNode->item(0)->nodeValue : '';
			$descrNode = $xp->query($xpathExpr['product']['mimeDescr'], $node);
			$nodeArray["descr"] = $descrNode->item(0) != null ? $descrNode->item(0)->nodeValue : '';
			$altNode = $xp->query($xpathExpr['product']['mimeAlt'], $node);
			$nodeArray["alt"] = $altNode->item(0) != null ? $altNode->item(0)->nodeValue : '';
			$purposeNode = $xp->query($xpathExpr['product']['mimePurpose'], $node);
			$nodeArray["purpose"] = $purposeNode->item(0) != null ? $purposeNode->item(0)->nodeValue : '';

			$value = $nodeArray["source"];

			array_push($valueArray, $nodeArray);
		}
		$data["value"] = $value;
		$data["valueArray"] = $valueArray;
		return $data;
	}


	/**
	 * @param $xp \DOMXPath
	 * @param $xp array
	 * Hilfsfunktion für collectDataFromBMECAT()
	 * Füllt das Array für Referenzen auf
	 */
	private function generateArrayReferences($xp, $xpathExpr, \DOMElement $domNode, $referenceType, $attributeName, $mappingGroupName) {
		$keyList = $xp->query($xpathExpr['globalAttributes']['aid']['id'], $domNode);
		$artId = $keyList->item(0)->nodeValue;

		$xPath = sprintf($xpathExpr['product']['referenceByType'] . '/' . $xpathExpr['product']['referenceId'], $referenceType);
		$referenceNodeList = $xp->query($xPath, $domNode);

		if ($referenceNodeList->length > 0) {
			$key = $artId . '#' . $mappingGroupName . '#' . $attributeName;

			if (!array_key_exists($key, $this->referencesList)) {
				$this->referencesList[$key] = array(
					'artId' => $artId,
					'fieldCollection' => $mappingGroupName,
					'attributeName' => $attributeName,
					'references' => array(),
				);
			}

			foreach ($referenceNodeList as $referenceNode) {
				if (!in_array($referenceNode->nodeValue, $this->referencesList[$key]['references'])) {
					$this->referencesList[$key]['references'][] = $referenceNode->nodeValue;
				}
			}
		}

		return null;
	}

	/**
	 * Ordnet das übergebene Produkt allen Kategorien zu die durch das Mapping festgelegt sind
	 * @param $item Concrete
	 * @param $artId
	 * @param $categoryClassName
	 * @param $productListElementName
	 * @return bool
	 */
	private function setCategories($item, $artId, $categoryClassName, $productListElementName) {
		if ($this->categoryDummy == null) {
			$BMEcategories = $this->xp->query(sprintf($this->xpathExpr['mapping']['categoriesToProduct'], $artId));
		} else {
			$BMEcategories = $this->categoryDummy;
		}

		$productHasCategory = false;
		foreach ($BMEcategories as $BMEcategory) { // Iterate over all BMECategoies for this Item
			$BMEcategoryId = $BMEcategory->nodeValue;
			$mappingArray = $this->getMappingForCategory($BMEcategoryId);

			foreach ($mappingArray as $PIMcategory => $mappingGroups) { // Iterate over all PIMCategories for this BMECategory
				$cacheKey = 'blackbit_' . $this->importKey . '_category_' . $PIMcategory;

				if (($category = Cache::load($cacheKey)) === false) {
					$category = Concrete::getById($PIMcategory);
					Cache::save($category, $cacheKey, array('bmecat_import_' . $this->importKey));
				}

				if ($category instanceof Concrete && $category->getClassName() == $categoryClassName) {

					// Zuordnung nur zu Blättern
					if (count($category->getChildren()) == 0) {

						// If this is a new item, it has to be saved first in order to generate an ID
						if (!$item->getId()) {
							$item->save();
						}

						$productHasCategory = true;
						$key = 'cat' . $category->getId();
						if (!array_key_exists($key, $this->categoryMappings)) {
							$this->categoryMappings[$key] = array();
						}

						if (!array_key_exists($productListElementName, $this->categoryMappings[$key])) {
							$this->categoryMappings[$key][$productListElementName] = array();
						}

						if (!in_array($item->getId(), $this->categoryMappings[$key][$productListElementName])) {
							$this->categoryMappings[$key][$productListElementName][] = $item->getId();
						}
					}
				}
			}
		}

		return $productHasCategory;
	}

	/**
	 * Calculates publishing status for given item
	 * @param Concrete $item
	 * @param $itemStatusFormula
	 * @param $BMEData
	 */
	private function updateStatus(Concrete $item, $itemStatusFormula, $BMEData) {
		// Status ermitteln
		$published = $item->getPublished();
		if (!empty($itemStatusFormula) && class_exists("JSContext")) {
			$currentObjectValues = $this->getAttributesArrayForObject($item);
			try {
				// $BMEData uis a huge array, if this results in runtime problems you can remove it from the JSContext and provide singel values that are actualy required
				$js = new \JSContext();
				$js->assign('BMEData', $BMEData);
				$js->assign('currentStatus', $published);
				$js->assign('currentObjectValues', $currentObjectValues);
				$published = ($js->evaluateScript($this->preJs . "\n\n" . $itemStatusFormula) == '1');
			} catch (\Exception $ex) {
				Logger::error("Error while calculating item status: {$ex}");
			}
		}
		$item->setPublished($published);
	}

	/*
	 * Fügt die Daten in die Attribute eines Objektes ein
	 */
	private function addData(Concrete $item, $BMEData, $itemAttributeElement, $articleData) {

		$currentValues = $this->getAttributesArrayForObject($item);
		foreach ($BMEData as $collectionName => $collectionValues) {

			// set class fiels or find collection and set collection fields
			if ($collectionName === "class") {
				$addToObject = $item;
			} else {
				$getter = 'get' . ucfirst($itemAttributeElement);
				$setter = 'set' . ucfirst($itemAttributeElement);
                $realCollectionName = 'Pimcore\\Model\\Object\\Fieldcollection\Data\\'. ucfirst($collectionName);


                // if collection element exists check if specific collection exists, if not -> create it
				// if no collection element exists -> create it and the collection
                /** @var Fieldcollection $itemCollectionsElement */
				$itemCollectionsElement = $item->$getter();
				if ($itemCollectionsElement !== null) {
					$currentCollections = $itemCollectionsElement->getItems();
					$theCollection = null;
					foreach ($currentCollections as $collection) {
						if ($collection instanceof $realCollectionName) {
							$theCollection = $collection;
						}
					}
					if ($theCollection == null) {
						$theCollection = new $realCollectionName();
						$itemCollectionsElement->add($theCollection);
						$item->setOmitMandatoryCheck(true);
						$item->save();
					}
				} else {
					$itemCollectionsElement = new Fieldcollection();
					$theCollection = new $realCollectionName();
					$itemCollectionsElement->add($theCollection);
					$item->$setter($itemCollectionsElement);
					$item->setOmitMandatoryCheck(true);
					$item->save();
				}
				// end if -> $theCollection contains a valid collection
				$addToObject = $theCollection;

			}
			if (is_array($collectionValues)) {
				foreach ($collectionValues as $fieldName => $fieldValue) {
					$this->setFieldData($addToObject, $fieldName, $fieldValue, $currentValues, $articleData);
				}
			}
		}
	}


	/*
	 * Hilfsfunktion  für addData()
	 * Übernimmt das eigendliche setzten der Daten in einer collection
	 */
	private function setFieldData($item, $fieldName, $fieldValue, $currentValues, $articleData) {
		try {
			// find language and remove language from $fieldName
			if (strpos($fieldName, "#")) {
				$language = $fieldValue['mappings']['locale'];
				$fieldName = substr($fieldName, 0, strpos($fieldName, "#"));
				$validLocales = Tool::getValidLanguages();
				if (!in_array($language, $validLocales)) {
					return;
				}
			} else {
				$language = "";
			}

			// Get current Value
			$getter = 'get' . ucfirst($fieldName);

			// Check if target property exists
			if (!method_exists($item, $getter)) {
				Logger::warning("Invalid mapping to non-existent field '" . $fieldName . "'");
				return;
			}

			$currentValue = '';
			try {
				$currentValue = $item->$getter($language);
			} catch (\Exception $e) {
				Logger::warn('Unable to get value for field "' . $fieldName . '" ' . $e->getMessage());
			}

			// Process JS
			$value = Importer::processJS($fieldValue["mappings"]["calculation"], $fieldValue["value"],
				$fieldValue["valueArray"], $this->preJs, $articleData, $currentValue, $currentValues);

			// Add Prefix for Key Fields
			if ($fieldValue["mappings"]["keyMapping"]) {
				$value = $this->getIdPrefix() . $value;
			}

			// Convert type of new Value
			if ($item instanceof Fieldcollection\Data\AbstractData) {
			    /** @var Data $fieldDefinition */
				$fieldDefinition = $item->getDefinition()->getFieldDefinition($fieldName);
				$fieldType = $fieldDefinition->getFieldtype();
			} else {
			    /** @var Concrete $item */
				$fieldDefinition = $item->getClass()->getFieldDefinition($fieldName);
				/** @var Data\Localizedfields $localized */
				$localized = $item->getClass()->getFieldDefinition('localizedfields');
				if (!$fieldDefinition && $localized) {
					$definitions = $localized->getFieldDefinitions();
					$fieldDefinition = $definitions[$fieldName];
				}
				$fieldType = $fieldDefinition->getFieldtype();
			}
			$value = $this->convertType($value, $currentValue, $fieldValue["mappings"]["format"], $fieldType);


			// Set new Value to field in item
			if ($fieldType === "multihref") {
                $allTypesAreAllowed = true;
				if (is_array($value)) {
					$allowedAassetTypes = $item->getClass()->getFieldDefinition($fieldName)->getAssetTypes();
					foreach ($value as $valueElement) {
						$elmentAllowed = false;
						foreach ($allowedAassetTypes as $allowedType) {
							if ($allowedType["assetTypes"] == $valueElement->getType()) {
								$elmentAllowed = true;
							}
						}
						if ($elmentAllowed == false) {
							$allTypesAreAllowed = false;
						}
					}
				}

				if (!$allTypesAreAllowed) {
					Logger::warn("Type conflict in field '{$fieldName}'. Please check if allowed assets for this multihref match the imported assets.");
					return;
				}
			}

			$setter = 'set' . ucfirst($fieldName);
			try {
				$item->$setter($value, $language);
			} catch (\Exception $e) {
				Logger::warn("Unable to set value for field '{$fieldName}. Method '{$setter}' is not defined");
			}
		} catch (\Exception $e) {
			Logger::error('Unable to set field data for field "{$fieldName}"');
		}
	}

	/*
	 * Hilfsfunktion  für addData() und setFieldData()
	 * Konvertiert die JS Rückgabe anhand des Typs des Ziel-Attributs
	 */
	private function convertType($rawValue, $currentValue, $format, $fieldType) {
		$value = $rawValue;

		switch ($fieldType) {
			case 'numeric':
				$decimalSep = null;
				$groupingSep = null;

				if ($format) {
					if (array_key_exists("decimalSeparator", $format)) {
						$decimalSep = $format['decimalSeparator'];
					}

					if (array_key_exists("groupingSeparator", $format)) {
						$groupingSep = $format['groupingSeparator'];
					}
				}

				$value = $this->parseNumber($value, $decimalSep, $groupingSep);
				break;

			case 'date':
			case 'datetime':
				if ($format && array_key_exists("dateFormat", $format)) {
					$date = \DateTime::createFromFormat($format['dateFormat'], $value);

					if ($date) {
						if ($fieldType === "date") {
							$date->setTime(0, 0);
						}
						$value = $date->getTimestamp();
					} else {
						$value = strtotime($value);
					}
				} else {
					$value = strtotime($value);
				}
				$value = new \DateTimeImmutable($value);
				break;

			case 'multiselect':
				$separator = null;
				if (array_key_exists("separator", $format)) {
					$separator = $format['separator'];
				}

				if ($separator) {
					$value = explode($separator, $value);
				} else {
					$value = (array)$value;
				}
				break;

			case 'image':
				if (!is_array($value)) {
					$value = $this->getAsset($value, $currentValue);
				} else {
					if (isset($value[0]["source"])) {
						$value = $this->getAsset($value[0]["source"], $currentValue);
					} else {
						$value = $this->getAsset($value[0], $currentValue);
					}
				}
				if (!($value instanceof Image)) {
					$value = $currentValue;
				}
				break;
			case 'multihref':
				if (!is_array($value)) {
					$value = array($value);
				}

				foreach ($value as $assetSource) {
					if (is_array($assetSource) && isset($assetSource["source"])) {
						$assetSourceURI = $assetSource["source"];
					} else {
						$assetSourceURI = $assetSource;
					}
					$assetToUpdate = null;
					$assetToUpdateId = -1;
					foreach ($currentValue as $key => $currentAsset) {
						$sourcePath = $currentAsset->getProperty("sourcePath");
						if ($sourcePath == $assetSourceURI) {
							$assetToUpdateId = $key;
							$assetToUpdate = $currentAsset;
						}
					}
					$newAsset = $this->getAsset($assetSourceURI, $assetToUpdate);

					if ($newAsset instanceof Asset) {
						$newAsset->setProperty("sourcePath", "Text", $assetSourceURI);
						$newAsset->save();

						if ($assetToUpdateId > -1) {
							$currentValue[$assetToUpdateId] = $newAsset;
						} else {
							array_push($currentValue, $newAsset);
						}
					}
				}

				$value = $currentValue;
				break;

		}
		return $value;
	}


	/*
	 * Versucht ein Asset zu laden
	 * Falls das neue Asset nicht geladen werden kann wird $currentValue zurück gegeben
	 */
	private function getAsset($value, $currentValue) {
		if (filter_var($value, FILTER_VALIDATE_URL) !== false) {
			try {
				$url = $value;
                $data = Tool::getHttpData($url);
			} catch (\Exception $e) {
				Logger::error("Unable to load uri '" . $value . "'");
				$value = $currentValue;
				return $value;
			}
		} else {
			$url = "file://" . $this->assetSource . "/" . $value;
			if (file_exists($url)) {
				$data = file_get_contents($url);
			} else {
				Logger::error("Unable to load uri '" . $value . "'");
				$value = $currentValue;
				return $value;
			}
		}

		if ($currentValue instanceof Asset) {
			$currentValue->setData($data);
			$currentValue->save();
			$value = $currentValue;
		} else {
			$pathPrefix = '/';
			if ($this->assetFolder instanceof Folder) {
				$pathPrefix = $this->assetFolder->getFullPath() . '/';
			}

			$filename = $value;
			$filename = File::getValidFilename($filename);
			$newFileName = $filename;
			$existing = null;
			$i = 1;
			do {
				$existing = Asset::getByPath($pathPrefix . $newFileName);
				if ($existing instanceof Asset) {
					$newFileName = $i . '-' . $filename;
					$i++;
				}
			} while (!is_null($existing) && $i <= 100);

			$parent = ($this->assetFolder instanceof Folder) ? $this->assetFolder->getId() : 1;
			$value = Asset::create($parent, array(
				'filename' => $newFileName,
				'data' => $data,
				'userOwner' => 1,
				'userModification' => 1,
			));
		}

		return $value;
	}

	/*
	 * Liefert alle aktuellen Werte eines Objekts als Array zurück
	 */
	public function getAttributesArrayForObject(Concrete $object, $attributes = array(), Data\Localizedfields $localizedFields = null, $language = "") {
		if ($localizedFields == null) {
			$fieldList = $object->getClass()->getFieldDefinitions();
			$languageNameTag = "";
		} else {
			$fieldList = $localizedFields->getFieldDefinitions();
			$languageNameTag = $language;
			if ($languageNameTag !== "") {
				$languageNameTag = '#' . $languageNameTag;
			}
		}

		foreach ($fieldList AS $name => $class) {
			if ($class instanceof Data\Input ||
				$class instanceof Data\Numeric ||
				$class instanceof Data\Checkbox ||
				$class instanceof Data\Select ||
				$class instanceof Data\Country ||
				$class instanceof Data\Language ||
				$class instanceof Data\Textarea ||
				$class instanceof Data\Wysiwyg
			) {
				$func = "get" . ucfirst($name);
				$attributes[$name . $languageNameTag] = $object->$func($language);
			} elseif ($class instanceof Data\Multiselect) {
				$func = "get" . ucfirst($name);
				if (is_array($object->$func($language))) {
					$attributes[$name . $languageNameTag] = implode(";", $object->$func($language));
				} else {
					$attributes[$name . $languageNameTag] = "";
				}
			} elseif ($class instanceof Data\Image) {
				$func = "get" . ucfirst($name);
				/** @var Asset $asset */
				$asset = $object->$func($language);
				if ($asset != null) {
					$attributes[$name . $languageNameTag] = $asset->getFullPath();
				} else {
					$attributes[$name . $languageNameTag] = null;
				}
			} elseif ($class instanceof Data\Datetime ||
				$class instanceof Data\Date
			) {
				$func = "get" . ucfirst($name);
				$datetime = $object->$func($language);
				if ($datetime != null) {
					$attributes[$name . $languageNameTag] = $datetime->get();
				} else {
					$attributes[$name . $languageNameTag] = null;
				}
			} elseif ($class instanceof Data\Link) {
				$func = "get" . ucfirst($name);
				$datetime = $object->$func($language);
				if ($datetime != null) {
					$attributes[$name . $languageNameTag] = $datetime->getHref();
				} else {
					$attributes[$name . $languageNameTag] = null;
				}
			} elseif ($class instanceof Data\Localizedfields) {
				// für abwärts kompatibilität und Anwendungen mit nur einer Sprache, werden die Texte der ersten Sprache ohne LanguageTag hinzugefügt
				$attributes = $this->getAttributesArrayForObject($object, $attributes, $class, $language);
				// Fügt für alle eingestellten Sprachen die sprachabhänigen Attriobute ein
				$validLanguages = Tool::getValidLanguages();
				foreach ($validLanguages as $languageTmp) {
					$attributes = $this->getAttributesArrayForObject($object, $attributes, $class, $languageTmp);
				}
			} else {
				Logger::info("Attribute $name of type " . get_class($class) . " is not transferd into JS array.");
			}
		}

		return $attributes;
	}


	/**
	 * Converts a string to a number
	 * @param string $value the string to be converted
	 * @param string $decimalSep decimal separator
	 * @param string $groupingSep grouping separator
	 * @return float the resulting number
	 */
	private function parseNumber($value, $decimalSep = null, $groupingSep = null) {
		if (!empty($groupingSep)) {
			$value = str_replace($groupingSep, '', $value);
		}

		if (!empty($decimalSep)) {
			$value = str_replace($decimalSep, '.', $value);
		}

		return (float)$value;
	}

	public function debug($msg) {
		echo $msg . PHP_EOL;
	}

	private function debugTimings($name, $timings) {
		$msg = $name;
		foreach ($timings as $key => $value) {
			if ($key === 'id') {
				$msg .= vsprintf('%9s: %6s', array($key, $value));
			} else {
				$msg .= vsprintf('%9s: %4f', array($key, $value));
			}
			$msg .= ' ';
		}

		$this->debug($msg);
	}

	/**
	 * @return int
	 */
	public function getDataportId() {
		return $this->dataportId;
	}

	/**
	 * @param int $dataportId
	 */
	public function setDataportId($dataportId) {
		$this->dataportId = $dataportId;
	}

	/**
	 * @return string
	 */
	public function getFile() {
		return $this->file;
	}

	/**
	 * @param string $file
	 */
	public function setFile($file) {
		$this->file = $file;
	}

	/**
	 * @return string
	 */
	public function getImportKey() {
		return $this->importKey;
	}

	/**
	 * @param string $importKey
	 */
	public function setImportKey($importKey) {
		$this->importKey = $importKey;
	}

	/**
	 * @param string $idPrefix
	 */
	public function setIdPrefix($idPrefix) {
		$this->idPrefix = $idPrefix;
	}
}