<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\Controller;

use Blackbit\PimBundle\Tools\Installer;
use Blackbit\PimBundle\lib\Pim\Feature;
use Blackbit\PimBundle\lib\Pim\Helper;
use Blackbit\PimBundle\lib\Pim\Import\Worker\Bmecat;
use Blackbit\PimBundle\lib\Pim\Item\Bmecat\Importer;
use Blackbit\PimBundle\model\BmeCategory;
use Blackbit\PimBundle\model\BmeFile;
use Blackbit\PimBundle\model\BmeProduct;
use Blackbit\PimBundle\model\BmeProductCategory;
use Blackbit\PimBundle\model\Categorymapping;
use Blackbit\PimBundle\model\CategorymappingField;
use Blackbit\PimBundle\model\Dataport;
use Pimcore\Cache;
use Pimcore\Controller\FrontendController;
use Pimcore\Db;
use Pimcore\Logger;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\ClassDefinition;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\DataObject\Fieldcollection\Definition;
use Pimcore\Model\DataObject\Folder;
use Pimcore\Model\DataObject\Listing;
use Pimcore\Tool;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Psr\Log\LoggerInterface;

/**
 * Konfiguration von BMEcat-Imports
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Medien GmbH, http://www.blackbit.de/
 * @Route("/Bmecat")
 */
class BmecatController extends FrontendController {
    /** @var LoggerInterface */
    private $logger;

	public function __construct(LoggerInterface $logger) {
		Feature::checkFeature(Feature::BMECAT);
		$this->logger = $logger;
	}

	/**
	 * Aus $file sollen die tatsächlichen Kategorien ausgelesen werden.
	 * Im Request wird im Parameter node die ID des Nodes übertragen, für den die Kinder zurückgeliefert werden müssen.
	 * Spezifikation siehe http://docs.sencha.com/extjs/3.4.0/#!/api/Ext.tree.TreeLoader
	 * Für den Root-Node wird als node-Parameter "_root" übermittelt
     *
     * @Route("/get-rawdata-tree")
	 */
	public function getRawdataTreeAction(Request $request) {
		$data = array();
		$dataportId = $request->get('dataportId');
		$file = Helper::getFileForDataport($dataportId);
		$node = $request->get('node');
		$filter = $request->get('filter');

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		if (empty($node) || !$dataport) {
			return new JsonResponse(array());
		}

		$version = null;
		if ($dataport && !empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);
			if (array_key_exists('version', $config)) {
				$version = $config['version'];
			}
		}

		$xpathExpr = Helper::getBmecatXpathForVersion($version);
		if (empty($xpathExpr)) {
			$this->logger->warning('Unsupported BMEcat version "' . $version . '"');
			return new JsonResponse(array());
		}

		$hash = md5_file($file);

		$bmeFileRepository = new BmeFile();
		$existing = $bmeFileRepository->countRows(['hash' => $hash]);
		if ($existing <= 0) {
			return new JsonResponse(array());
		}

        $bmeCategoryRepository = new BmeCategory();
		if ($node === '_root') {
			$children = $bmeCategoryRepository->find([
			        'file = ?' => $hash,
                    'parentId IS NULL' => ''
            ]);

			if (empty($children)) {
				// BMEcat enthält keinen Kategoriebaum
				$data[] = array(
					'id' => '_dummy',
					'name' => 'Dummykategorie (BMEcat enthält keinen Kategoriebaum)',
					'iconCls' => 'task',
					'itemCount' => 0,
					'dataType' => 'category',
					'leaf' => true,
				);
				return new JsonResponse($data);
			}
		} else {
			$children = $bmeCategoryRepository->find([
                'file = ?', $hash,
                'parentId IS NULL' => ''
            ]);
		}

		$autoExpandCategories = array();

		$filter = str_replace('%', '', $filter);
		if (!empty($filter)) {
			$filteredCategories = $bmeCategoryRepository->find([
                'file = ?' => $hash,
                'LOWER(`name`) LIKE ?' => mb_strtolower($filter).'%'
            ]);
			foreach ($filteredCategories as $cat) {
				$autoExpandCategories = array_merge($autoExpandCategories, Helper::findBmecatParentsCache($cat['categoryId'], $hash));
			}
		}

		$autoExpandCategories = array_unique($autoExpandCategories);


		$result = array();

		$bmeProductCategoryRepository = new BmeProductCategory();
		// Kategorien auslesen
		foreach ($children as $child) {
			$catId = $child['categoryId'];
			$catName = $child['name'];
			$catParent = $child['parentId'];

			$mappingModel = new Categorymapping();
			$mappings = array();
			$existingMappings = $mappingModel->find(array(
				'dataportId = ?' => $dataport['id'],
				'sourceCategoryId = ?' => $catId,
			));

			foreach ($existingMappings as $m) {
				$mappings[] = (int)$m['targetCategoryId'];
			}

			$childCategories = $bmeCategoryRepository->countRows([
			        'file = ?' => $hash,
                    'parentId = ?' => $catId
            ]);


			$products = $bmeProductCategoryRepository->find([
			        'file = ?' => $hash,
                    'categoryId = ?' => $catId
            ]);

			$hasChildren = $childCategories > 0 || count($products) > 0;

			// Kindkategorien rekursiv auflösen
			$categoryIds = $this->findSubCategoriesCache($hash, array($catId));
			$categoryIds[] = $catId;

			$subtreeChildren = $bmeProductCategoryRepository->find([
                'file' => $hash,
                'categoryId IN (?)', $categoryIds
            ]);

			// Artikel-IDs für die angegebene Kategorie (und Kinder) finden
			$itemIds = array();
			foreach ($subtreeChildren as $subtreeChild) {
				$itemIds[] = $subtreeChild['productId'];
			}

			$itemIds = array_unique($itemIds);

			$def = array(
				'id' => $catId,
				'name' => $catName,
				'iconCls' => $hasChildren ? 'task-folder' : 'task',
				'itemCount' => count($itemIds),
				'dataType' => 'category',
				'mappings' => $mappings,
			);

			if ($hasChildren) {
				// root immer ausklappen
				$def['expanded'] = ($node === '_root' || in_array($catId, $autoExpandCategories));
			} else {
				$def['leaf'] = true;
			}

			$result[] = $def;
		}


		if ($node !== '_root') {
		    $db = Db::get();
			// Artikel-IDs für die angegebene Kategorie finden
			$select = $db->select()
				->from(array('p' => Installer::TABLE_BME_PRODUCT), 'p.xml')
				->join(array('p2c' => Installer::TABLE_BME_PRODUCT_CATEGORY), 'p2c.file = p.file AND p2c.productId = p.productId', null)
				->columns('p.productId')
				->where('p2c.categoryId = ?', array($node))
				->where('p2c.file = ?', array($hash));
			$items = $db->fetchAll($select);

			foreach ($items as $item) {
				$itemId = $item['productId'];
				$name = '';

				try {
					$xml = @new \SimpleXMLElement($item['xml']);
					if ($xml instanceof \SimpleXMLElement) {
						$nameNode = $xml->xpath($xpathExpr['globalAttributes']['shortDescription']['id']);

						if (!empty($nameNode)) {
							$name = (string)$nameNode[0];
						}
					}
				} catch (\Exception $e) {
					$this->logger->error('Unable to get product data for item "' . $itemId . '" from cache');
				}

				$result[] = array(
					'id' => $node . ': ' . $itemId,
					'realId' => $itemId,
					'name' => $name,
					'iconCls' => 'pim_icon_product',
					'itemCount' => '',
					'dataType' => 'product',
					'leaf' => true,
				);
			}
		}

		return new JsonResponse($result);
	}


	/**
	 * Liefert aus dem PIM-Kategoriebaum für den angegebenen Datenport die Kinder des übergebenen Nodes.
	 * Root des Baumes ist das im Datenport konfigurierte Rootelement.
	 * Spezifikation siehe http://docs.sencha.com/extjs/3.4.0/#!/api/Ext.tree.TreeLoader
	 * Für den Root-Node wird als node-Parameter "_root" übermittelt
     *
     * @Route("/get-pim-tree")
	 */
	public function getPimTreeAction(Request $request) {
		$data = array();

		$dataportId = $request->get('dataportId');
		$node = $request->get('node');
		$filter = $request->get('filter');

		if (empty($node)) {
			return new JsonResponse(array());
		}

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		if (!$dataport || empty($dataport['sourceconfig'])) {
			return new JsonResponse(array());
		}

		$config = unserialize($dataport['sourceconfig']);

		$rootFolder = Folder::getByPath($config['categoryfolder']);
		if (!($rootFolder instanceof Folder)) {
			return new JsonResponse(array());
		}

		$children = array();
		if ($node === '_root') {
			$children = $rootFolder->getChildren();
		} else {
			$object = Concrete::getById($node);

			if ($object instanceof Concrete) {
				$children = $object->getChildren();
			}
		}


		$nameGetter = empty($config['categoryNameElement']) ? null : ('get' . ucfirst($config['categoryNameElement']));

		$autoExpandCategories = array();

		if (!empty($filter)) {
			$filteredCategories = array();

			$categoryClass = ClassDefinition::getById($config['categoryClassId']);
			if ($categoryClass instanceof ClassDefinition) {
                $listClass = 'Pimcore\\Model\\DataObject\\' . ucfirst($categoryClass->getName()).'\\Listing';

				if (@class_exists($listClass)) {
					/**
					 * @var $list Listing
					 */
					$list = new $listClass();
					$list->setCondition('o_path LIKE "' . $rootFolder->getFullPath() . '/%"');

					foreach ($list as $cat) {
						if ($cat instanceof Concrete && $cat->getClassId() == $config['categoryClassId']) {
							try {
								$catName = $cat->$nameGetter();

								if (strpos(strtolower($catName), strtolower($filter)) !== false) {
									$filteredCategories[] = $cat->getId();
								}
							} catch (\Exception $e) {
								// Ignore
								$this->logger->error('Object "' . $cat->getFullPath() . '" has no method "' . $nameGetter . '()');
							}
						}
					}
				}
			}

			foreach ($filteredCategories as $catId) {
				$autoExpandCategories = array_merge($autoExpandCategories, Helper::findCategoryParents($catId, $categoryClass, $rootFolder->getFullPath()));
			}
		}

		$autoExpandCategories = array_unique($autoExpandCategories);

		foreach ($children as $child) {
			if ($child instanceof Folder || ($child instanceof Concrete && $child->getClassId() == $config['categoryClassId'])) {
				$icon = 'task-folder';
				$name = $child->getKey();

				if ($child instanceof Concrete && method_exists($child, $nameGetter)) {
					try {
						$name = $child->$nameGetter();
					} catch (\Exception $e) {
						$this->logger->error('Object "' . $child . '" has no method "' . $nameGetter . '()');
					}

					$icon = 'pimcore_icon_object';
				}

				if (empty($name)) {
					$name = $child->getKey();
				}


				$childData = array(
					'id' => $child->getId(),
					'name' => $name,
					'iconCls' => $icon,
				);

				if ($child->hasChilds()) {
					$childData['expanded'] = ($node === '_root' || in_array($child->getId(), $autoExpandCategories));
				} else {
					$childData['leaf'] = true;
					$childData['iconCls'] = 'task';
				}

				$data[] = $childData;
			}
		}


		return new JsonResponse($data);
	}

    /**
     * @Route("/get-rawdata-attributes")
     */
	public function getRawdataAttributesAction(Request $request) {
		$dataportId = $request->get('dataportId');
		$file = Helper::getFileForDataport($dataportId);
		$node = $request->get('node');
		$isProduct = $request->get('isProduct') === 'true';

		$response = array('success' => false, 'currentCategory' => $node, 'data' => array());

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		if (empty($node) || !$dataport) {
			return new JsonResponse(array());
		}

		$version = null;
		if ($dataport && !empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);
			if (array_key_exists('version', $config)) {
				$version = $config['version'];
			}
		}

		$xpathExpr = Helper::getBmecatXpathForVersion($version);
		if (empty($xpathExpr)) {
			$this->logger->warning('Unsupported BMEcat version "' . $version . '"');
			return new JsonResponse(array());
		}

		$hash = md5_file($file);
		$bmeFileRepository = new BmeFile();

		$existing = $bmeFileRepository->countRows([
		        'hash = ?' => $hash
        ]);
		if ($existing <= 0) {
			return new JsonResponse(array());
		}



		// Vereinigungsmenge aller Attribute. Gloable Attribute sind vordefiniert. Preise werden gesondert behandelt.
		$attributes = array();
		$globalAttributes = $xpathExpr['globalAttributes'];


		$itemIds = array();
		if ($isProduct) {
			$itemIds[] = $node;
		} else {
		    $bmeProductCategory = new BmeProductCategory();
			if ($node === '_dummy') {
				$items = $bmeProductCategory->find([
				        'file = ?' => $hash
                ]);
			} else {
				// Kindkategorien rekursiv auflösen
				$categoryIds = $this->findSubCategoriesCache($hash, array($node));
				$categoryIds[] = $node;
				$items = $bmeProductCategory->find([
				        'file = ?' => $hash,
                        'category_id IN (?)' => $categoryIds
                ]);
			}

			foreach ($items as $idNode) {
				$itemIds[] = $idNode['productId'];
			}

			$itemIds = array_unique($itemIds);
		}

		$features = array();
		$prices = array();
		$mimes = array();
		$references = array();

		$maxDataExamples = 5;

		if (!empty($itemIds)) {
		    $bmeProductRepository = new BmeProduct();
			$allItems = $bmeProductRepository->find([
                'file = ?' => $hash,
                'product_id IN (?)' => $itemIds
            ]);

			foreach ($allItems as $item) {
				try {
					$xml = @new \SimpleXMLElement($item['xml']);

					$itemFeatureNodes = $xml->xpath($xpathExpr['product']['features']);

					// Features auslesen
					/**
					 * @var $itemFeatureNode \SimpleXMLElement
					 */
					foreach ($itemFeatureNodes as $itemFeatureNode) {
						$featureValueNodes = $itemFeatureNode->xpath($xpathExpr['product']['featureValues']);

						$featureName = Helper::getSimpleXMLValue($itemFeatureNode, $xpathExpr['product']['featureName']);
						if (!array_key_exists($featureName, $features)) {
							$features[$featureName] = array();
						}

						if (count($features[$featureName]) < $maxDataExamples) {
							$featureValues = array();

							// Values auslesen
							foreach ($featureValueNodes as $featureValueNode) {
								$featureValues[] = (string)$featureValueNode;
							}

							if (count($featureValues) == 1) {
								$features[$featureName][] = $featureValues[0];
							} else if (count($featureValues) > 1) {
								$features[$featureName][] = '[' . join(';', $featureValues) . ']';
							} else {
								$features[$featureName][] = '';
							}
						}
					}

					// Beispielwerte für globale Attribute bestimmen
					foreach ($globalAttributes as $key => $globalAttr) {
						if (count($globalAttributes[$key]['values']) >= $maxDataExamples) {
							continue;
						}

						$globalAttributes[$key]['values'][] = Helper::getSimpleXMLValue($xml, $globalAttr['id']);
					}


					// Mimes
					$mimeNodes = $xml->xpath($xpathExpr['product']['mimeBase']);

					/**
					 * @var $mimeNode \SimpleXMLElement
					 */
					foreach ($mimeNodes as $mimeNode) {
						$mimeSource = Helper::getSimpleXMLValue($mimeNode, $xpathExpr['product']['mimeSource']);
						$key = Helper::getAssetType($mimeSource);

						if (!array_key_exists($key, $mimes)) {
							$mimes[$key] = array(
								'id' => 'mime:' . $key,
								'name' => 'Asset "' . $key . '"',
								'values' => array(),
							);
						}

						if (count($mimes[$key]['values']) < $maxDataExamples && Helper::getAssetType($mimeSource) == $key) {
							$mimes[$key]['values'][] = $mimeSource;
						}
					}



					// Preise
					$priceNodes = $xml->xpath($xpathExpr['product']['prices']);
					$itemPrices = array();

					/**
					 * @var $priceNode \SimpleXMLElement
					 */
					foreach ($priceNodes as $priceNode) {
						$priceData = $this->getPriceData($priceNode, $xpathExpr);
						if ($priceData['type'] != null && $priceData['amount'] != null) {
							$type = $priceData['type'];

							if (!array_key_exists($type, $itemPrices)) {
								$itemPrices[$type] = array();
							}

							unset($priceData['type']);
							$itemPrices[$type][] = $priceData;
						}
					}

					foreach ($itemPrices as $type => $itemPriceData) {
						$p = array();

						foreach ($itemPriceData as $price) {
							$p[] = $price['amount'];
						}

						if (!array_key_exists($type, $prices)) {
							$prices[$type] = array();
						}

						if (count($prices[$type]) < $maxDataExamples) {
							$prices[$type][] = '[' . join(',', $p) . ']';
						}
					}


					// Referenzen
					$referenceNodes = $xml->xpath(implode('/', array($xpathExpr['product']['reference'], $xpathExpr['product']['referenceType'])));
					foreach ($referenceNodes as $referenceNode) {
						if ($referenceNode instanceof \SimpleXMLElement) {
							$type = (string)$referenceNode;

							if (!empty($type) && !in_array($type, $references)) {
								$references[] = $type;
							}
						}
					}
				} catch (\Exception $e) {
					$this->logger->error('Unable to parse product xml for product "' . $item['productId'] . '" in "' . $hash . '"');
				}
			}
		}

		foreach ($globalAttributes as $globalAttr) {
			$attributes[] = array(
				'id' => $globalAttr['id'],
				'name' => $globalAttr['name'],
				'values' => join(';', $globalAttr['values']),
				'groupKey' => 'global',
			);
		}

		foreach ($mimes as $mime) {
			$attributes[] = array(
				'id' => $mime['id'],
				'name' => $mime['name'],
				'values' => implode(';', $mime['values']),
				'groupKey' => 'media',
			);
		}

		foreach ($features as $name => $feature) {
			$attributes[] = array(
				'id' => sprintf($xpathExpr['product']['featureByName'], str_replace('"', '\\"', $name)),
				'name' => $name,
				'values' => implode(';', $feature),
				'groupKey' => 'specific',
			);
		}

		foreach ($prices as $type => $priceData) {
			$attributes[] = array(
				'id' => 'price:' . $type,
				'name' => 'Preis ("' . $type . '")',
				'values' => implode(';', $priceData),
				'groupKey' => 'global',
			);
		}

		// Referenzen
		foreach ($references as $ref) {
			$attributes[] = array(
				'id' => 'reference:' . $ref,
				'name' => 'Artikelrelation ("' . $ref . '")',
				'values' => '',
				'groupKey' => 'references',
			);
		}

        $data = array();
		foreach ($attributes as $attr) {
			$data[] = array(
				'id' => $attr['id'],
				'name' => $attr['name'],
				'values' => $attr['values'],
				'groupKey' => $attr['groupKey'],
			);
		}

		$response['success'] = true;
		$response['data'] = $data;
		return new JsonResponse($response);
	}

    /**
     * @Route("/get-pim-attributes")
     */
	public function getPimAttributesAction(Request $request) {
		$categoryId = $request->get('node');
		$dataportId = $request->get('dataportId');
		$sourceCategory = $request->get('sourceCategory');
		$file = Helper::getFileForDataport($dataportId);
		$hash = md5_file($file);

		$attributes = $this->_getPimAttributes($dataportId, $sourceCategory, $categoryId, $hash);

		return new JsonResponse(array(
			'success' => true,
			'isMapped' => $attributes['isMapped'],
			'currentCategory' => $attributes['currentCategory'],
			'data' => $attributes['data'],
		));
	}

    /**
     * @Route("/get-attribute-mappings-for-source")
     */
	public function getAttributeMappingsForSourceAction(Request $request) {
		$dataportId = $request->get('dataportId');
		$xpath = $request->get('xpath');
		$file = Helper::getFileForDataport($dataportId);
		$hash = md5_file($file);

		$data = $this->findAttributeMappingsForId($dataportId, $hash, $xpath, 'xpath');

		if (is_array($data)) {
			return new JsonResponse(array(
				'success' => true,
				'data' => $data
			));
		}
		return new JsonResponse(array(
            'success' => false,
        ));
	}

    /**
     * @Route("/get-attribute-mappings-for-target")
     */
	public function getAttributeMappingsForTargetAction(Request $request) {
		$dataportId = $request->get('dataportId');
		$id = $request->get('field');
		$file = Helper::getFileForDataport($dataportId);
		$hash = md5_file($file);

		$data = $this->findAttributeMappingsForId($dataportId, $hash, $id, 'key');

		if (is_array($data)) {
			return new JsonResponse(array(
				'success' => true,
				'data' => $data
			));
		}
		return new JsonResponse(array(
            'success' => false,
        ));
	}


	private function findAttributeMappingsForId($dataportId, $hash, $id, $idField) {
		if (empty($dataportId) || empty($hash) || empty($id) || empty($idField)) {
			return null;
		}
		if ($idField !== 'xpath' && $idField !== 'key') {
			return null;
		}

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		$config = array();
		$categoryClassId = null;
		$version = '1.2';
		if ($dataport && !empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);
			if (array_key_exists('version', $config)) {
				$version = $config['version'];
			}
			if (array_key_exists('categoryClassId', $config)) {
				$categoryClassId = $config['categoryClassId'];
			}
		}

		$targetconfig = unserialize($dataport['targetconfig']);
		$targetClass = ClassDefinition::getById($targetconfig['itemClass']);

		if (!($targetClass instanceof ClassDefinition)) {
			return null;
		}

		$xpathExpr = Helper::getBmecatXpathForVersion($version);
		if (empty($xpathExpr)) {
			$this->logger->warning('Unsupported BMEcat version "' . $version . '"');
			return null;
		}

		$mappingModel = new Categorymapping();
		$categoryMappings = $mappingModel->find(array(
			'dataportId = ?' => $dataportId,
		));

		$categoryNameCache = array();
		$categoryNameGetter = empty($config['categoryNameElement']) ? null : ('get' . ucfirst($config['categoryNameElement']));
		$fieldNameCache = array();
		$fieldCache = array();
		$sourceFieldNameCache = array();

		$sourceCategoryNameCache = array();

		$mappings = array();
		foreach ($categoryMappings as $mapping) {

			if (!array_key_exists($mapping['sourceCategoryId'], $sourceCategoryNameCache)) {
				if ($mapping['sourceCategoryId'] === '_dummy') {
					$sourceCategoryNameCache[$mapping['sourceCategoryId']] = 'Dummykategorie (BMEcat enthält keinen Kategoriebaum)';
				} else {
				    $bmeCategoryRepository = new BmeCategory();
					$category = $bmeCategoryRepository->findOne([
                        'file = ?' => $hash,
                        'categoryId = ?' => $mapping['sourceCategoryId']
                    ]);

					if (!empty($category)) {
						$sourceCategoryNameCache[$mapping['sourceCategoryId']] = $category['name'];
					} else {
						$sourceCategoryNameCache[$mapping['sourceCategoryId']] = '*Not found*';
					}
				}
			}

			$categoryMappingFieldRepository = new CategorymappingField();
			foreach ($categoryMappingFieldRepository->find([
                'id = ?' => $mapping['id']
            ]) as $field) {
				$isMapped = false;
				if ($idField === 'xpath') {
					$isMapped = $field['xpath'] == $id;
				} else {
					$fieldParts = explode('.', $id);

					if (count($fieldParts) === 2) {
						$fcName = $fieldParts[0];
						$fieldData = $fieldParts[1];

						$localeParts = explode('#', $fieldData);
						if (count($localeParts) === 2) {
							$isMapped = $field['fieldCollection'] == $fcName && $localeParts[0] == $field['field'] && $field['locale'] == $localeParts[1];
						} else {
							$isMapped = $field['fieldCollection'] == $fcName && $localeParts[0] == $field['field'] && empty($field['locale']);
						}
					}
				}

				if ($isMapped) {
					if (!array_key_exists($field['xpath'], $sourceFieldNameCache)) {
						// Namen raussuchen, wenn es ein globales Attribute ist. Ansonsten versuchen, den Featurenamen
						// auszulesen. Xpath-Ausdruck ist Fallback
						$name = null;
						foreach ($xpathExpr['globalAttributes'] as $globalAttr) {
							if ($globalAttr['id'] == $field['xpath']) {
								$name = $globalAttr['name'];
								break;
							}
						}

						if (empty($name)) {
							if (preg_match('/FNAME="(.+?)"/', $field['xpath'], $matches)) {
								if (count($matches) >= 2) {
									$name = $matches[1];
								}
							}
						}

						if (empty($name)) {
							$name = $field['xpath'];
						}

						$sourceFieldNameCache[$field['xpath']] = $name;
					}

					if (!array_key_exists($mapping['targetCategoryId'], $categoryNameCache)) {
						$category = Concrete::getById($mapping['targetCategoryId']);

						if ($category instanceof Concrete && $category->getClassId() == $categoryClassId) {
							$name = $category->getKey();

							if (method_exists($category, $categoryNameGetter)) {
								try {
									$name = $category->$categoryNameGetter();
								} catch (\Exception $e) {
									$this->logger->error('Object "' . $category . '" has no method "' . $categoryNameGetter . '()');
								}
							}

							if (empty($name)) {
								$name = $category->getKey();
							}

							$categoryNameCache[$mapping['targetCategoryId']] = $name;
						} else {
							$categoryNameCache[$mapping['targetCategoryId']] = '*Invalid Category*';
						}
					}


					$fieldNameKey = $field['fieldCollection'] . '.' . $field['field'];
					if (!empty($field['locale'])) {
						$fieldNameKey .= '#' . $field->locale;
					}

					if (!array_key_exists($field['fieldCollection'], $fieldCache)) {
						$fieldCache[$field['fieldCollection']] = array();

						if ($field['fieldCollection'] === 'class') {
							$classFields = Helper::getClassFields($targetClass);
							foreach ($classFields as $classField) {
								$fieldCache[$field['fieldCollection']][] = $classField;
							}
						} else {
							$fcFields = Helper::getFieldcollectionFields(Definition::getByKey($field['fieldCollection']));
							foreach ($fcFields as $fcField) {
								$fieldCache[$field['fieldCollection']][] = $fcField;
							}
						}
					}

					if (!array_key_exists($fieldNameKey, $fieldNameCache)) {
						$name = null;
						$key = $field['field'];
						if (!empty($field['locale'])) {
							$key .= '#' . $field['locale'];
						}
						foreach ($fieldCache[$field['fieldCollection']] as $fieldDef) {
							if ($fieldDef['key'] == $key) {
								$name = $fieldDef['name'];
							}
						}

						if (is_null($name)) {
							$name = '*Invalid field*';
						}
						$fieldNameCache[$fieldNameKey] = $name;
					}

					$mappingHash = array($field['id'], $field['fieldCollection'], $field['field'], $field['locale']);

					$r = array(
						'hash' => implode('#', $mappingHash),
						'id' => $field['id'],
						'fieldCollection' => $field['fieldCollection'],
						'field' => $field['field'],
						'locale' => $field['locale'],
						'targetCategoryId' => $mapping['targetCategoryId'],
					);

					$r['groupId'] = $mapping['sourceCategoryId'];
					$r['groupName'] = $sourceCategoryNameCache[$mapping['sourceCategoryId']];
					$r['targetCategoryName'] = $categoryNameCache[$mapping['targetCategoryId']] . ' (' . $mapping['targetCategoryId'] . ')';

					if ($idField === 'xpath') {
						$r['targetFieldName'] = $fieldNameCache[$fieldNameKey];
					} else {
						$r['targetFieldName'] = $sourceFieldNameCache[$field['xpath']];
					}

					$mappings[] = $r;
				}
			}
		}

		return $mappings;
	}

    /**
     * @Route("/delete-field-mapping")
     */
	public function deleteFieldMappingAction(Request $request) {
		$id = $request->get('id');
		$fieldCollection = $request->get('fieldCollection');
		$field = $request->get('field');
		$locale = $request->get('locale');

		$model = new CategorymappingField();
		$model->deleteWhere(['id' => $id, 'fieldCollection' => $fieldCollection, 'field' => $field, 'locale' => $locale]);

		return new JsonResponse(array(
			'success' => true,
			'id' => $id,
			'fieldCollection' => $fieldCollection,
			'field' => $field,
			'locale' => $locale
		));
	}

	private function _getPimAttributes($dataportId, $sourceCategoryId, $targetCategoryId, $hash) {
		$result = array('isMapped' => false, 'data' => array());

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		$category = Concrete::getById($targetCategoryId);

		if ($dataport && !empty($dataport['sourceconfig'])) {
			$mappingModel = new Categorymapping();

			$existing = 0;
			if (!empty($sourceCategoryId)) {
				$existing = $mappingModel->countRows(array(
					'dataportId = ?' => $dataport['id'],
					'sourceCategoryId = ?' => $sourceCategoryId,
					'targetCategoryId = ?' => $targetCategoryId,
				));
			}

			$config = unserialize($dataport['sourceconfig']);
			$targetconfig = unserialize($dataport['targetconfig']);


			if (is_array($config) && array_key_exists('categoryClassId', $config) &&
				$category instanceof Concrete && $category->getClassId() == $config['categoryClassId'] &&
				is_array($targetconfig) && array_key_exists('itemClass', $targetconfig)) {

				$targetClass = ClassDefinition::getById($targetconfig['itemClass']);

				if ($targetClass instanceof ClassDefinition) {
					$classFields = Helper::getClassFields($targetClass);
					$fieldcollectionFields = Helper::getRecursiveFieldcollectionFields($targetCategoryId, $category->getClass(), $config['categoryfolder'], $config['categoryAttributeElement']);

					if ($existing > 0) {
						$mappings = Helper::findMappings($dataport['id'], $sourceCategoryId, $hash);

						if (array_key_exists($category->getId(), $mappings)) {
							$classMappings = $mappings[$category->getId()]['class'];

							foreach ($classFields as $field) {
								$type = $field['type'];

								if ($type === 'objectbricks' || $type === 'fieldcollections') {
									continue;
								}

								$xpath = null;
								$format = '';
								$calculation = '';
								$inherited = false;
								$keyMapping = false;
								$isMapped = false;

								if (is_array($classMappings) && array_key_exists($field['key'], $classMappings)) {
									$xpath = $classMappings[$field['key']]['xpath'];
									$inherited = $classMappings[$field['key']]['inherited'];
									$calculation = $classMappings[$field['key']]['calculation'];
									$keyMapping = $classMappings[$field['key']]['keyMapping'] == '1';
									$format = $classMappings[$field['key']]['format'];
									$isMapped = true;
								}

								$result['data'][] = array(
									'id' => 'class.' . $field['key'],
									'key' => $field['key'],
									'name' => $field['name'],
									'dataType' => $type,
									'groupKey' => 'class',
									'groupName' => 'global',
									'groupPosition' => 0,
									'mapped' => $isMapped,
									'xpath' => $xpath,
									'inherited' => $inherited === true,
									'calculation' => $calculation,
									'format' => $format,
									'keyMapping' => $keyMapping,
									'tooltip' => $field['tooltip'],
								);
							}


							// Fieldcollections
							foreach ($fieldcollectionFields as $fc => $fields) {
								foreach ($fields as $field) {
									$type = $field['type'];

									$xpath = null;
									$format = null;
									$calculation = '';
									$inherited = false;
									$isMapped = false;

									$fcMappings = array_key_exists($fc, $mappings[$category->getId()]) ? $mappings[$category->getId()][$fc] : array();

									if (is_array($fcMappings) && array_key_exists($field['key'], $fcMappings)) {
										$xpath = $fcMappings[$field['key']]['xpath'];
										$inherited = $fcMappings[$field['key']]['inherited'];
										$calculation = $fcMappings[$field['key']]['calculation'];
										$format = $fcMappings[$field['key']]['format'];
										$isMapped = true;
									}

									$result['data'][] = array(
										'id' => $fc . '.' . $field['key'],
										'key' => $field['key'],
										'name' => $field['name'],
										'dataType' => $type,
										'groupKey' => $fc,
										'groupName' => $fc,
										'groupPosition' => 0,
										'mapped' => $isMapped,
										'xpath' => $xpath,
										'inherited' => $inherited === true,
										'calculation' => $calculation,
										'format' => $format,
										'keyMapping' => false,  // Fieldcollections must not contains keys
										'tooltip' => $field['tooltip'],
									);
								}
							}

							$result['isMapped'] = true;
						}

						$result['currentCategory'] = $category->getId();
					} else {
						foreach ($classFields as $field) {
							$result['data'][] = array(
								'id' => 'class.' . $field['key'],
								'key' => $field['key'],
								'name' => $field['name'],
								'dataType' => $field['type'],
								'groupKey' => 'class',
								'groupName' => 'global',
								'groupPosition' => 0,
								'mapped' => false,
								'xpath' => null,
								'inherited' => false,
								'calculation' => '',
								'format' => null,
								'keyMapping' => false,
								'tooltip' => $field['tooltip'],
							);
						}

						foreach ($fieldcollectionFields as $fc => $fields) {
							foreach ($fields as $field) {
								$result['data'][] = array(
									'id' => $fc . '.' . $field['key'],
									'key' => $field['key'],
									'name' => $field['name'],
									'dataType' => $field['type'],
									'groupKey' => $fc,
									'groupName' => $fc,
									'groupPosition' => 0,
									'mapped' => false,
									'xpath' => null,
									'inherited' => false,
									'calculation' => '',
									'format' => null,
									'keyMapping' => false,  // Fieldcollections must not contains keys
									'tooltip' => $field['tooltip'],
								);
							}
						}
					}
				}

			}
		}

		return $result;
	}

    /**
     * @Route("/save-category-mapping")
     */
	public function saveCategoryMappingAction(Request $request){
		$result = array('success' => false);

		$dataportId = $request->get('dataportId');
		$sourceId = $request->get('sourceId');
		$targetId = $request->get('targetId');

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		$targetCategory = Concrete::getById($targetId);

		if ($dataport && !empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);

			if ($sourceId && is_array($config) && array_key_exists('categoryClassId', $config) &&
				$targetCategory instanceof Concrete && $targetCategory->getClassId() == $config['categoryClassId']) {

				$mappingModel = new Categorymapping();
				$existing = $mappingModel->countRows(array(
					'dataportId = ?' => $dataport['id'],
					'sourceCategoryId = ?' => $sourceId,
					'targetCategoryId = ?' => $targetCategory->getId(),
				));

				if ($existing === 0) {
					$mapping = $mappingModel->create(array(
						'dataportId' => $dataport['id'],
						'sourceCategoryId' => $sourceId,
						'targetCategoryId' => $targetCategory->getId(),
					));

					if (is_array($mapping) && $mapping['id']) {
						$result['success'] = true;
					}
				} else {
					$result['success'] = true;
				}
			}
		}

		return new JsonResponse($result);
	}

    /**
     * @Route("/save-attribute-mapping")
     */
	public function saveAttributeMappingAction(Request $request) {
		$result = array('success' => false);

		$mappings = json_decode($request->get('data'));

		$categoryId = $request->get('node');
		$dataportId = $request->get('dataportId');
		$sourceCategory = $request->get('sourceCategory');
		$file = Helper::getFileForDataport($dataportId);
		$hash = md5_file($file);

		$table = new Dataport();
		$dataport = $table->get($dataportId);

		if ($dataport && $mappings) {
			if (!is_array($mappings)) {
				$mappings = array($mappings);
			}

			foreach ($mappings as $mapping) {
				$this->_saveAttributeMapping($dataportId, $sourceCategory, $categoryId, $mapping->id, $mapping->xpath, $mapping->format, $mapping->dataType, $mapping->keyMapping, $mapping->calculation);
			}

			$attributes = $this->_getPimAttributes($dataportId, $sourceCategory, $categoryId, $hash);
			$result = array(
				'success' => true,
				'isMapped' => $attributes['isMapped'],
				'currentCategory' => $attributes['currentCategory'],
				'data' => $attributes['data'],
			);
		}

		return new JsonResponse($result);
	}

    /**
     * @Route("/get-mappings-for-source")
     */
	public function getMappingsForSourceAction(Request $request) {
		$node = $request->get('node');
		$dataportId = $request->get('dataportId');

		$data = $this->findMappingForSource($dataportId, $node);

		if ($data === null) {
            return new JsonResponse(array('success' => false));
		}

		return new JsonResponse(array('success' => true, 'data' => $data));
	}

    /**
     * @Route("/delete-mappings-by-id")
     */
	public function deleteMappingsByIdAction(Request $request) {
		$mappingIdString = $request->get('data');
		$dataportId = $request->get('dataportId');

		if (!$mappingIdString || !$dataportId) {
			return new JsonResponse(array('success' => false));
		}

		$mappingIds = json_decode($mappingIdString);

		if (!is_array($mappingIds)) {
			$mappingIds = (array)$mappingIds;
		}

		$mappingModel = new Categorymapping();

		foreach ($mappingIds as $mappingId) {
			$mappingModel->delete($mappingId);
		}

		return new JsonResponse(array('success' => true));
	}

    /**
     * @Route("/get-mappings-for-target")
     */
	public function getMappingsForTargetAction(Request $request) {
		$node = $request->get('node');
		$dataportId = $request->get('dataportId');
		$file = Helper::getFileForDataport($dataportId);
		$hash = md5_file($file);

		$data = $this->findMappingForTarget($dataportId, $node, $hash);

		if (is_null($data)) {
			return new JsonResponse(array('success' => false));
		}

		return new JsonResponse(array('success' => true, 'data' => $data));
	}

    /**
     * @Route("/get-example-data")
     */
	public function getExampleDataAction(Request $request) {
		$dataportId = $request->get('dataportId');
		$file = Helper::getFileForDataport($dataportId);
		$xpath = $request->get('xpath');
		$node = $request->get('node');
		$hash = md5_file($file);

		$data = $this->findExampleData($dataportId, $xpath, $node, $hash);

		if (!is_array($data)) {
			return new JsonResponse(array('success' => false));
		}
		return new JsonResponse(array('success' => true, 'data' => $data));
	}

    /**
     * @Route("/get-preview")
     */
	public function getPreviewAction(Request $request) {
		$data = array();

		$dataportId = $request->get('dataportId');
		$file = Helper::getFileForDataport($dataportId);
		$hash = md5_file($file);

		$targetField = $request->get('targetField');
		$xpath = $request->get('xpath');
		$sourceCategoryId = $request->get('sourceCategoryId');
		$targetCategoryId = $request->get('targetCategoryId');
		$calculation = $request->get('calculation');

		$offset = (int)$request->get('start');
		$limit = (int)$request->get('limit');

		if (!$offset) {
			$offset = 0;
		}

		if (!$limit) {
			$limit = 5;
		}

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		$version = '1.2';
		if ($dataport && !empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);
			if (array_key_exists('version', $config)) {
				$version = $config['version'];
			}
		}

		$xpathExpr = Helper::getBmecatXpathForVersion($version);
		if (empty($xpathExpr)) {
            return new JsonResponse(array('success' => false));
		}

		$mappings = Helper::findMappings($dataport['id'], $sourceCategoryId, $hash, true);

		$total = 0;
		if (array_key_exists($targetCategoryId, $mappings)) {
			$preJs = file_get_contents(implode(DIRECTORY_SEPARATOR, array(dirname(__FILE__), '..', 'lib', 'Pim', 'Item', 'libraries.js')));

			$mapping = $mappings[$targetCategoryId];

			// Fieldidentifier splitten
			$fieldId = explode('.', $targetField);

			if (count($fieldId) === 2 && array_key_exists($fieldId[0], $mapping)) {
				$classMapping = $mapping[$fieldId[0]];

				$hasMapping = array_key_exists($fieldId[1], $classMapping);
				if ($hasMapping || !empty($calculation)) {

					if ($hasMapping && empty($calculation)) {
						$fieldMapping = $classMapping[$fieldId[1]];
						$calculation = $fieldMapping["calculation"];
					}

					$db = Db::get();
					$select = $db->select()
						->from(array('p' => Installer::TABLE_BME_PRODUCT), 'p.xml')
						->join(array('p2c' => Installer::TABLE_BME_PRODUCT_CATEGORY), 'p2c.file = p.file AND p2c.productId = p.productId', null)
						->columns('p.productId')
						->where('p2c.categoryId = ?', array($sourceCategoryId))
						->where('p2c.file = ?', array($hash))
						->order('p.productId ASC')
						->limit($limit, $offset);
					$selectCount = $db->select()
						->from(array('p' => Installer::TABLE_BME_PRODUCT), 'count(*)')
						->join(array('p2c' => Installer::TABLE_BME_PRODUCT_CATEGORY), 'p2c.file = p.file AND p2c.productId = p.productId', null)
						->where('p2c.categoryId = ?', array($sourceCategoryId))
						->where('p2c.file = ?', array($hash));

					$items = $db->fetchAll($select);
					$total = (int)$db->fetchOne($selectCount);

					foreach ($items as $item) {
						$value = null;
						$valueArray = array();

						try {
							$itemNode = @new \SimpleXMLElement($item['xml']);

							if ($hasMapping) {
								$value = $this->calculateFieldValue($itemNode, $xpath, $xpathExpr);
							}

							if (!empty($calculation)) {
								$sourceProduct = dom_import_simplexml($itemNode);
								$articleData = Helper::nodeToArray($sourceProduct);
								$value = Importer::processJS($calculation, $value, $valueArray, $preJs, $articleData, '', array());
							}

							$data[] = array(
								'id' => $item['productId'],
								'preview' => $value,
							);
						} catch (\Exception $e) {
							$this->logger->error('Unable to parse xml tor product "' . $item['productId'] . '" in "' . $hash . '"');
						}
					}


				}
			}
		}

		if (!is_array($data)) {
			return new JsonResponse(array('success' => false));
		}
		return new JsonResponse(array('success' => true, 'total' => $total, 'data' => $data));
	}



	public function getDebugAction(Request $request){
		Cache::clearAll();

		$itemAid = $request->get('itemId');
		$dataportId = $request->get('dataportId');
		$targetField = $request->get('targetField');
		$xpath = $request->get('xpath');
		$sourceCategoryId = $request->get('sourceCategoryId');
		$targetCategoryId = $request->get('targetCategoryId');
		$calculation = $request->get('calculation');



		$table = new Dataport();
		$dataport = $table->get($dataportId);
		$targetconfig = unserialize($dataport['targetconfig']);
		$sourceconfig = unserialize($dataport['sourceconfig']);

		$file = Helper::getFileForDataport($dataportId);
		$hash = md5_file($file);

		$version = '1.2';
		if ($dataport && !empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);
			if (array_key_exists('version', $config)) {
				$version = $config['version'];
			}
		}

		$xpathExpr = Helper::getBmecatXpathForVersion($version);
		if (empty($xpathExpr)) {
			echo 'Version ist ungültig';
			return;
		}

		$mappings = Helper::findMappings($dataport['id'], $sourceCategoryId, $hash, true);

		// Fieldidentifier splitten
		$fieldId = explode('.', $targetField);
		$collectionName = $fieldId[0];
		$fieldName = $fieldId[1];
        $language = '';

		// find language and remove language from $fieldName
		if (strpos($fieldName, "#")) {
			$fieldName = substr($fieldName, 0, strpos($fieldName, "#"));
		}


		// Get Produkt vom DB and from XML
		// get value and valueArray
		// execute JS-Script
		$messages = "";
		if (array_key_exists($targetCategoryId, $mappings)) {
			$preJs = file_get_contents(implode(DIRECTORY_SEPARATOR, array(dirname(__FILE__), '..', 'lib', 'Pim', 'Item', 'libraries.js')));

			$mapping = $mappings[$targetCategoryId];


			if (count($fieldId) === 2 && array_key_exists($fieldId[0], $mapping)) {
				$classMapping = $mapping[$fieldId[0]];

				$hasMapping = array_key_exists($fieldId[1], $classMapping);
				if ($hasMapping || !empty($calculation)) {

					if ($hasMapping && empty($calculation)) {
						$fieldMapping = $classMapping[$fieldId[1]];
						$calculation = $fieldMapping["calculation"];
					}

					$db = Db::get();
					$items = array();
					if (!empty($itemAid)){
						$select = $db->select()
						->from(array('p' => Installer::TABLE_BME_PRODUCT), 'p.xml')
						->join(array('p2c' => Installer::TABLE_BME_PRODUCT_CATEGORY), 'p2c.file = p.file AND p2c.productId = p.productId', null)
						->columns('p.productId')
						->where('p2c.categoryId = ?', array($sourceCategoryId))
						->where('p2c.file = ?', array($hash))
						->where('p.productId = ?', array($itemAid))
						->order('p.productId ASC');

						$items = $db->fetchAll($select);
					}

					if(count($items) === 0){
						if (!empty($itemAid)){
							$messages .= '<div style="color:red;">Zu eingegeben Id wurde kein Artikel gefunden.</div><br>';
						}

						$select = $db->select()
						->from(array('p' => Installer::TABLE_BME_PRODUCT), 'p.xml')
						->join(array('p2c' => Installer::TABLE_BME_PRODUCT_CATEGORY), 'p2c.file = p.file AND p2c.productId = p.productId', null)
						->columns('p.productId')
						->where('p2c.categoryId = ?', array($sourceCategoryId))
						->where('p2c.file = ?', array($hash))
						->order('p.productId ASC');

						$items = $db->fetchAll($select);

					}
					if(count($items) == 0){
						$messages .= '<h1  style="color:red;">Es wurde kein gültiger Artikel für diese Zuordnung gefunden.</h1>';
					}
					else{
						$item = $items[0];
                        $messages .= '<h1> Debug info für '.$item['productId'].'</h1>';
					}


					$value = null;


					$dom = Helper::getDomDocument($file);
					$xp = new \DOMXPath($dom);

					$worker = new Bmecat(true);
					$worker->setDataportId($dataportId);
					$worker->setFile($file);
					$worker->setImportKey(123);

					if (array_key_exists('idPrefix', $targetconfig) && !empty($targetconfig['idPrefix'])) {
						$worker->setIdPrefix($targetconfig['idPrefix']);
					}
					$sourceProduct = $xp->query(vsprintf($xpathExpr['product']['byId'], array($item['productId'])))->item(0);
					$BMEData = $worker->collectDataFromBMECAT($xp, $xpathExpr, $sourceProduct);
					$valueArray =  $BMEData["fieldValues"][$collectionName][$fieldName]['valueArray'];



					try {
						$itemNode = @new \SimpleXMLElement($item['xml']);
						if ($hasMapping) {
							$value = $this->calculateFieldValue($itemNode, $xpath, $xpathExpr);
						}

						$sourceProduct = dom_import_simplexml($itemNode);
						$articleData = Helper::nodeToArray($sourceProduct);


						if (!empty($calculation)) {
							$result = Importer::processJS($calculation, $value, $valueArray, $preJs, $articleData, '', array());
						}

					} catch (\Exception $e) {
						$this->logger->error('Unable to parse xml tor product "' . $item['productId'] . '" in "' . $hash . '"');
					}
				}
			}
		}










		// Load Object
		$itemClassId = $targetconfig['itemClass'];
		$itemClass = ClassDefinition::getById($itemClassId);
		$itemClassName = $itemClass->getName();
		$item = $worker->getItem($itemClassName, null, null, $BMEData['keyValues'], null);

		// Get current Values
		$currentValue = '';
		$currentValues = '';
		if ($item instanceof AbstractObject){

			if ($collectionName === "class") {
				$elementToGetValueFrom = $item;
			} else {

				$itemAttributeElement = $sourceconfig['itemAttributeElement'];

				$getter = 'get' . ucfirst($itemAttributeElement);
				$realCollectionName = "Object_Fieldcollection_Data_" . ucfirst($collectionName);


				// if collection element exists check if specific collection exists, if not -> create it
				// if no collection element exists -> create it and the collection
				$itemCollectionsElement = $item->$getter();
				if ($itemCollectionsElement != null) {
					$currentCollections = $itemCollectionsElement->getItems();
					foreach ($currentCollections as $collection) {
						if ($collection instanceof $realCollectionName) {
							$elementToGetValueFrom = $collection;
						}
					}
					if ($elementToGetValueFrom == null) {
						$currentValue = "Field Collection nicht gefunden 1.";
					}
				} else {
					$currentValue = "Field Collection nicht gefunden 2 .";
				}
			}

			$getter = 'get' . ucfirst($fieldName);
			try {
				if($elementToGetValueFrom != null ){
					$currentValue = $elementToGetValueFrom->$getter($language);
				}
			} catch (\Exception $e) {
				$currentValue = 'Keine aktuellen Daten gefunden.';
			}

			$currentValues = $worker->getAttributesArrayForObject($item);

		}
		else {
			$currentValue = "Kein Object gefunden";
		}


		// Print out the Data
		?>

		<style>
		.debug-data th {
			font-weight: bold;
			padding: 5px;
			border: 1px solid #000;
			width:150px;
		}
		.debug-data td {
			padding: 5px;
			border: 1px solid #000;
			max-width:800px;
		}
		.scrolldiv {
			height: 300px;
			 overflow : auto;
		}
		</style>

		<?php
		echo '<div style="margin:5px;">';
			echo "<br>$messages<br><br>";

			//echo "$dataportId <bR>$targetField <br> $xpath <br>$sourceCategoryId <br>$targetCategoryId<br><hr>$calculation<hr>";

			echo '<table class="debug-data">';
				echo '<tr>';
					echo '<th>value</th>';
					echo '<td>'.$value.'</td>';
				echo '</tr>';
				echo '<tr>';
					echo '<th>valueArray</th>';
					echo '<td><pre>'.print_r($valueArray, true).'</pre></td>';
				echo '</tr>';
				echo '<tr>';
					echo '<th>articleData</th>';
					echo '<td><div class="scrolldiv"><pre>'.print_r($articleData, true).'</pre></div></td>';
				echo '</tr>';

				echo '<tr>';
					echo '<th>currentValue</th>';
					echo '<td><pre>'.print_r($currentValue, true).'</pre></td>';
				echo '</tr>';

				echo '<tr>';
					echo '<th>currentObjectData</th>';
					echo '<td><pre>'.print_r($currentValues, true).'</pre></td>';
				echo '</tr>';
				echo '<tr>';
					echo '<th>JS-Formel</th>';
					echo '<td><pre>'.print_r($calculation, true).'</pre></td>';
				echo '</tr>';
				echo '<tr>';
					echo '<th>Ergebnis</th>';
					echo '<td><pre>'.print_r($result, true).'</pre></td>';
				echo '</tr>';
			echo '</table>';
		echo '</div>';
	}


	private function findExampleData($dataportId, $xpath, $node, $hash) {
		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		$version = '1.2';
		if ($dataport && !empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);
			if (array_key_exists('version', $config)) {
				$version = $config['version'];
			}
		}

		$xpathExpr = Helper::getBmecatXpathForVersion($version);
		if (empty($xpathExpr)) {
			return null;
		}

		$data = array();

		$bmeProductCategoryRepository = new BmeProductCategory();
		if ($node === '_dummy') {
			$itemIdNodes = $bmeProductCategoryRepository->find([
                'file = ?' => $hash
            ]);
		} else {
			// Kindkategorien rekursiv auflösen
			$categoryIds = $this->findSubCategoriesCache($hash, array($node));
			$categoryIds[] = $node;

			$itemIdNodes = $bmeProductCategoryRepository->find([
                'file = ?' => $hash,
                'categoryId IN (?)' => $categoryIds
            ]);
		}


		$itemIds = array();
		foreach ($itemIdNodes as $idNode) {
			$itemIds[] = $idNode['productId'];
		}

		$itemIds = array_unique($itemIds);
		if (empty($itemIds)) {
			return array();
		}

		$bmeProductRepository = new BmeProduct();
		$items = $bmeProductRepository->find([
            'file = ?' => $hash,
            'productId IN (?)' => $itemIds
        ]);

		foreach ($items as $item) {
			$value = null;

			try {
				$itemNode = @new \SimpleXMLElement($item['xml']);
				if ($itemNode != null) {
					$value = $this->calculateFieldValue($itemNode, $xpath, $xpathExpr);
				}

				if (!empty($value)) {
					if (!array_key_exists($value, $data)) {
						$data[$value] = array();
					}

					$data[$value][] = $item['productId'];
				}
			} catch (\Exception $e) {
				$this->logger->error('Unable to parse xml tor product "' . $item['productId'] . '" in "' . $hash . '"');
			}
		}

		$result = array();

		foreach ($data as $value => $ids) {
			$result[] = array(
				'value' => $value,
				'count' => count($ids),
				'ids' => implode(';', $ids),
			);
		}


		return $result;
	}

	/**
	 * @param \SimpleXMLElement $itemNode
	 * @param string $xpath
	 * @param array $xpathExpr
	 * @return null|string
	 */
	private function calculateFieldValue($itemNode, $xpath, $xpathExpr) {
		$value = null;
		if (!($itemNode instanceof \SimpleXMLElement)) {
			return null;
		}

		if (strpos($xpath, 'price:') === 0) {
			$priceData = explode(':', $xpath);
			$priceType = $priceData[1];

			$priceNodes = $itemNode->xpath(sprintf($xpathExpr['product']['pricesByType'], $priceType));

			$value = '[';
			$prices = array();
			foreach ($priceNodes as $price) {
				$prices[] = Helper::getSimpleXMLValue($price, $xpathExpr['product']['priceAmount']);
			}

			$value .= implode(';', $prices);

			$value .= ']';
		} else if (strpos($xpath, 'mime:') === 0) {
			$mimeData = explode(':', $xpath);
			$mimeType = $mimeData[1];

			$mimeNodes = $itemNode->xpath($xpathExpr['product']['mimeBase']);

			$mimes = array();
			foreach ($mimeNodes as $mime) {
				$mimeValue = Helper::getSimpleXMLValue($mime, $xpathExpr['product']['mimeSource']);
				if (Helper::getAssetType($mimeValue) == $mimeType) {
					$mimes[] = $mimeValue;
				}
			}

			if (!empty($mimes)) {
				sort($mimes);
				$value = '[' . implode(';', $mimes) . ']';
			}
		} else {

			$x = $xpath;

			if (substr($xpath, 0, strlen($xpathExpr['product']['features'])) == $xpathExpr['product']['features']) {
				// Features
				$x .= '/' . $xpathExpr['product']['featureValues'];
			}

			$value = Helper::getSimpleXMLValue($itemNode, $x);
		}

		return $value;
	}

	private function findMappingForSource($dataportId, $node){
		$data = array();

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		if (!$dataport) {
			return null;
		}

		$config = unserialize($dataport['sourceconfig']);

		if (!is_array($config) || !array_key_exists('categoryClassId', $config) || !array_key_exists('categoryClassId', $config)) {
			return null;
		}

		$mappingModel = new Categorymapping();
		$categoryMappings = $mappingModel->find(array(
			'dataportId = ?' => $dataportId,
			'sourceCategoryId = ?' => $node,
		));

		$nameGetter = empty($config['categoryNameElement']) ? null : ('get' . ucfirst($config['categoryNameElement']));

		foreach ($categoryMappings as $row) {
			$name = '';

			/**
			 * @var $target Concrete
			 */
			$target = Concrete::getById($row['targetCategoryId']);

			if ($target instanceof Concrete && $target->getClassId() == $config['categoryClassId'] && method_exists($target, $nameGetter)) {
				try {
					$name = $target->$nameGetter();
				} catch (\Exception $e) {
					// Invalid name
					$name = '*invalid*';
				}
			}

			$data[] = array(
				'id' => $row['id'],
				'categoryId' => $row['targetCategoryId'],
				'name' => $name
			);
		}


		return $data;
	}


	private function findMappingForTarget($dataportId, $node, $hash){
		$data = array();

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		if (!$dataport) {
			return null;
		}

		$config = unserialize($dataport['sourceconfig']);

		if (!is_array($config) || !array_key_exists('categoryClassId', $config) || !array_key_exists('categoryClassId', $config)) {
			return null;
		}

		$version = null;
		if ($dataport && !empty($dataport->sourceconfig)) {
			if (array_key_exists('version', $config)) {
				$version = $config['version'];
			}
		}

		$xpathExpr = Helper::getBmecatXpathForVersion($version);
		if (empty($xpathExpr)) {
			return null;
		}

		$mappingModel = new Categorymapping();
		$categoryMappings = $mappingModel->find(array(
			'dataportId = ?' => $dataportId,
			'targetCategoryId = ?' => $node,
		));


		$bmeCategoryRepository = new BmeCategory();
		foreach ($categoryMappings as $row) {
			$name = $bmeCategoryRepository->findOne([
			    'file' => $hash,
                'categoryId' => $row['sourceCategoryId']
            ]);

			$data[] = array(
				'id' => $row['id'],
				'categoryId' => $row['sourceCategoryId'],
				'name' => $name['name']
			);
		}

		return $data;
	}

	private function _saveAttributeMapping($dataportId, $sourceCategoryId, $targetCategoryId, $targetAttribute, $xpath, $format, $type, $keyMapping, $calculation) {
		$success = false;

		// json_decode liefert stdClass, wird hier umgewandelt in ein array
		if ($format instanceof \stdClass) {
			$format = (array)$format;
		}

		$defaults = Helper::getMappingDefaults();
		if (array_key_exists($type, $defaults)) {
			$newFormat = $defaults[$type];
			if (is_array($format)) {
				foreach ($format as $attr => $value) {
					$newFormat[$attr] = $value;
				}
			}

			$format = serialize($newFormat);
		} else {
			$format = '';
		}

		// Sicherstellen, dass leere Strings statt null oder anderen falsy values verwendet werden
		if (!$calculation) {
			$calculation = '';
		}

		if (!$xpath) {
			$xpath = '';
		}

		$keyMapping = $keyMapping === true;

		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);

		$targetCategory = Concrete::getById($targetCategoryId);

		if ($dataport && !empty($dataport['sourceconfig'])) {
			$config = unserialize($dataport['sourceconfig']);
			$targetconfig = unserialize($dataport['targetconfig']);

			if ($sourceCategoryId && is_array($config) && array_key_exists('categoryClassId', $config) &&
				$targetCategory instanceof Concrete && $targetCategory->getClassId() == $config['categoryClassId']
				&& is_array($targetconfig) && array_key_exists('itemClass', $targetconfig)) {

				$targetClass = ClassDefinition::getById($targetconfig['itemClass']);

				if ($targetClass instanceof ClassDefinition) {
					$mappingModel = new Categorymapping();
					$existing = $mappingModel->findOne(array(
						'dataportId = ?' => $dataport['id'],
						'sourceCategoryId = ?' => $sourceCategoryId,
						'targetCategoryId = ?' => $targetCategory->getId(),
					));

					if ($existing && $existing['id'] > 0) {
						/**
						 * Übergebens Attribut hat die Form "class.shortDescription#en" und wird hier in seine Bestandteile zerlegt
						 */
						$targetMapping = explode('.', $targetAttribute);
						if (count($targetMapping) === 2) {
							$mappingType = $targetMapping[0];
							$attributeKey = $targetMapping[1];

							$locale = '';
							$attributeParts = explode('#', $attributeKey);
							$attributeName = $attributeParts[0];
							if (count($attributeParts) === 2) {
								$locale = $attributeParts[1];
							}


							$fields = array();
							if ($mappingType === 'class') {
								$fields = Helper::getClassFields($targetClass);
							} else {
								$fieldcollectionFields = Helper::getRecursiveFieldcollectionFields($targetCategoryId, $targetCategory->getClass(), $config['categoryfolder'], $config['categoryAttributeElement']);
								if (array_key_exists($mappingType, $fieldcollectionFields)) {
									$fields = $fieldcollectionFields[$mappingType];
								}
							}

							foreach ($fields as $field) {
								if ($field['key'] == $attributeKey) {

									$attributeMappingModel = new CategorymappingField();
									$existingMapping = $attributeMappingModel->find(['id' => $existing['id'], 'fieldCollection' => $mappingType, 'field' => $attributeName, 'locale' => $locale]);

									if ($existingMapping) {
										$saveResult = $attributeMappingModel->update(array(
											'xpath' => $xpath,
											'keyMapping' => $keyMapping,
											'format' => $format,
											'calculation' => $calculation,
										), ['id' => $existing['id'], 'fieldCollection' => $mappingType, 'field' => $attributeName, 'locale' => $locale]);
									} else {
										$saveResult = $attributeMappingModel->create(array(
											'id' => $existing['id'],
											'fieldCollection' => $mappingType,
											'field' => $attributeName,
											'locale' => $locale,
											'xpath' => $xpath,
											'keyMapping' => $keyMapping,
											'format' => $format,
											'calculation' => $calculation,
										));
									}

									if ($saveResult) {
										$success = true;
									}
								}
							}
						}
					}
				}
			}
		}

		return $success;
	}

	/**
	 * @param \DOMXPath $xp
	 * @param array $nodes
	 * @param array $xpathExpr
	 * @return array
	 */
	private function findSubCategories($xp, $nodes, $xpathExpr) {
		$children = array();
		$childrenXpath = implode('/', array($xpathExpr['category']['base'], $xpathExpr['category']['child']));

		foreach ($nodes as $node) {
			// Finde alle Kinder
			$childNodes = $xp->query(sprintf($childrenXpath, $node));

			$subNodes = array();
			foreach ($childNodes as $childNode) {
				// Hat ein Node weitere Kinder -> Rekursion
				$idNode = $xp->query($xpathExpr['category']['catId'], $childNode)->item(0);

				if ($idNode != null) {
					$subNodes[] = $idNode->nodeValue;
				}
			}

			$children = array_merge($children, $subNodes, $this->findSubCategories($xp, $subNodes, $xpathExpr));
		}

		return array_unique($children);
	}

	/**
	 * @param string $hash
	 * @param array $categoryIds
	 * @return array
	 */
	private function findSubCategoriesCache($hash, $categoryIds) {
		$children = array();

		if (is_array($categoryIds)) {
		    $bmeCategoryRepository = new BmeCategory();
			foreach ($categoryIds as $catId) {
				$childIds = $bmeCategoryRepository->find([
				        'file' => $hash,
                        'parentId' => $catId
                ]);
				$cids = array();
				foreach ($childIds as $cid) {
					$cids[] = $cid['categoryId'];
				}

				$children = array_merge($children, $cids, $this->findSubCategoriesCache($hash, $cids));
			}
		}

		return array_unique($children);
	}

	/**
	 * @param \SimpleXMLElement $priceNode
	 * @param array $xpathExpr
	 * @return array
	 */
	private function getPriceData($priceNode, $xpathExpr) {
		return array(
			'type' => Helper::getSimpleXMLValue($priceNode, $xpathExpr['product']['priceType']),
			'amount' => Helper::getSimpleXMLValue($priceNode, $xpathExpr['product']['priceAmount']),
			'currency' => Helper::getSimpleXMLValue($priceNode, $xpathExpr['product']['priceCurrency']),
			'tax' => Helper::getSimpleXMLValue($priceNode, $xpathExpr['product']['priceTax']),
			'factor' => Helper::getSimpleXMLValue($priceNode, $xpathExpr['product']['priceFactor']),
			'lowerBound' => Helper::getSimpleXMLValue($priceNode, $xpathExpr['product']['priceLowerBound']),
			'territory' => Helper::getSimpleXMLValue($priceNode, $xpathExpr['product']['priceTerritory']),
		);
	}

	public function testAction() {
		$dataportId = 2;
		$file = Helper::getFileForDataport($dataportId);
		$version = '1.2';
		$importKey = '538dc22752141';

		$data = Helper::compareToImport($file, $version, $importKey);

		return new JsonResponse($data);
	}
}
