<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\Tools;

use Blackbit\PimBundle\model\ImportStatus;
use Doctrine\DBAL\DBALException;
use Pimcore\Db;
use Pimcore\Extension\Bundle\Installer\AbstractInstaller;
use Pimcore\Model\User\Permission\Definition;
use Psr\Log\LoggerInterface;

/**
 * Created by JetBrains PhpStorm.
 * User: Dennis
 * Date: 18.07.12
 * Time: 13:51
 * To change this template use File | Settings | File Templates.
 */
class Installer extends AbstractInstaller {
	const TABLE_DATAPORT = 'plugin_pim_dataport';
	const TABLE_RAWITEM = 'plugin_pim_rawitem';
	const TABLE_RAWITEMDATA = 'plugin_pim_rawitemData';
	const TABLE_RAWITEMFIELD = 'plugin_pim_rawitemField';
	const TABLE_FIELDMAPPING = 'plugin_pim_fieldmapping';
	const TABLE_GROUPMAPPING = 'plugin_pim_groupmapping';
	const TABLE_GROUPMAPPINGFIELD = 'plugin_pim_groupmappingfield';
	const TABLE_GROUPMAPPINGTARGET = 'plugin_pim_groupmappingtarget';
	const TABLE_SHOPCONFIG = 'plugin_pim_shopconfig';
	const TABLE_IMPORTSTATUS = 'plugin_pim_importstatus';
	const TABLE_CATEGORYMAPPING = 'plugin_pim_categorymapping';
	const TABLE_CATEGORYMAPPING_FIELDS = 'plugin_pim_categorymapping_fields';
	const TABLE_BME_IMPORTLOG_CATEGORIES = 'plugin_pim_bmecat_importlog_categories';
	const TABLE_BME_IMPORTLOG_PRODUCTS = 'plugin_pim_bmecat_importlog_products';
	const TABLE_BME_IMPORTLOG_FEATURES = 'plugin_pim_bmecat_importlog_features';
	const TABLE_BME_FILE = 'plugin_pim_bmecat_file';
	const TABLE_BME_CATEGORY = 'plugin_pim_bmecat_category';
	const TABLE_BME_PRODUCT = 'plugin_pim_bmecat_product';
	const TABLE_BME_PRODUCT_CATEGORY = 'plugin_pim_bmecat_product_category';
	
	const BMECAT_CONFIG_FILE = 'globalBmecatAttributes.xml';

	/** @var LoggerInterface */
	private $logger;

    public function __construct(LoggerInterface $logger)
    {
        parent::__construct();
        $this->logger = $logger;
    }

    public function canBeInstalled()
    {
        return !$this->isInstalled();
    }

    public function canBeUninstalled()
    {
        return $this->isInstalled();
    }

    /**
	 * @return string $statusMessage
	 */
	public function install() {
		$this->createTables();

		// Creare config dir and copy default config to that directory
		if (!is_dir(self::getConfigPath())) {
			if(!@mkdir(self::getConfigPath(), 0755, true) && !is_dir(self::getConfigPath())) {
			    throw new \Exception('Could not create directory');
            }
		}

		$targetFilePath = self::getConfigPath() . DIRECTORY_SEPARATOR . self::BMECAT_CONFIG_FILE;

		if (!file_exists($targetFilePath)) {
			copy(__DIR__.'/../Resources/install/config/'. self::BMECAT_CONFIG_FILE, $targetFilePath);
		}

		// Install privilege to access this plugin
        Definition::create('plugin_bb_pim');

		return $this->isInstalled();
	}

	/**
	 * @return boolean $isInstalled
	 */
	public function isInstalled() {
        $permission = Definition::getByKey('plugin_bb_pim');

        $filename = self::getConfigPath() . DIRECTORY_SEPARATOR . self::BMECAT_CONFIG_FILE;

        return self::checkTables() && file_exists($filename) && is_readable($filename)
            && $permission instanceof Definition;
	}

	/**
	 * @return string $statusMessage
	 */
	public function uninstall() {
		$this->dropTables();

		return !self::isInstalled();
	}

	public function needsReloadAfterInstall(){
		return true;
	}

	public function createTables() {
		$result = true;

		// do the tables exist already?
		if(self::checkTables()) return true;

		try {
			$db = Db::get();

			$db->query("
			CREATE TABLE IF NOT EXISTS " . self::TABLE_DATAPORT . " (
				`id`				INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
				`name`			    VARCHAR(50) NOT NULL,
				`description`		VARCHAR(255) NOT NULL DEFAULT '',
				`sourcetype`		ENUM('xml', 'csv', 'excel', 'pimcore', 'bmecat') NOT NULL,
				`sourceconfig`		TEXT NOT NULL,
				`targetconfig`		TEXT NOT NULL,
				PRIMARY KEY (`id`)
			) ENGINE=InnoDb DEFAULT CHARSET=utf8;");

			$db->query('CREATE TABLE IF NOT EXISTS ' . self::TABLE_RAWITEM . ' (
				`id` int(11) UNSIGNED NOT NULL AUTO_INCREMENT,
				`dataportId` int(11) UNSIGNED NOT NULL,
				`inserted` datetime NOT NULL,
				`updated` datetime NOT NULL,
				`hash` varchar(32) NOT NULL,
				`priority` varchar(255) NOT NULL DEFAULT \'\',
				PRIMARY KEY (`id`),
				KEY `fk_rawItem_dataport` (`dataportId`),
				CONSTRAINT `fk_rawItem_dataport` FOREIGN KEY (`dataportId`) REFERENCES ' . self::TABLE_DATAPORT . ' (`id`),
				INDEX `index_rawItem_dataportId_priority` (`dataportId`, `priority`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8');

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_RAWITEMDATA . " (
				`rawItemId` int(11) UNSIGNED NOT NULL,
				`fieldNo` int(11) UNSIGNED NOT NULL,
				`value` text,
				PRIMARY KEY (`rawItemId`,`fieldNo`),
				CONSTRAINT `fk_rawitemdata_rawitem` FOREIGN KEY (`rawItemId`) REFERENCES " . self::TABLE_RAWITEM . " (`id`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_RAWITEMFIELD . " (
				`dataportId` int(11) UNSIGNED NOT NULL,
				`fieldNo` int(11) UNSIGNED NOT NULL,
				`name` varchar(100) NOT NULL DEFAULT '',
				`priority` TINYINT NOT NULL DEFAULT 0,
				PRIMARY KEY (`dataportId`,`fieldNo`),
				CONSTRAINT `fk_rawitemfield_dataport` FOREIGN KEY (`dataportId`) REFERENCES " . self::TABLE_DATAPORT . " (`id`) ON DELETE CASCADE ON UPDATE CASCADE,
				INDEX `index_rawItemField_dataportId_priority` (`dataportId`, `priority`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_FIELDMAPPING . " (
				`dataportId` int(11) UNSIGNED NOT NULL,
				`fieldName` varchar(255) NOT NULL,
				`locale` varchar(10) NOT NULL DEFAULT '',
				`fieldNo` int(11) UNSIGNED NULL,
				`keyMapping` BOOLEAN NOT NULL DEFAULT 0,
				`format` TEXT NOT NULL,
				`calculation` TEXT,
				`brickName` varchar(255) DEFAULT NULL,
	  			`targetBrickField` varchar(255) DEFAULT NULL,
				PRIMARY KEY (`dataportId`,`fieldName`,`locale`),
				CONSTRAINT `fk_fieldmapping_dataport` FOREIGN KEY (`dataportId`) REFERENCES " . self::TABLE_DATAPORT . " (`id`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_GROUPMAPPING . " (
				`id` int(11) UNSIGNED NOT NULL AUTO_INCREMENT,
				`dataportId` int(11) UNSIGNED NOT NULL,
				`active` BOOLEAN NOT NULL,
				`created` datetime NOT NULL,
				`updated` datetime NOT NULL,
				PRIMARY KEY (`id`),
				CONSTRAINT `fk_groupmapping_dataport` FOREIGN KEY (`dataportId`) REFERENCES " . self::TABLE_DATAPORT . " (`id`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_GROUPMAPPINGFIELD . " (
				`groupmappingId` int(11) UNSIGNED NOT NULL,
				`fieldNo` int(11) UNSIGNED NOT NULL,
				`value` text,
				PRIMARY KEY (`groupmappingId`, `fieldNo`),
				CONSTRAINT `fk_groupmappingfield_groupmapping` FOREIGN KEY (`groupmappingId`) REFERENCES " . self::TABLE_GROUPMAPPING . " (`id`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_GROUPMAPPINGTARGET . " (
				`groupmappingId` int(11) UNSIGNED NOT NULL,
				`documentId` int(11) UNSIGNED NOT NULL,
				`editable` varchar(255) NOT NULL,
				`purgemappings` tinyint(1) NOT NULL DEFAULT 0,
				PRIMARY KEY (`groupmappingId`, `documentId`, `editable`),
				CONSTRAINT `fk_groupmappingtarget_groupmapping` FOREIGN KEY (`groupmappingId`) REFERENCES " . self::TABLE_GROUPMAPPING . " (`id`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_SHOPCONFIG . " (
				`key` varchar(255) NOT NULL,
				`value` text NOT NULL,
				PRIMARY KEY (`key`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_IMPORTSTATUS . " (
				`key` varchar(255) NOT NULL,
				`importType` int(11) UNSIGNED NOT NULL,
				`dataportId` int(11) UNSIGNED NOT NULL,
				`startDate` DATETIME NOT NULL,
				`endDate` DATETIME NULL,
				`lastUpdate` DATETIME NOT NULL,
				`status` int(11) UNSIGNED NOT NULL DEFAULT 0,
				`totalItems` int(11) UNSIGNED NOT NULL DEFAULT 0,
				`doneItems` int(11) UNSIGNED NOT NULL DEFAULT 0,
				PRIMARY KEY (`key`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_CATEGORYMAPPING . " (
				`id` int(11) UNSIGNED NOT NULL AUTO_INCREMENT,
				`dataportId` int(11) UNSIGNED NOT NULL,
				`sourceCategoryId` varchar(255) NOT NULL,
				`targetCategoryId` int(11) UNSIGNED NOT NULL,
				PRIMARY KEY (`id`),
				CONSTRAINT `uq_categorymapping` UNIQUE (`dataportId`, `sourceCategoryId`, `targetCategoryId`),
				CONSTRAINT `fk_cgmapping_dataport` FOREIGN KEY (`dataportId`) REFERENCES " . self::TABLE_DATAPORT . " (`id`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_CATEGORYMAPPING_FIELDS . " (
				`id` int(11) UNSIGNED NOT NULL,
				`fieldCollection` varchar(255),
				`field` varchar(255) NOT NULL,
				`locale` varchar(10) NOT NULL DEFAULT '',
				`xpath` varchar(1024) NOT NULL,
				`keyMapping` BOOLEAN NOT NULL DEFAULT 0,
				`format` TEXT NOT NULL,
				`calculation` TEXT NOT NULL,
				PRIMARY KEY (`id`, `fieldCollection`, `field`, `locale`),
				CONSTRAINT `fk_cgmappingfields_cgmapping` FOREIGN KEY (`id`) REFERENCES " . self::TABLE_CATEGORYMAPPING . " (`id`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_BME_IMPORTLOG_CATEGORIES . " (
				`importKey` varchar(255) NOT NULL,
				`categoryId` varchar(255) NOT NULL,
				`hash` varchar(32) NOT NULL,
				PRIMARY KEY (`importKey`, `categoryId`),
				CONSTRAINT `fk_categorylog_importstatus` FOREIGN KEY (`importKey`) REFERENCES " . self::TABLE_IMPORTSTATUS . " (`key`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_BME_IMPORTLOG_PRODUCTS . " (
				`importKey` varchar(255) NOT NULL,
				`productId` varchar(255) NOT NULL,
				`hash` varchar(32) NOT NULL,
				PRIMARY KEY (`importKey`, `productId`),
				CONSTRAINT `fk_productlog_importstatus` FOREIGN KEY (`importKey`) REFERENCES " . self::TABLE_IMPORTSTATUS . " (`key`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_BME_IMPORTLOG_FEATURES . " (
				`importKey` varchar(255) NOT NULL,
				`name` varchar(255) NOT NULL,
				PRIMARY KEY (`importKey`, `name`),
				CONSTRAINT `fk_featurelog_importstatus` FOREIGN KEY (`importKey`) REFERENCES " . self::TABLE_IMPORTSTATUS . " (`key`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_BME_FILE . " (
				`hash` varchar(32) NOT NULL,
				`created` DATETIME NOT NULL,
				PRIMARY KEY (`hash`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_BME_CATEGORY . " (
				`file` varchar(32) NOT NULL,
				`categoryId` varchar(255) NOT NULL,
				`name` varchar(255) NOT NULL,
				`parentId` varchar(255),
				`hash` varchar(32) NOT NULL,
				PRIMARY KEY (`file`, `categoryId`),
				CONSTRAINT `fk_bme_category_file` FOREIGN KEY (`file`) REFERENCES " . self::TABLE_BME_FILE . " (`hash`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_BME_PRODUCT . " (
				`file` varchar(32) NOT NULL,
				`productId` varchar(255) NOT NULL,
				`xml` TEXT NOT NULL,
				`hash` varchar(32) NOT NULL,
				PRIMARY KEY (`file`, `productId`),
				CONSTRAINT `fk_bme_product_file` FOREIGN KEY (`file`) REFERENCES " . self::TABLE_BME_FILE . " (`hash`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

			$db->query("CREATE TABLE IF NOT EXISTS " . self::TABLE_BME_PRODUCT_CATEGORY . " (
				`file` varchar(32) NOT NULL,
				`productId` varchar(255) NOT NULL,
				`categoryId` varchar(255) NOT NULL,
				PRIMARY KEY (`file`, `productId`, `categoryId`),
				CONSTRAINT `fk_bme_product2category_file` FOREIGN KEY (`file`) REFERENCES " . self::TABLE_BME_FILE . " (`hash`) ON DELETE CASCADE ON UPDATE CASCADE
			) ENGINE=InnoDB DEFAULT CHARSET=utf8");

		} catch (\Exception $e) {
			$result = false;
			$this->logger->error("Failed to create tables: " . $e->getMessage());
			$this->uninstall();
		}

		return $result;
	}

    public static function checkTables() {
        $schemaManager = Db::get()->getSchemaManager();
        foreach([
            self::TABLE_DATAPORT, self::TABLE_RAWITEM, self::TABLE_RAWITEMDATA, self::TABLE_RAWITEMFIELD, self::TABLE_FIELDMAPPING, self::TABLE_GROUPMAPPING, self::TABLE_GROUPMAPPINGFIELD, self::TABLE_GROUPMAPPINGTARGET, self::TABLE_SHOPCONFIG, self::TABLE_IMPORTSTATUS, self::TABLE_CATEGORYMAPPING, self::TABLE_CATEGORYMAPPING_FIELDS, self::TABLE_BME_IMPORTLOG_CATEGORIES
        ] as $tableName) {
            if(count($schemaManager->listTableColumns($tableName)) === 0) {
                return false;
            }
        }

        return true;
    }

	private function dropTables() {
		$result = true;

		try {
			$db = Db::get();

			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_BME_IMPORTLOG_CATEGORIES . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_CATEGORYMAPPING_FIELDS . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_CATEGORYMAPPING . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_IMPORTSTATUS . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_SHOPCONFIG . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_GROUPMAPPINGTARGET . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_GROUPMAPPINGFIELD . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_GROUPMAPPING . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_FIELDMAPPING . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_RAWITEMFIELD . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_RAWITEMDATA . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_RAWITEM . "`");
			$db->query("DROP TABLE IF EXISTS `" . self::TABLE_DATAPORT . "`");

		} catch (DBALException $e) {
			$result = false;
			$this->logger->error('Failed to remove tables: '. $e->getMessage());
		}

		return $result;
	}

	/**
	 * Maintenance Hook
	 */
	public function maintenance() {
		$abortionThreshold = ImportStatus::ABORTION_THRESHOLD;

		$sql = 'UPDATE ' . self::TABLE_IMPORTSTATUS . ' SET `status` = ?, `endDate` = NOW()'
			. ' WHERE lastUpdate <= DATE_SUB(NOW(), INTERVAL ' . $abortionThreshold . ' SECOND)'
			. ' AND `status` = ?';

		$db = Db::get();
		$db->query($sql, array(
			ImportStatus::STATUS_ABORTED,
			ImportStatus::STATUS_RUNNING,
		));
	}

	public static function getConfigPath() {
		return implode(DIRECTORY_SEPARATOR, array(PIMCORE_PRIVATE_VAR, 'bundles', 'BlackbitPim'));
	}
}
