/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

pimcore.registerNS("pimcore.plugin.Pim.VariantConfig");
pimcore.plugin.Pim.VariantConfig = Class.create({
    variantContainer: null,
    variantStatusContainer: null,
    menuSourceType: null,
    menuSourceId: null,
    menuSelectedAction: null,

    initialize: function (layoutId) {
        this.layoutId = layoutId;

        this.initTreeModel();
        // create layout
        this.getLayout();
    },
    getVariantTree: function () {
        if (!this.tree) {
            this.treeFilterValue = '';
            var filterTreeTask = new Ext.util.DelayedTask(function(){
                this.filterTree();
            }.bind(this));

            this.tree = Ext.create('Ext.tree.Panel', {
                id: "Variant_dataport_tree",
                region: "west",
                collapsible: false,
                useArrows: false,
                autoScroll: true,
                animate: false,
                containerScroll: true,
                border: true,
                width: 350,
                split: true,
                lines: true,
                root: {
                    text: 'Variants',
                    nodeType: 'async',
                    expanded: true,
                    id: '0'
                },
                rootVisible: false,
                store: this.dataStore,
                listeners: {
                    click: {
                        element: 'el', //bind to the underlying el property on the panel
                        fn: function (sm, node, index) {
                            if (sm && sm.record && sm.record.id && sm.record.data.type != 'folder') {
                                this.id = sm.record.id;
                                this.onTreeNodeClick()
                            }
                        }.bind(this)
                    }
                },
                tbar: {
                    items: [
                        '->',
                        {
                            xtype: 'textfield',
                            name: 'variantFilter',
                            width: 120,
                            listeners: {
                                'change': function(combo, newValue, oldValue) {
                                    this.treeFilterValue = newValue;
                                    filterTreeTask.delay(500);
                                }.bind(this)
                            }
                        },
                        {
                            iconCls: "pimcore_icon_reload",
                            handler: function () {
                                this.treeFilterValue = '';
                                Ext.ComponentQuery.query('[name=variantFilter]')[0].setValue('');
                                Ext.ComponentQuery.query('[name=variantFilter]')[0].setRawValue('');
                                this.reloadTree();
                            }.bind(this)
                        }
                    ]
                }
            });

            this.tree.on('itemcontextmenu', function (view, rec, node, index, event) {
                event.stopEvent();
                this.menuSourceType = rec.data.type;
                this.menuSourceId = rec.data.id;
                this.onTreeNodeContextmenu(rec.id, event);
            }.bind(this))
            this.tree.on('containercontextmenu', function (view, event, eOpts) {
                event.stopEvent();
                this.menuSourceType = 'root';
                this.menuSourceId = '0';
                if (event.record && event.record.data) {
                    this.menuSourceType = event.record.data.type;
                    this.menuSourceId = event.record.id;
                }
                var menu = Ext.create('Ext.menu.Menu');
                menu.add(this.menuAddFolder());
                menu.add(this.menuAddItem());
                menu.showAt(event.getXY());
            }.bind(this))
        }
        return this.tree;
    },

    initTreeModel: function () {
        this.dataStore = Ext.create('Ext.data.TreeStore', {
            model: 'VariantTreeModel',
            remoteFilter: false
        });
    },

    getContainer: function () {
        if (!this.variantContainer) {
            this.variantContainer = new Ext.TabPanel({
                region: "center",
                listeners: {
                    'tabchange': function (tabpanel, tab) {
                        var dataportId = tab.dataportId;
                        var tree = this.tree;
                        this.tree.getRootNode().cascade(function () {
                            if (this.id === dataportId) {
                                tree.setSelection(this);
                            }
                        });
                    }.bind(this)
                }
            });
        }

        return this.variantContainer;
    },

    promptNewName: function () {
        Ext.MessageBox.prompt(t('pim.add_' + this.menuSelectedAction), t('pim.enter_new_' + this.menuSelectedAction + '_name'), this.addVariant.bind(this), null, null, "");
    },

    addVariant: function (button, value, object) {
        if (button == "ok") {

            Ext.Ajax.request({
                url: "/admin/BlackbitPim/variantconfig/add",
                params: {
                    name: value,
                    type: this.menuSelectedAction,
                    parentId: this.menuSourceId
                },
                success: function (response) {
                    try {
                        var response = Ext.decode(response.responseText);
                        if (!(response && response.success)) {
                            pimcore.helpers.showNotification(t("error"), t("pim.error_creating_variant"), "error", t(response.errorMessage));
                        }
                    } catch (e) {
                        pimcore.helpers.showNotification(t("error"), t("pim.error_creating_variant"), "error");
                    }
                    this.reloadTree();
                }.bind(this)
            });
        }
    },

    /**
     * create tab panel
     * @returns Ext.Panel
     */
    getLayout: function () {
        var tabPanel = Ext.getCmp("pimcore_panel_tabs");
        var existingTab = Ext.getCmp(this.layoutId);
        if(existingTab) {
            tabPanel.setActiveItem(this.layoutId);
            return existingTab;
        }

        if (!this.layout) {
            if (!this.variantStatusContainer) {
                this.variantStatusContainer = Ext.create('pimcore.plugin.Pim.VariantStatusPanel', {});
            }

            var centerLayout = new Ext.Panel({
                id: 'variant_center_layout',
                region: 'center',
                border: false,
                layout: "border",
                closable: false,
                items: [this.variantStatusContainer, this.getContainer()]
            });

            this.layout = new Ext.Panel({
                id: this.layoutId,
                title: t('pim.variant_config'),
                iconCls: "pimcore_icon_settings",
                border: false,
                layout: "border",
                closable: true,
                items: [this.getVariantTree(), centerLayout]
            });

            // add event listener
            var layoutId = this.layoutId;
            this.layout.on("destroy", function () {
                pimcore.globalmanager.remove(layoutId);
                this.variantStatusContainer.destroyComponent();
            }.bind(this));

            // find the main tabpanel and add ours
            tabPanel.add(this.layout);
            tabPanel.setActiveItem(this.layoutId);

            // show the panels by refreshing
            pimcore.layout.refresh();
        }

        return this.layout;
    },

    getTreeNodeListeners: function () {
        return {
            'click': this.onTreeNodeClick,
            'contextmenu': this.onTreeNodeContextmenu
        };
    },

    filterTree: function() {
        if (this.tree) {
            this.dataStore.filterBy(function(record){
                if (record.data.text.toLowerCase().indexOf(this.treeFilterValue.toLowerCase()) !== -1) {
                    return true;
                }

                if (!record.data.leaf) {
                    if (record.data.root) {
                        return true;
                    }

                    var childNodes = record.childNodes;
                    var hide = true;
                    if (childNodes) {
                        var i;
                        for (i = 0; i < childNodes.length; i++) {
                            var childNode = childNodes[i];
                            if (childNode.get("visible")) {
                                hide = false;
                                break;
                            }
                        }
                    }

                    return !hide;
                }
            }.bind(this));
        }
    },

    reloadTree: function () {
        if (this.tree) {
            this.dataStore.clearFilter(true);
            this.dataStore.getProxy().extraParams.selectedItem = this.menuSourceId;
            this.dataStore.reload();
        }
    },

    getOpenPanels: function (dataportId) {
        var panels = [];
        try {
            var container = this.getContainer();

            container.items.each(function (child, index, total) {
                if (child.dataport && child.dataport.id == dataportId) {
                    panels.push(child);
                }
            });

        } catch (e) {
            console.log(e);
        }
        return panels;
    },

    removeOpenPanel: function (dataportId) {
        var container = this.getContainer();

        var toBeRemoved = this.getOpenPanels(dataportId);
        Ext.each(toBeRemoved, function (panel) {
            container.remove(panel);
        });
    },

    onTreeNodeClick: function () {
        if (this.id > 0) {
            var container = this;
            var existingPanels = container.getOpenPanels(this.id);

            if (existingPanels.length > 0) {
                existingPanels[0].show();
            } else {
                var variantContainer = container.getContainer();
                variantContainer.add(Ext.create('pimcore.plugin.Pim.VariantPanel', {dataportId: this.id})).show();
            }

            pimcore.layout.refresh();
        }
    },

    onTreeNodeContextmenu: function (index, event) {
        var menu = Ext.create('Ext.menu.Menu');
        if (this.menuSourceType == 'folder') {
            menu.add(this.menuAddFolder());
            menu.add(this.menuAddItem());
        }
        menu.add(Ext.create('Ext.menu.Separator', {}));
        menu.add(this.menuGenerate());
        menu.add(Ext.create('Ext.menu.Separator', {}));
        menu.add(Ext.create('Ext.menu.Item', {
            text: t('delete'),
            iconCls: "pimcore_icon_delete",
            handler: function () {
                Ext.Msg.confirm(t('delete'), t('pim.delete_' + this.menuSourceType + '_message'), function (btn) {
                    if (btn == 'yes') {
                        Ext.Ajax.request({
                            url: "/admin/BlackbitPim/variantconfig/delete/"+index,
                            success: function (response) {
                                try {
                                    var response = Ext.decode(response.responseText);
                                    if (!(response && response.success)) {
                                        pimcore.helpers.showNotification(t("error"), t("pim.error_creating_variant"), "error", t(response.errorMessage));
                                    }
                                } catch (e) {
                                    pimcore.helpers.showNotification(t("error"), t("pim.error_creating_variant"), "error");
                                }
                                this.removeOpenPanel(response.id);
                                this.menuSourceId = response.parentId;
                                this.reloadTree();
                            }.bind(this)
                        });
                    }
                }.bind(this));
            }.bind(this)
        }));

        menu.showAt(event.getXY());
    },

    menuAddFolder: function () {
        return Ext.create('Ext.menu.Item', {
            text: t("pim.add_folder"),
            iconCls: "pimcore_icon_folder",
            handler: function() {
                this.menuSelectedAction = 'folder';
                this.promptNewName();
            }.bind(this)
        });
    },

    menuAddItem: function () {
        return Ext.create('Ext.menu.Item', {
            text: t("pim.add_variant"),
            iconCls: "pimcore_icon_add",
            handler: function() {
                this.menuSelectedAction = 'variant';
                this.promptNewName();
            }.bind(this)
        });
    },

    menuGenerate: function() {
        return Ext.create('Ext.menu.Item', {
            text: t("pim.generate_variant" + ((this.menuSourceType === 'folder')? '_all': '')),
            iconCls: "pimcore_icon_system",
            handler: function() {
                Ext.Msg.confirm(t('generate'), t('pim.generate_message'), function (btn) {
                    if (btn == 'yes') {
                        Ext.Ajax.request({
                            url: "/admin/BlackbitPim/variantgenerator/add",
                            params: {
                                variantId: this.menuSourceId,
                                type: this.menuSourceType
                            }
                        });
                    }
                }.bind(this));
            }.bind(this)
        });
    },
});
