<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\Parser;

use PhpOffice\PhpSpreadsheet\RichText\RichText;
use PhpOffice\PhpSpreadsheet\Worksheet\RowCellIterator;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use Pimcore\Logger;
use Psr\Log\LoggerInterface;

/**
 * Parses Excel-Files according to given configuration
 */
class ExcelParser implements Parser
{
    use ResourceBasedParser;

    /** @var array */
    private $config;

    /** @var LoggerInterface */
    private $logger;

    public function __construct(array $config, LoggerInterface $logger)
    {
        $this->config = $config;
        $this->logger = $logger;
    }

    /**
     * Verarbeitet eine Excel-Datei anhand der übergebenen Konfiguration
     *
     * @param int|null $limit Maximale Anzahl an Einträgen, die zurückgeliefert werden sollen
     *
     * @return array Aus der Datei ausgelesene Einträge; max $limit, wenn $limit != null
     */
    public function parse($limit = null)
    {
        $source = $this->config['file'];

        try {
            $filePathOrUrl = $this->getFileOrUrl($source);
        } catch (\InvalidArgumentException $e) {
            $this->logger->warning($e->getMessage());
        }

        if (empty($this->config['fields'])) {
            $this->logger->error("No field definition given in config");
            
            return [];
        }
        
        $excelFile = \PhpOffice\PhpSpreadsheet\IOFactory::load($filePathOrUrl);
        
        // Get worksheet
        $worksheet = $excelFile->getSheetByName($this->config['sheet']);
        if (!$worksheet instanceof Worksheet) {
            $worksheet = $excelFile->getSheet(0);
        }
        
        $fieldsToIndexes = [];
        $mergedRangedCellValues = [];

        $items = [];
        try {
            if ($worksheet !== false) {
                foreach ($worksheet->getRowIterator() as $row) {
                    /** @var RowCellIterator $cellIterator */
                    $cellIterator = $row->getCellIterator();

                    if ($row->getRowIndex() === 1) {
                        if ($this->config['hasHeader'] == 1) {
                            foreach ($this->config['fields'] as $field => $values) {
                                foreach ($cellIterator as $cell) {
                                    
                                    // Get field index if value of cell equals configured column. e.g.: Value1,Value2
                                    if (trim($cell->getValue()) === trim($values['column']) || $cell->getColumn() === $values['column']) {
                                        $fieldsToIndexes[$cell->getColumn()] = $field;
                                    }
                                }
                            }
                            
                            continue;
                        }
                        
                        foreach ($this->config['fields'] as $field => $values) {
                            $colIndex = (int)$values['column'];
                            $fieldsToIndexes[$colIndex] = $field;
                        }
                    }
                    
                    $item = [];
                    
                    foreach ($fieldsToIndexes as $index => $field) {
                        foreach ($cellIterator as $cell) {
                            
                            if ($cell->getColumn() === $index) {
                                $value = $cell->getFormattedValue();
                                
                                // Check value for string like NULL and set value to null
                                if (strtoupper($value) === 'NULL') {
                                    $value = null;
                                }

                                /*
                                 * Handle merged range value cells
                                 *
                                 * In ranges the cell of first row and first column of the range contains the value
                                 * all other ranged cells contains a marker with the ranged cells, e.g. A2:A7,
                                 * so to set the value for each ranged cells, store the value in array to access
                                 * it when cell is not an merged range value cell.
                                 */
                                $isInMergeRange = $cell->isInMergeRange();
                                if ($isInMergeRange && $cell->isMergeRangeValueCell()) {
                                    $mergedRangedCellValues[$cell->getMergeRange()] = $value;
                                } elseif($isInMergeRange && !$cell->isMergeRangeValueCell()) {
                                    $value = $mergedRangedCellValues[$cell->getMergeRange()];
                                }
                                
                                if ($value instanceof RichText) {
                                    $value = $value->getPlainText();
                                }
                                
                                $item[$field] = $value;
                            }
                        }
                    }
                    
                    $items[] = $item;
                    
                    if (null !== $limit && \count($items) >= $limit) {
                        break;
                    }
                }
            }
        } catch (\Exception $e) {
            $this->logger->error("Unable to parse Excel file \"{$source}\": " . $e);
            
            return [];
        }

        if(empty($limit)) {
            $this->archive($filePathOrUrl, $source);
        }

        return $items;
    }
}
