/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

pimcore.registerNS("pimcore.plugin.Pim.VariantConfig");
pimcore.plugin.Pim.VariantPanel = Ext.extend(Ext.TabPanel, {
    configPanel: null,
    previewPanel: null,
    manualPanel: null,
    dataport: null,
    dataportPanel: null,
    sourceConfigPanel: null,
    sourceConfigBasePanel: null,
    sourceConfigRawitemPanel: null,
    manyToManyObjectRelation: null,

    initComponent: function () {
        this.configPanel = Ext.create('Ext.Panel', {
            title: t('pim.dataport_configpanel'),
            iconCls: '',
            scrollable: true,
            buttons: [
                {
                    text: t("save"),
                    iconCls: "pimcore_icon_save",
                    handler: this.save.bind(this)
                },
                {
                    text: t("pim.generate_variant"),
                    iconCls: "pimcore_icon_system",
                    handler: this.generate.bind(this)
                }
            ]
        });

        this.configPanel.on('afterrender', function (panel) {
            this.getEl().mask();
            this.dataportPanel = this.createDataportPanel();
            this.sourceConfigPanel = Ext.create('Ext.Panel', {
                flex: 1,
                border: false
            });

            panel.add(this.dataportPanel);
            panel.add(this.sourceConfigPanel);

            pimcore.layout.refresh();

            this.populateDataportForm();
        }.bind(this));

        Ext.apply(this, {
            title: t('pim.loading'),
            iconCls: 'pimcore_icon_loading',
            closable: true,
            activeTab: 0,
            items: [this.configPanel, this.manualPanel, this.previewPanel],
            listeners: {
                'tabchange': function () {
                    pimcore.layout.refresh();
                }
            }
        });

        pimcore.plugin.Pim.DataportPanel.superclass.initComponent.call(this);
    },

    populateDataportForm: function () {
        this.dataportPanel.getForm().load({
            url: '/admin/BlackbitPim/variantconfig/get',
            params: {
                id: this.dataportId
            },
            failure: function (form, action) {
                this.getEl().unmask();
                pimcore.helpers.showNotification(t("error"), t(action.result.errorMessage), "error");
            }.bind(this),
            success: function (form, action) {
                this.getEl().unmask();
                this.dataport = action.result.data;

                this.setTitle(this.dataport.name);
                this.setIconCls('');

                this.renderSourceConfigPanel(this.dataport.sourcetype);
            }.bind(this)
        });
    },

    createDataportPanel: function () {
        var variantConfigStore = Ext.create('Ext.data.JsonStore', {
            model: 'DataPortModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/variantconfig/get-variant-configs',
                reader: {
                    type: 'json',
                    rootProperty: 'data'
                }

            },
            autoLoad: true
        });

        var form = Ext.create('Ext.form.Panel', {
            border: false,
            padding: 5,
            defaults: {
                width: 600
            },
            labelWidth: 300,
            layout: {
                type: 'vbox',
                align: 'streched'
            },
            items: [
                {
                    xtype: 'textfield',
                    name: 'name',
                    fieldLabel: t('pim.dataport_name'),
                    allowBlank: false
                },
                {
                    xtype: 'textfield',
                    name: 'description',
                    fieldLabel: t('pim.dataport_description')
                },
                {
                    xtype: 'combo',
                    hiddenName: 'sourcetype',
                    name: 'sourcetype',
                    fieldLabel: t('pim.dataport_sourcetype'),
                    forceSelection: true,
                    lazyInit: false,
                    editable: false,
                    triggerAction: 'all',
                    store: variantConfigStore,
                    valueField: 'id',
                    displayField: 'name',
                    value: '',
                    listeners: {
                        'change': function(combo, newValue, oldValue) {
                            if(this.sourceConfigRawitemPanel && this.sourceConfigRawitemPanel.store.data.items.length > 0) {
                                Ext.MessageBox.show({
                                    title: t('pim.dataport_sourcetype.change.question'),
                                    msg: t('pim.dataport_sourcetype.change.warning'),
                                    buttons: Ext.MessageBox.YESNO,
                                    fn: function (e) {
                                        switch (e) {
                                            case 'yes':
                                                this.renderSourceConfigPanel(newValue);
                                                break;
                                            default:
                                                combo.suspendEvent('change');
                                                combo.select(oldValue);
                                                combo.resumeEvent('change');
                                        }
                                    }.bind(this),
                                    icon: Ext.MessageBox.QUESTION
                                });
                            } else {
                                this.renderSourceConfigPanel(newValue);
                            }
                        }.bind(this)
                    }
                }
            ]
        });

        return form;
    },

    renderSourceConfigPanel: function (sourcetype) {
        if(!this.dataport) {
            return;
        }

        this.sourceConfigPanel.removeAll();
        this.setActiveTab(0);
        this.sourceConfigBasePanel = null;
        this.sourceConfigRawitemPanel = null;

        var additionalColumns = this.getColumnConfigForSourcetype(sourcetype);

        // Rohdatenfeldkonfiguration
        if (typeof PimcoreVariantconfigModel == 'undefined') {
            var readerFields = [
                {name: 'fieldNo', allowBlank: false, type: 'integer'},
                {name: 'name', allowBlank: false},
            ];

            Ext.each(additionalColumns, function (column) {
                readerFields.push({name: column.dataIndex});
            });

            Ext.define('PimcoreVariantconfigModel', {
                extend: 'Ext.data.Model',
                fields: readerFields,
                idProperty: 'fieldNo',
                root: 'fields'
            });
        }

        var store = new Ext.data.JsonStore({
            model: 'PimcoreVariantconfigModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/variantconfig/get-rawitemfield-config/' + this.dataport.id,
                reader: {
                    type: 'json',
                    rootProperty: 'fields'
                }
            },
            autoLoad: true
        });

        this.manyToManyObjectRelation = new pimcore.object.tags.manyToManyObjectRelation(
            this.getSourceconfig('objectData'),
            {
                "fieldtype": "manyToManyObjectRelation",
                "width": "",
                "height": "",
                "maxItems": "1",
                "queryColumnType": "text",
                "phpdocType": "array",
                "relationType": true,
                "visibleFields": null,
                "lazyLoading": false,
                "classes": [
                    {"classes": "Product"}
                ],
                "pathFormatterClass": "",
                "name": "Template Object",
                "title": "Template Object",
                "tooltip": "",
                "mandatory": true,
                "noteditable": false,
                "index": false,
                "locked": false,
                "style": "",
                "permissions": null,
                "datatype": "data",
                "invisible": false,
                "visibleGridView": false,
                "visibleSearch": false
            }
        );

        // Panel für Importkonfiguration
        this.sourceConfigBasePanel = new Ext.form.FormPanel({
            border: false,
            padding: 5,
            height: 120,
            defaults: {
                width: 600
            },
            labelWidth: 300,
            items: [
                this.manyToManyObjectRelation.getLayoutEdit()
            ]
        });

        var columnConfig = [];
        columnConfig.push({
            header: "#",
            width: 40,
            fixed: true,
            dataIndex: 'fieldNo',
            editor: {
                xtype: 'numberfield',
                allowBlank: false
            }
        });
        columnConfig.push({
            header: "name",
            width: 'auto',
            dataIndex: 'name',
            flex: 2,
            editor: {
                xtype: 'textfield',
                allowBlank: false
            }
        });

        Ext.each(additionalColumns, function (column) {
            columnConfig.push(column);
        });

        var rowEditing = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToMoveEditor: 1,
            autoCancel: true
        });

        this.sourceConfigRawitemPanel = Ext.create('Ext.grid.Panel', {
            title: '',
            store: store,
            columns: columnConfig,
            flex: 1,
            border: false,
            frame: false,
            autoScroll: true,
            layout: 'fit',
            minHeight:300,
            dockedItems: [{
                xtype: 'toolbar',
                dock: 'top',
                items: [
                    {
                        text: t('add'),
                        handler: this.onAddDataport.bind(this),
                        iconCls: "pimcore_icon_add"
                    },
                    '-',
                    {
                        text: t('delete'),
                        handler: this.onDeleteDataport.bind(this),
                        iconCls: "pimcore_icon_delete"
                    }
                ]
            }],
            listeners: {
                edit: function (editor, e) {
                    this.onAddDataport();
                }.bind(this)
            },
            plugins: [rowEditing],
            viewConfig: {
                plugins: {
                    ptype: 'gridviewdragdrop',
                    dragText: t('pim.dataport.change-order-hint')
                }
            }
        });

        this.sourceConfigPanel.add(this.sourceConfigBasePanel, this.sourceConfigRawitemPanel);

        pimcore.layout.refresh();
    },

    onAddDataport: function () {
        if (this.sourceConfigRawitemPanel instanceof Ext.grid.Panel) {
            var store = this.sourceConfigRawitemPanel.getStore();

            var lastRecord = this.sourceConfigRawitemPanel.getStore().getAt(this.sourceConfigRawitemPanel.getStore().getCount()-1);

            if(!lastRecord || lastRecord.get('name')) {
                var maxNum = 0;
                if(lastRecord) {
                    maxNum = lastRecord.get('fieldNo');
                }
                var u = new store.model();
                u.set('fieldNo', maxNum + 1);

                store.add(u);
            }
        }
    },

    onDeleteDataport: function () {
        if (this.sourceConfigRawitemPanel instanceof Ext.grid.Panel) {
            var rec = this.sourceConfigRawitemPanel.getSelectionModel().getSelection()[0];
            if (!rec) {
                return false;
            }

            Ext.Msg.confirm(
                t('delete'),
                t("Are you sure?"),
                function (button) {
                    if (button == 'yes') {
                        this.sourceConfigRawitemPanel.getStore().remove(rec);
                        this.save();
                    }
                }.bind(this)
            );
        }
    },

    getSourceconfig: function (key) {
        if (this.dataport.sourceconfig && key in this.dataport.sourceconfig) {
            return this.dataport.sourceconfig[key];
        }

        return null;
    },

    getColumnConfigForSourcetype: function (sourcetype) {
        var columns = [];

        Ext.each(this.dataport.fields, function (field) {
            columns.push({
                header: field.name,
                width: 'auto',
                dataIndex: field.name,
                flex: 2,
                editor: {
                    xtype: field.type,
                    allowBlank: false
                }
            });
        });

        return columns;
    },

    save: function (cb) {
        if (!Ext.isFunction(cb)) {
            cb = function () {
            };
        }

        if (this.dataportPanel.getForm().isValid() && this.sourceConfigBasePanel.getForm().isValid()) {
            this.getEl().mask();
            var baseData = this.dataportPanel.getForm().getFieldValues();
            var sourceData = Ext.util.JSON.encode(this.sourceConfigBasePanel.getForm().getFieldValues());

            var gridData = [];
            if (this.sourceConfigRawitemPanel instanceof Ext.grid.Panel) {
                this.sourceConfigRawitemPanel.getStore().each(function (record) {
                    var data = {};
                    var f = record.fields;

                    for (var i = 0, len = f.length; i < len; i++) {
                        var field = f[i];
                        data[field.name] = record.get(field.name);
                    }

                    gridData.push(data);
                });
            }

            var rawitemData = Ext.util.JSON.encode(gridData);

            baseData['id'] = this.dataport.id;
            baseData['sourceData'] = sourceData;
            baseData['objectData'] = Ext.util.JSON.encode(this.manyToManyObjectRelation.getValue());
            baseData['rawitemData'] = rawitemData;

            Ext.Ajax.request({
                url: "/admin/BlackbitPim/variantconfig/update",
                method: 'post',
                params: baseData,
                success: function (response) {
                    response = Ext.decode(response.responseText);

                    if (!(response && response.success)) {
                        this.getEl().unmask();
                        pimcore.helpers.showNotification(t("error"), t(response.errorMessage), "error");

                        cb(new Error(t(response.errorMessage)));
                    } else {
                        this.populateDataportForm();
                        cb();
                    }
                }.bind(this),
                failure: function (response) {
                    this.getEl().unmask();
                    response = Ext.decode(response.responseText);

                    pimcore.helpers.showNotification(t("error"), t(response.errorMessage), "error");
                    cb(new Error(t(response.errorMessage)));
                }.bind(this)
            });
        }
    },

    generate: function (cb) {
        if (!Ext.isFunction(cb)) {
            cb = function () {
            };
        }
        Ext.Ajax.request({
            url: "/admin/BlackbitPim/variantgenerator/generate",
            method: 'post',
            params: {id: this.dataport.id},
            success: function (response) {
                response = Ext.decode(response.responseText);

                if (!(response && response.success)) {
                    this.getEl().unmask();
                    pimcore.helpers.showNotification(t("error"), t(response.errorMessage), "error");

                    cb(new Error(t(response.errorMessage)));
                } else {
                    cb();
                }
            }.bind(this),
            failure: function (response) {
                this.getEl().unmask();
                response = Ext.decode(response.responseText);

                pimcore.helpers.showNotification(t("error"), t(response.errorMessage), "error");
                cb(new Error(t(response.errorMessage)));
            }.bind(this)
        });
    }
});
