<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\Item;

use Blackbit\PimBundle\Controller\ImportconfigController;
use Blackbit\PimBundle\lib\Pim\Hooks\Helper;
use Blackbit\PimBundle\lib\Pim\Import\Javascript;
use Blackbit\PimBundle\model\Fieldmapping;
use Blackbit\PimBundle\model\RawItemData;
use Blackbit\PimBundle\model\RawItemField;
use Pimcore\File;
use Pimcore\Model\Asset;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\ClassDefinition;
use Pimcore\Model\DataObject\ClassDefinition\Data;
use Pimcore\Model\DataObject\ClassDefinition\Data\Input;
use Pimcore\Model\DataObject\ClassDefinition\Data\Localizedfields;
use Pimcore\Model\DataObject\Classificationstore;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\DataObject\Data\ElementMetadata;
use Pimcore\Model\DataObject\Data\Geopoint;
use Pimcore\Model\DataObject\Data\Hotspotimage;
use Pimcore\Model\DataObject\Data\ImageGallery;
use Pimcore\Model\DataObject\Data\ObjectMetadata;
use Pimcore\Model\DataObject\Fieldcollection;
use Pimcore\Model\DataObject\Listing;
use Pimcore\Model\DataObject\Objectbrick\Definition;
use Pimcore\Model\DataObject\Service;
use Pimcore\Model\Element\ElementInterface;
use Pimcore\Model\Element\ValidationException;
use Pimcore\Model\Version;
use Pimcore\Tool;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\GenericEvent;

/**
 * Importiert Rohdaten aus der DB in Artikel- und Produktobjekte
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Medien GmbH, http://www.blackbit.de/
 */
class Importer implements ImporterInterface {
    const HASH_PROP_PREFIX = 'importhash_';

    /** @var LoggerInterface  */
    private $logger;

    protected $dataport;
    protected $rawItems;
    protected $groupMapping;

    protected $hooks;

    protected $assetFolder;
    protected $assetSource;

    private $changedObjectIds = array();

    private $idPrefix = '';
    private $ignoreHashCheck = false;

    public function __construct(LoggerInterface $logger) {
        $this->hooks = new Helper();
        $this->logger = $logger;
    }

    /**
     * @param bool $ignoreHashCheck
     */
    public function setIgnoreHashCheck($ignoreHashCheck)
    {
        $this->ignoreHashCheck = (bool)$ignoreHashCheck;
    }

    /**
     * Importiert alle vorhandenen Rohdatensätze in die dazugehörigen Artikel und Produkte. Nicht vorhandene Artikel
     * und Produkte werden angelegt.
     */
    public function import($dataport, $rawItems, $groupMappings) {
        $this->dataport = $dataport;
        $this->rawItems = $rawItems;
        $this->groupMapping = $groupMappings;

        $sourceconfig = unserialize($this->dataport['sourceconfig']);
        $targetconfig = unserialize($this->dataport['targetconfig']);
        if (array_key_exists('idPrefix', $targetconfig) && !empty($targetconfig['idPrefix'])) {
            $this->idPrefix = $targetconfig['idPrefix'];
        }

        $itemClassId = $targetconfig['itemClass'];
        $itemFolderPath = $targetconfig['itemFolder'];

        $assetFolder = Asset::getByPath($targetconfig['assetFolder']);
        if (!($assetFolder instanceof Asset\Folder)) {
            $assetFolder = Asset::getById(1);
        }
        $this->assetFolder = $assetFolder;

        $assetSource = $sourceconfig['assetSource'];
        $this->assetSource = $assetSource;


        $class = ClassDefinition::getById($itemClassId);
        if (!($class instanceof ClassDefinition)) {
            $this->logger->error('No class found for id '. $itemClassId);
            return;
        }

        $itemFolder = AbstractObject::getByPath($itemFolderPath);
        if (!($itemFolder instanceof AbstractObject)) {
            $itemFolder = AbstractObject::getById(1);
        }

        $itemClassName = $class->getName();

        $itemClassPath = 'Pimcore\\Model\\DataObject\\' . ucfirst($itemClassName);
        $itemListClassName = $itemClassPath . '\\Listing';

        $rawItemDataTable = new RawItemData();
        $fieldMappingTable = new Fieldmapping();

        $fieldTable = new RawItemField();
        $existingFields = $fieldTable->find(array('dataportId = ?' => $this->dataport['id']));

        $keyMapping = null;

        $mappings = $fieldMappingTable->find(array('dataportId = ?' => $this->dataport['id']));
        foreach ($mappings as $row) {
            if (!$keyMapping && !empty($row['keyMapping'])) {
                $keyMapping = $row;
                break;
            }
        }

        AbstractObject::setHideUnpublished(false);

        $rawItemDataRepository = new RawItemData();
        foreach ($this->rawItems as $rawItem) {
            try {
                $this->logger->info("Pim Plugin: Importing raw item {$rawItem['id']}");

                $event = new GenericEvent($rawItem, [
                    'dataportId' => $this->dataport['id'],
                    'shouldSkipItem' => false,
                ]);
                \Pimcore::getEventDispatcher()->dispatch('pim.beforeImport', $event);
                if ($event->getArgument('shouldSkipItem')) {
                    $this->logger->info('Skipping item "' . $rawItem['id'] . '" after evaluating results from event "pim.beforeImport"');
                    continue;
                }

                $rawItemDataRows = $rawItemDataRepository->find([
                    'rawItemId = ?' => $rawItem['id'],
                ]);
                $rawItemData = array();
                foreach ($rawItemDataRows as $row) {
                    $rawItemData['field_' . $row['fieldNo']] = $row;
                }

                // Mit Dummy-Daten auffüllen
                foreach ($existingFields as $field) {
                    if (!array_key_exists('field_' . $field['fieldNo'], $rawItemData)) {
                        $rawItemData['field_' . $field['fieldNo']] = array();
                    }
                }

                foreach ($rawItemDataRows as $row) {
                    foreach ($existingFields as $field) {
                        if($field['fieldNo'] == $row['fieldNo']) {
                            $rawItemData[$field['name']] = $row;
                            continue 2;
                        }
                    }
                }

                $newObjectCreated = false;

                if($keyMapping['fieldNo'] > 0) {
                    // Nach existierenden Artikeln suchen: Verbindung über das Schlüsselattribut herstellen
                    $keyField = $rawItemDataTable->findOne([
                        'rawItemId = ?' => $rawItem['id'],
                        'fieldNo = ?' => $keyMapping['fieldNo'],
                    ]);
                    $keyValue = $keyField['value'];
                } else {
                    $keyValue = implode('_', array_map(function($item) {
                        return (string)$item['value'];
                    }, $rawItemData));
                }

                // Ggf. unserialize
                if (isset($sourceconfig['fields']['field_' . $keyMapping['fieldNo']]['multiValues']) && $sourceconfig['fields']['field_' . $keyMapping['fieldNo']]['multiValues'] === true) {
                    $keyValue = unserialize($keyValue);
                }

                $keyValues = (array)$keyValue;

                foreach($keyValues as $keyIndex => $keyValue) {
                    if (!empty($keyMapping['calculation']) && Javascript::isEngineAvailable($targetconfig['javascriptEngine'])) {
                        $keyValue = Javascript::evaluateScript(
                            $keyMapping['calculation'], $targetconfig['javascriptEngine'],
                            [
                                'rawItemData' => $rawItemData,
                                'value' => $keyValues, // must be $keyValues not $keyValue to allow same callback function for key calculation and setting field value
                                'keyIndex' => $keyIndex,
                            ]
                        );

                        $this->logger->debug('Key column ('.$keyMapping['fieldName'].') callback return: '.\var_export($keyValue, true));
                    }

                    $def = new Input();
                    $def->setName($keyMapping['fieldName']);
                    $keyValue = $this->map($keyMapping, $keyValue, null, $def);

                    $item = null;
                    if ($keyMapping !== null) {
                        try {
                            // Items heraussuchen, bei denen das Schlüsselattribut auf den Wert aus dem Rohdatensatz gesetzt ist
                            $column = $keyMapping['fieldName'];
                            if ($column === 'key') {
                                $column = 'o_key';
                                $keyValue = Service::getValidKey($keyValue, 'object');
                            } elseif ($column === 'path') {
                                $column = 'o_path';
                                $keyValue = $this->sanitizePathString($keyValue, $targetconfig);
                            }

                            $list = [];
                            if ($column === 'fullpath') {
                                $keyValue = $this->sanitizePathString($keyValue, $targetconfig);
                                $item = $itemClassPath::getByPath($keyValue);
                            } else {
                                /** @var Listing $list */
                                $list = new $itemListClassName();
                                $list->setUnpublished(true);
                                $list->setCondition("`$column` = ?", [$keyValue]);
                                $list->setObjectTypes([
                                    AbstractObject::OBJECT_TYPE_OBJECT,
                                    AbstractObject::OBJECT_TYPE_FOLDER,
                                    AbstractObject::OBJECT_TYPE_VARIANT,
                                ]);
                            }

                            if (!($item instanceof $itemClassPath)) {
                                $count = \count($list);
                                if ($count === 1) {
                                    $item = $list->current();
                                } elseif ($count > 1) {
                                    $this->logger->error(
                                        "Getting items by key field '{$keyMapping['fieldName']}' with value " .
                                        "'{$keyValue}' does not return a unique result"
                                    );
                                    continue;
                                }
                            }
                        } catch (\Exception $e) {
                            $this->logger->error(
                                "Unable to get item by key attribute '{$keyMapping['fieldName']}', value '{$keyValue}'"
                            );
                        }
                    }

                    // Neuen Artikel anlegen, wenn kein bestehender gefunden wurde.
                    if ($item === null) {
                        if ($this->hooks->shouldCreateItem($this->dataport['id'], $rawItemData)
                            && ($targetconfig['mode'] & ImportconfigController::MODE_CREATE)
                            && !empty($keyValue)) {

                            $item = $this->createObject(
                                $itemClassName, $itemClassId, $itemFolder, $this->getIdPrefix().$keyValue
                            );
                            $newObjectCreated = true;
                        } else {
                            // Rohdatensatz überspringen, wenn kein Objekt angelegt werden soll
                            $this->logger->info("Skipped item {$rawItem['id']}");
                            continue;
                        }
                    }

                    $event = new GenericEvent(
                        $item, [
                            'shouldSkipItem' => false,
                            'dataportId'     => $this->dataport['id'],
                            'rawItem'        => $rawItem,
                            'rawItemData'    => $rawItemData,
                            'mappings'       => $mappings,
                            'sourceconfig'   => $sourceconfig,
                            'assetFolder'    => $assetFolder,
                            'keyIndex'       => $keyIndex,
                        ]
                    );
                    \Pimcore::getEventDispatcher()->dispatch('pim.afterFindItem', $event);

                    if ($event->getArgument('shouldSkipItem')) {
                        $this->logger->info(
                            'Skipping item "'.$rawItem['id'].'" after evaluating results from event "pim.afterFindItem"'
                        );
                        continue;
                    }

                    // Abbrechen, falls kein Artikel angelegt werden konnte oder kein befülltes Schlüsselattribut gefunden wurde
                    if (!$item instanceof Concrete) {
                        $this->logger->info("No item could be created for raw item {$rawItem['id']}");
                        continue;
                    }

                    if ($this->hooks->shouldDeleteItem($this->dataport['id'], $rawItemData)) {
                        $item->delete();
                        continue;
                    }

                    // Skip item if the rawdata has not changed since the last import
                    if (!$this->ignoreHashCheck && !$this->hasRawdataChanged($this->dataport['id'], $rawItem, $item)) {
                        $this->logger->info(
                            'Skipping '.$item->getFullPath().' because object property "'.self::HASH_PROP_PREFIX
                            .$this->dataport['id'].'" matches hash of raw data values'
                        );
                        continue;
                    }
                    $item->setProperty(self::HASH_PROP_PREFIX.$this->dataport['id'], 'text', $rawItem['hash']);

                    // Daten zuweisen
                    $currentObjectValues = $this->getAttributesArrayForObject($item);
                    if ($newObjectCreated || ($targetconfig['mode'] & ImportconfigController::MODE_EDIT)) {
                        $event = new GenericEvent(
                            $item, [
                                'dataportId'          => $this->dataport['id'],
                                'rawItem'             => $rawItem,
                                'rawItemData'         => $rawItemData,
                                'mappings'            => $mappings,
                                'sourceconfigFields'  => $sourceconfig['fields'],
                                'assetFolder'         => $assetFolder,
                                'currentObjectValues' => $currentObjectValues,
                                'keyIndex'            => $keyIndex,
                            ]
                        );

                        \Pimcore::getEventDispatcher()->dispatch('pim.beforeMapData', $event);
                        $rawItemData = $event->getArgument('rawItemData');

                        $this->mapData(
                            $item, $rawItem, $rawItemData, $mappings, $sourceconfig['fields'], $currentObjectValues,
                            $targetconfig, $keyIndex
                        );

                        $event = new GenericEvent(
                            $item, [
                                'dataportId'          => $this->dataport['id'],
                                'rawItem'             => $rawItem,
                                'rawItemData'         => $rawItemData,
                                'mappings'            => $mappings,
                                'sourceconfigFields'  => $sourceconfig['fields'],
                                'assetFolder'         => $assetFolder,
                                'currentObjectValues' => $currentObjectValues,
                            ]
                        );

                        \Pimcore::getEventDispatcher()->dispatch('pim.afterMapData', $event);
                        $rawItemData = $event->getArgument('rawItemData');
                    }

                    $purgeIds = [];
                    if (is_array($this->groupMapping) && count($this->groupMapping) > 0) {
                        $targetIds = array();

                        foreach ($this->groupMapping as $groupMapping) {
                            $matches = true;
                            foreach ($groupMapping['conditions'] as $field => $value) {
                                $matches = array_key_exists($field, $rawItemData)
                                    && trim($rawItemData[$field]['value']) === trim($value);
                                if (!$matches) {
                                    break;
                                }
                            }

                            // Zuordnung vornehmen
                            $targets = $groupMapping['target'];
                            foreach ($targets as $target) {
                                if ($matches) {
                                    $targetIds[] = $target['objectId'];
                                } elseif ($target['purge'] == 1) {
                                    $purgeIds[] = $target['objectId'];
                                }
                            }
                        }

                        $targetIds = array_unique($targetIds);

                        // TODO Object_Category und editable aus der dataport config auslesen, sobald es da drin ist
                        $categoryClass = $targetconfig['categoryClass'];
                        $editable = $targetconfig['fieldnameProducts'];


                        // Iteriere über die Dependecies des Items und
                        // 1. entferne die Kategorie aus $targetIds falls sie darin steht
                        // 2. lösche es aus den Kategorien raus die in $purgeIds drin stehen
                        foreach ($item->getDependencies()->getRequiredBy() as $dep) {
                            if ($dep['type'] === 'object') {
                                if (in_array($dep['id'], $targetIds)) {
                                    $key = array_search($dep['id'], $targetIds);
                                    unset($targetIds[$key]);
                                    $targetIds = array_values($targetIds);
                                } elseif (in_array($dep['id'], $purgeIds)) {
                                    /** @var Concrete $depCategory */
                                    $depCategory = $categoryClass::getById($dep['id']);
                                    if ($depCategory != null) {
                                        $getter = 'get'.ucfirst($editable);
                                        $setter = 'set'.ucfirst($editable);
                                        $elements = $depCategory->$getter();
                                        foreach ($elements as $key => $element) {
                                            if ($element->getId() == $item->getId()) {
                                                unset($elements[$key]);
                                                break;
                                            }
                                        }
                                        $elements = array_values($elements);
                                        $depCategory->$setter($elements);
                                        $depCategory->setUserModification(0);
                                        $depCategory->save();
                                        $this->markObjectAsChanged($depCategory->getId());
                                    }
                                }
                            }
                        }


                        // Iteriere über $targetIds und füge das Item in die Kategorie ein
                        foreach ($targetIds as $targetId) {
                            $object = Concrete::getById($targetId);
                            if (!$object || \get_class($object) !== $categoryClass) {
                                continue;
                            }

                            $getter = 'get'.ucfirst($editable);
                            $setter = 'set'.ucfirst($editable);

                            $elements = $object->$getter();
                            $elements[] = $item;
                            $object->$setter($elements);
                            $object->setUserModification(0);
                            $object->save();

                            $this->markObjectAsChanged($object->getId());
                        }
                    }


                    // Speichern
                    $item->setOmitMandatoryCheck(true);

                    $event = new GenericEvent(
                        $rawItem, [
                            'dataportId'         => $this->dataport['id'],
                            'rawItem'            => $rawItem,
                            'rawItemData'        => $rawItemData,
                            'mappings'           => $mappings,
                            'sourceconfigFields' => $sourceconfig['fields'],
                            'assetFolder'        => $assetFolder,
                            'saveAnyway'         => false,
                        ]
                    );
                    \Pimcore::getEventDispatcher()->dispatch('pim.beforeSave', $event);

                    $oldItem = null;
                    try {
                        $latestVersion = $item->getLatestVersion(true);
                        if ($latestVersion instanceof Version) {
                            $oldItem = $latestVersion->getData();
                            if (!$oldItem instanceof $item) {
                                // This could be the case if unserialization goes wrong (and $oldItem is an __PHP_Incomplete_Class_Name)
                                $oldItem = null;
                            }
                        }
                    } catch (\Exception $e) {
                        $this->logger->error('Unable to load latest version for object');
                    }

                    $fieldsToTest = $this->hooks->fieldsToTestForIsModified($item);
                    if ($event->getArgument('saveAnyway') || $this->isModified($item, $oldItem, $fieldsToTest) || $newObjectCreated) {
                        $item->setUserModification(0);

                        try {
                            $item->save();
                        } catch (\Exception $e) {
                            $additionalDetails = [];

                            if($e instanceof ValidationException) {
                                /** @var \Exception $subItem */
                                foreach($e->getSubItems() as $subItem) {
                                    $additionalDetails[] = $subItem->getMessage();
                                }
                            }

                            $this->logger->error(
                                'Object #'.$item->getId().' could not be saved. Reverted changes. Error: '.$e->getMessage().' '.\implode(', ', $additionalDetails)
                            );
                        }

                        $this->markObjectAsChanged($item->getId());
                    }
                }
            } catch (\Exception $ex) {
                $this->logger->error("Unable to import item: {$ex}");
            } catch (\Error $ex) {
                $this->logger->error("Unable to import item: {$ex}");
            }
        }

        \Pimcore::collectGarbage();
    }


    protected function getAttributesArrayForObject(Concrete $object){
        $attributes = [
            [
               'id' => $object->getId(),
               'key'  => $object->getKey(),
               'path' => $object->getPath(),
            ],
        ];
        foreach ($object->getClass()->getFieldDefinitions() as $fieldDefinition) {
            $attributes[] = $this->getSerializedObjectValue($object, $fieldDefinition);
        }

        $attributes = \array_merge(...$attributes);

        return $attributes;
    }

    private function getSerializedObjectValue(Concrete $object, Data $field, array $getterArguments = []) {
        $getter = 'get'.ucfirst($field->getName());
        $value = \call_user_func_array([$object, $getter], $getterArguments);

        $indexAppendix = '';
        if($getterArguments) {
            // weird idea to store field and getter arguments concatenated as array index - but to keep backwards compatibility this behaviour is kept
            $indexAppendix = '#'.\implode('-', $getterArguments);
        }
        $indexName = $field->getName().$indexAppendix;
        $attributes = [];

        // if anything else needs to be evaluated, please add the definition and how to serialize it
        if ($field instanceof ClassDefinition\Data\Multiselect){
            $attributes[$indexName] = '';
            if (\is_array($value)) {
                $attributes[$indexName] = implode(';', $value);
            }
        } elseif ($field instanceof ClassDefinition\Data\Image){
            $attributes[$indexName] = null;
            if ($value instanceof \Pimcore\Model\Asset) {
                $attributes[$indexName] = $value->getFullPath();
            }
        } elseif ($field instanceof ClassDefinition\Data\Datetime || $field instanceof ClassDefinition\Data\Date){
            $attributes[$indexName] = null;
            if ($value instanceof \DateTimeInterface) {
                $attributes[$indexName] = $value->format('U');
            }
        } elseif ($field instanceof ClassDefinition\Data\Localizedfields) {
            $validLanguages = Tool::getValidLanguages();
            foreach ($validLanguages as $languageCode) {
                foreach ($field->getFieldDefinitions() as $localizedFieldDefinition) {
                    $attributes[] = $this->getSerializedObjectValue($object, $localizedFieldDefinition, [$languageCode]);
                }
            }
            $attributes = \array_merge(...$attributes);
        } elseif(\is_scalar($value)) {
            $attributes[$indexName] = $value;
        }

        return $attributes;
    }


    public function mapData(\Pimcore\Model\DataObject\AbstractObject $target, $rawItem, $rawItemData, $mappings, $sourceconfigFields, $currentObjectValues, $targetConfig, $keyIndex) {
        $javascriptEngine = $targetConfig['javascriptEngine'];
        $originalTarget = $target;

        foreach ($mappings as $mapping) {
            $target = $originalTarget;
            try {
                if (!empty($mapping['targetBrickField']) && !empty($mapping['brickName'])){
                    $brickfieldGetter = 'get'.ucfirst($mapping['targetBrickField']);
                    $brickGetter =  'get'.ucfirst($mapping['brickName']);
                    if (!method_exists($target, $brickfieldGetter)) {
                        $this->logger->warning('Invalid mapping to non-existent brickfield "' . $mapping['targetBrickField'] . '"');
                        continue;
                    }
                    $brickField =  $target->$brickfieldGetter();

                    if (!$brickField instanceof \Pimcore\Model\DataObject\Objectbrick){
                        $this->logger->warning('"' . $mapping['targetBrickField'] . '" is not a brick field.');
                        continue;
                    }
                    if (!method_exists($brickField, $brickGetter)) {
                        $this->logger->warning('"' . $mapping['brickName'] . '" is not a valid brick in '.$mapping['targetBrickField'].'.');
                        continue;
                    }
                    $brick = $brickField->$brickGetter();

                    if (!$brick instanceof \Pimcore\Model\DataObject\Objectbrick\Data\AbstractData){
                        if ($this->hooks->shouldCreateBrickByRawdata($this->dataport['id'], $mapping, $rawItemData, $target)) {
                            $newBrickClass = "\\Pimcore\\Model\\DataObject\\Objectbrick\\Data\\" . ucfirst($mapping['brickName']);
                            /** @var \Pimcore\Model\DataObject\Objectbrick\Data\AbstractData $brick */
                            $brick = new $newBrickClass($target);
                            $brick->setFieldname($brickField->getFieldname());
                        }
                        else {
                            continue;
                        }
                    }
                    $target = $brick;
                }

                $getter = 'get' . ucfirst($mapping['fieldName']);

                // Check if target property exists
                if (!method_exists($target, $getter)) {
                    $this->logger->warning('Invalid mapping to non-existent field "' . $mapping['fieldName'] . '"');
                    continue;
                }


                $key = null;

                if ($mapping['fieldNo'] != null) {
                    $key = "field_".$mapping['fieldNo'];
                }

                $value = '';
                if ($key !== null && array_key_exists($key, $rawItemData)) {
                    $value = $rawItemData[$key]['value'];
                }
                if ($value === null) {
                    $value = '';
                }


                // Ggf. unserialize
                if(isset($sourceconfigFields['field_' . $mapping['fieldNo']]['multiValues']) && $sourceconfigFields['field_' . $mapping['fieldNo']]['multiValues'] === true) {
                    $value = unserialize($value);
                }

                $currentValue = null;
                try {
                    $method = new \ReflectionMethod($target, $getter);
                    $params = $method->getParameters();

                    if ($method->getNumberOfParameters() >= 1 && $params[0]->name === "language") {
                        $currentValue = $target->$getter($mapping['locale']);
                    } else {
                        $currentValue = $target->$getter();
                    }
                } catch (\Exception $e) {
                    $this->logger->warning("Unable to get value for field '{$mapping['fieldName']}'. " . $e->getMessage());
                }

                if(!empty($mapping['format'])) {
                    $format = unserialize($mapping['format']);
                    if(!empty($currentValue) && !empty($format['writeProtected'])) {
                        $this->logger->info('Field "'.$mapping['fieldName']. '" is populated and configured to be write-protected -> won\'t change value');
                        continue;
                    }
                }

                if ($target instanceof \Pimcore\Model\DataObject\Objectbrick\Data\AbstractData){
                    $def = $target->getDefinition()->getFieldDefinition($mapping['fieldName']);
                } else {
                    /** @var Concrete $target */
                    $def = $target->getClass()->getFieldDefinition($mapping['fieldName']);
                    /** @var ClassDefinition\Data\Localizedfields $localized */
                    $localized = $target->getClass()->getFieldDefinition('localizedfields');
                    if (!$def && $localized) {
                        $def = $localized->getFieldDefinition($mapping['fieldName']);
                    }
                }

                if (\in_array($mapping['fieldName'], ['path', 'key'])) {
                    $def = new Input();
                    $def->setName($mapping['fieldName']);
                }

                if($mapping['fieldName'] === 'published') {
                    $def = new Data\Checkbox();
                }

                if (!$def instanceof \Pimcore\Model\DataObject\ClassDefinition\Data) {
                    $this->logger->error('Invalid mapping for field ' . $mapping['fieldName'] . ': No valid target found');
                    continue;
                }

                // Berechnungen ausführen
                if (!empty($mapping['calculation']) && Javascript::isEngineAvailable($javascriptEngine)) {
                    try {
                        $jsParams = [
                            'rawItem' => $rawItem,
                            'rawItemData' => $rawItemData,
                            'value' => $value,
                            'currentValue' => $currentValue,
                            'currentObjectData' => $currentObjectValues,
                            'currentObject' => $target,
                            'keyIndex' => $keyIndex,
                        ];

                        if (!is_scalar($currentValue)) {
                            $jsParams['currentValue'] = (array) \json_decode(\json_encode($currentValue), true);
                        }

                        $value = Javascript::evaluateScript($mapping['calculation'], $javascriptEngine, $jsParams);
                        if (is_object($value)){
                            $value = self::objectToArray($value);
                        }

                        $this->logger->debug('Callback return for field ' . $mapping['fieldName'] . ': '.\var_export($value, true));
                    } catch (\Exception $ex) {
                        $this->logger->error("Error while calculating attribute value for field '{$mapping['fieldName']}': {$ex}");
                    }
                }


                // Konvertierung
                $value = $this->map($mapping, $value, $currentValue, $def, $target);

                // Process value for field key to get a valid filename
                if($mapping['fieldName'] === 'key') {
                    $value = Service::getValidKey($value, 'object');
                    if (empty($value)) {
                        continue;
                    }
                } elseif ($mapping['fieldName'] === 'path') {
                    if(empty($value)) {
                        $parentObject = Service::createFolderByPath($targetConfig['itemFolder']);
                        $value = $targetConfig['itemFolder'];
                    } else {
                        $parentObject = \Pimcore\Model\DataObject\AbstractObject::getByPath($value);
                        if (!$parentObject instanceof \Pimcore\Model\DataObject\AbstractObject) {
                            $folderPath = (0 === strpos($value, $targetConfig['itemFolder'])) ? $value : $targetConfig['itemFolder'] . '/' . $value;
                            $parentObject = Service::createFolderByPath($folderPath);
                        }
                    }

                    $target->setParent($parentObject);
                    // if parent is of the same type, set to type variant
                    if ($parentObject->getType() === AbstractObject::OBJECT_TYPE_OBJECT
                        && true === ClassDefinition::getByName($parentObject->getClassName())->allowVariants
                    ) {
                        $target->setType(AbstractObject::OBJECT_TYPE_VARIANT);
                    }
                }

                if ($target instanceof \Pimcore\Model\DataObject\Objectbrick\Data\AbstractData){
                    if (is_array($value)){
                        $testvalue =  array_filter($value);
                    } elseif(\is_object($value)) {
                        $classReflection = new \ReflectionClass($target);
                        $properties = $classReflection->getProperties();

                        $testvalue = false;
                        foreach($properties as $property) {
                            if($property->getName() !== 'type' && $property->class === \get_class($target) && $property->getValue($target)) {
                                $testvalue = true;
                                break;
                            }
                        }
                    } else {
                        $testvalue = $value;
                    }

                    $brickGetter = 'get' . ucfirst($mapping['brickName']);
                    if ( !empty($testvalue) || $brickField->$brickGetter() instanceof \Pimcore\Model\DataObject\Objectbrick\Data\AbstractData ){
                        $brickSetter = 'set' . ucfirst($mapping['brickName']);
                        $brickField->$brickSetter($target);
                    } else {
                        continue;
                    }
                }

                $setter = 'set' . ucfirst($mapping['fieldName']);
                try {
                    $method = new \ReflectionMethod($target, $setter);
                    $params = $method->getParameters();

                    if ($method->getNumberOfParameters() > 1 && $params[1]->name === "language") {
                        $target->$setter($value, $mapping['locale']);
                    } else {
                        $target->$setter($value);
                    }
                } catch (\Exception $e) {
                    $this->logger->warning("Unable to set value for field '{$mapping['fieldName']}. Method '{$setter}' is not defined");
                }

            } catch (\Exception $e) {
                $klass = get_class($target);
                $this->logger->warning("Error while mapping value of field '{$mapping['fieldNo']}', raw item '{$rawItem['id']}',
						 target ({$klass}): {$e}");
            }
        }
    }

    /**
     * Applies mapping to given value and returns the result
     * @param array $mapping
     * @param mixed $rawValue
     * @param mixed $currentValue
     * @param ClassDefinition\Data $fieldDefinition
     * @param \Pimcore\Model\DataObject\Concrete|\Pimcore\Model\DataObject\Objectbrick\Data\AbstractData $dataObject
     * @return mixed the final value, created by transforming $rawValue using $mapping
     */
    public function map($mapping, $rawValue, $currentValue, ClassDefinition\Data $fieldDefinition, $dataObject = null) {
        $value = $rawValue;

        // In den Zieldatentyp umwandeln
        $format = [];

        if (!empty($mapping['format'])) {
            $format = unserialize($mapping['format']);
        }

        switch ($fieldDefinition->getFieldtype()) {
            case 'checkbox':
                $value = (bool)$value;
                break;
            case 'numeric':
                $decimalSep = null;
                $groupingSep = null;

                if ($format) {
                    if (array_key_exists("decimalSeparator", $format)) {
                        $decimalSep = $format['decimalSeparator'];
                    }

                    if (array_key_exists("groupingSeparator", $format)) {
                        $groupingSep = $format['groupingSeparator'];
                    }
                }
                if ($value !== ''){
                    $value = $this->parseNumber($value, $decimalSep, $groupingSep);
                }
                break;

            case 'date':
            case 'datetime':
                if ($format && array_key_exists("dateFormat", $format)) {
                    $date = \DateTime::createFromFormat($format['dateFormat'], $value);

                    if ($date) {
                        if ($fieldDefinition->getFieldtype() === "date") {
                            $date->setTime(0, 0);
                        }

                        $value = $date->getTimestamp();
                    } else {
                        $value = strtotime($value);
                    }
                } else {
                    $value = strtotime($value);
                }

                if(empty($value)) {
                    $value = null;
                } else {
                    $value = new \DateTime('@'.(int)$value);
                }

                break;

            case 'multiselect':
                $separator = null;
                if (array_key_exists("separator", $format)) {
                    $separator = $format['separator'];
                }

                if ($separator) {
                    $value = explode($separator, $value);
                } else {
                    $value = (array) $value;
                }

                break;

            case 'image':
                $assetSource = $value;
                if (!$currentValue instanceof Asset\Image || empty($format['writeProtected'])) {
                    $value = $this->getAsset($assetSource, ['overwrite' => !empty($format['overwrite'])]);
                }

                if(!$value instanceof Asset\Image) {
                    $value = $currentValue;
                }
                break;

            case 'hotspotimage':
                $assetSource = $value;
                $asset = null;
                if (!$currentValue instanceof Asset\Image || empty($format['writeProtected'])) {
                    $asset = $this->getAsset($assetSource, ['overwrite' => !empty($format['overwrite'])]);
                }

                if(!$asset instanceof Asset\Image) {
                    $value = $currentValue;
                } else {
                    $value = new Hotspotimage($asset);
                }

                break;
            case 'quantityValue':
                $tmpValue = $value;
                $value = new \Pimcore\Model\DataObject\Data\QuantityValue();
                $value->setValue($tmpValue[0]);

                try {
                    $unit = \Pimcore\Model\DataObject\QuantityValue\Unit::getByAbbreviation($tmpValue[1]);
                    $value->setUnitId($unit->getId());
                } catch (\Exception $e) {
                    $this->logger->error("Cannot find unit with abbreviation '" . $tmpValue[1] . "': " . $e->getMessage());
                }
                break;
            case 'fieldcollections':
                /** @var Fieldcollection $currentValue */

                $purgeitems = false;
                if (array_key_exists('purgeitems', $format)) {
                    $purgeitems = $format['purgeitems'] === true;
                }

                $fieldCollection = null;
                if($purgeitems === false) {
                    $fieldCollection = $currentValue;
                }

                if(!$fieldCollection instanceof Fieldcollection) {
                    $fieldCollection = new Fieldcollection();
                }

                $allowedTypes = $fieldDefinition->getAllowedTypes();

                foreach($allowedTypes as $fieldCollectionType) {
                    foreach((array)$value as $itemData) {
                        $fieldCollectionClass = '\Pimcore\\Model\\DataObject\\Fieldcollection\\Data\\'.ucfirst($fieldCollectionType);
                        /** @var Fieldcollection\Data\AbstractData $fieldCollectionItem */
                        $fieldCollectionItem = new $fieldCollectionClass();
                        /** @var Fieldcollection\Definition $fieldCollectionItemDefinition */
                        $fieldCollectionItemDefinition = $fieldCollectionItem->getDefinition();

                        foreach ($itemData as $field => $fieldValues) {
                            $setter = 'set' . ucfirst($field);
                            if (!method_exists($fieldCollectionItem, $setter)) {
                                continue 2;
                            }

                            $fieldCollectionItemFieldDefinition = $fieldCollectionItemDefinition->getFieldDefinition($field);
                            if(!$fieldCollectionItemFieldDefinition instanceof Data) {
                                $fieldCollectionItemFieldDefinition = $fieldCollectionItemDefinition->getFieldDefinition('localizedfields')->getFieldDefinition($field);
                            }

                            $getter = 'get' . ucfirst($field);

                            foreach((array)$fieldValues as $fieldValue) {
                                $fieldValue = (array)$fieldValue;
                                $setValue = $fieldValue[0];
                                $setterArguments = \array_slice($fieldValue, 1);

                                $fieldValue = $this->map($mapping, $setValue, \call_user_func_array([$fieldCollectionItem, $getter], $setterArguments), $fieldCollectionItemFieldDefinition);
                                \array_unshift($setterArguments, $fieldValue);
                                \call_user_func_array([$fieldCollectionItem, $setter], $setterArguments);
                            }
                        }

                        // CHECK ALL EXISTING FIELDCOLLECTION ITEMS IF EQUAL AND ONLY IF NOT EQUAL ADD NEW FIELDCOLLECTION ITEM
                        foreach ($fieldCollection->getItems() as $existingFieldCollectionItem) {
                            if(\get_class($existingFieldCollectionItem) !== \get_class($fieldCollectionItem)) {
                                continue;
                            }

                            $fieldCollection1 = new Fieldcollection;
                            $fieldCollection1->add($fieldCollectionItem);

                            $fieldCollection2 = new Fieldcollection;
                            $fieldCollection2->add($existingFieldCollectionItem);

                            if($fieldDefinition->getDiffVersionPreview($fieldCollection1) !== $fieldDefinition->getDiffVersionPreview($fieldCollection2)) {
                                continue;
                            }

                            continue 2;
                        }

                        $fieldCollection->add($fieldCollectionItem);
                    }
                }

                $value = $fieldCollection;
                break;
            case 'nonownerobjects':
            case 'reverseManyToManyObjectRelation':
            case 'objects':
            case 'manyToManyObjectRelation':
            case 'manyToManyRelation':
                $value = (array)$value;

                $purgeitems = false;
                if (array_key_exists('purgeitems', $format)) {
                    $purgeitems = $format['purgeitems'] === true;
                }

                $objects = [];
                if ($purgeitems === false) {
                    $objects = $currentValue;
                }

                foreach ($value as $objectIdentifier) {
                    if (!\is_string($objectIdentifier)) {
                        $objects = $value;
                        break;
                    }
                    $parts = explode(':', $objectIdentifier);
                    if (count($parts) < 3) {
                        continue;
                    }

                    $type = $parts[0];
                    if (!class_exists($type)) {
                        $base = 'manyToManyRelation' === $fieldDefinition->getFieldtype() ? '\Pimcore\\Model\\'
                            : '\Pimcore\\Model\\DataObject\\';
                        $type = $base . ucfirst($type);
                    }
                    $method = 'getBy' . ucfirst($parts[1]);

                    try {
                        $arg1 = $parts[2];
                        $arg2 = 1;
                        $arg3 = '';
                        $arg4 = '';
                        // overwrite arguments in case of localized field handling
                        if (count($parts) > 3) {
                            $arg2 = $parts[3];
                            $arg3 = (isset($parts[4])) ? $parts[4] : 'default';
                            $arg4 = (isset($parts[5])) ? $parts[5] : 1;
                        }

                        $targetObject = $type::$method($arg1, $arg2, $arg3, $arg4);
                    } catch (\Exception $e) {
                        $this->logger->debug(
                            $e->getMessage() . ' Searched for ' . $type . ' with ' . $parts[1] . '="' . $parts[2] . '"'
                        );
                    }

                    if ($targetObject instanceof $type) {
                        $objects[] = $targetObject;
                    }
                }
                $value = $objects;

                break;
            case 'objectsMetadata':
            case 'advancedManyToManyObjectRelation':
                $value = (array)$value;

                $purgeitems = false;
                if (array_key_exists('purgeitems', $format)) {
                    $purgeitems = $format['purgeitems'] === true;
                }

                $objects = [];
                if($purgeitems === false) {
                    $objects = $currentValue;
                }

                foreach($value as $objectData) {
                    if(!is_array($objectData)) {
                        $objectData = [
                            'query' => $objectData,
                        ];
                    }
                    if(!\is_string($objectData['query'])) {
                        $objects = $value;
                        break;
                    }
                    $parts = explode(':', $objectData['query']);
                    if (count($parts) < 3) {
                        continue;
                    }

                    $type = $parts[0];
                    if(!class_exists($type)) {
                        $type = '\Pimcore\\Model\\DataObject\\'.ucfirst($type);
                    }

                    $getterMethod = 'get'.ucfirst($parts[1]);

                    $targetObject = null;
                    foreach($objects as $index => $existingMetaObject) {
                        $existingObject = $existingMetaObject->getObject();
                        if($existingObject instanceof $type && $existingObject->$getterMethod() == $parts[2]) {
                            $targetObject = $existingObject;
                            unset($objects[$index]);
                            break;
                        }
                    }

                    if($targetObject === null) {
                        $queryMethod = 'getBy'.ucfirst($parts[1]);
                        $targetObject = $type::$queryMethod($parts[2], 1);
                    }


                    if ($targetObject instanceof $type) {
                        $metaData = array();
                        foreach ($objectData as $key => $value) {
                            if ($key === 'query') {
                                continue;
                            }

                            if(\is_scalar($value)) {
                                $metaData[$key] = $value;
                            }
                        }

                        $metaObject = new ObjectMetadata($fieldDefinition->getName(), \array_keys($metaData), $targetObject);
                        foreach($metaData as $key => $value) {
                            $metaSetter = 'set'.ucfirst($key);
                            $metaObject->$metaSetter($value);
                        }
                        $objects[] = $metaObject;
                    }
                }
                $value = $objects;

                break;
            case 'multihref':
            case 'manyToManyObjectRelation':
                $purgeitems = false;
                if (array_key_exists('purgeitems', $format)) {
                    $purgeitems = $format['purgeitems'] === true;
                }

                $value = (array)$value;

                if (!is_array($currentValue)) {
                    $currentValue = array();
                }

                // Bestehende Einträge nicht übernehmen, wenn purgeItems gesetzt ist
                if ($purgeitems) {
                    $currentValue = array();
                }

                $overwrite = false;
                if ($format && array_key_exists("overwrite", $format)) {
                    $overwrite = $format['overwrite'] === true;
                }

                foreach ($value as $assetSource) {
                    $assetToUpdateId = null;

                    $newAsset = $this->getAsset($assetSource, ['overwrite' => $overwrite]);

                    if ($newAsset instanceof Asset) {
                        $assetSourceURI = $newAsset->getProperty('sourcePath');
                        if ($assetSourceURI) {
                            // Suche nach vorhandenen assets anhand des Attributes 'sourcePath'
                            foreach ($currentValue as $key => $currentAsset) {
                                $sourcePath = $currentAsset->getProperty('sourcePath');
                                if ($sourcePath === $assetSourceURI) {
                                    $assetToUpdateId = $key;
                                }
                            }
                        }

                        // assetSource array durchlaufen und alle Einträge außer 'url' als Metadaten setzten
                        $metaData = array();
                        if (is_array($assetSource)) {
                            foreach ($assetSource as $key => $value) {
                                if (\in_array($key, ['url','query'])) {
                                    continue;
                                }
                                if (is_array($value)) {
                                    foreach (Tool::getValidLanguages() as $language) {
                                        if (isset($value[$language])) {
                                            $md = [
                                                'name' => $key,
                                                'language' => $language,
                                                'data' => $value[$language],
                                            ];
                                            $metaData[] = $md;
                                        }
                                    }
                                } else {
                                    $md = [
                                        'name' => $key,
                                        'language' => '',
                                        'data' => $value,
                                    ];
                                    $metaData[] = $md;
                                }
                            }
                            $newAsset->setMetadata($metaData);
                        }

                        $newAsset->save();

                        if ($assetToUpdateId !== null) {
                            $currentValue[$assetToUpdateId] = $newAsset;
                        } else {
                            $currentValue[] = $newAsset;
                        }
                    } else {
                        $objectIdentifier = $assetSource;

                        $parts = explode(':', $objectIdentifier);
                        if (\count($parts) < 3) {
                            continue;
                        }

                        $type = $parts[0];
                        if(!class_exists($type)) {
                            $type = 'Pimcore\\Model\\DataObject\\'.ucfirst($type);
                        }
                        $method = 'getBy'.ucfirst($parts[1]);

                        try {
                            $targetObject = $type::$method($parts[2], 1);
                        } catch (\Exception $e) {
                            $this->logger->debug($e->getMessage().' Searched for '.$type.' with '.$parts[1].'="'.$parts[2].'"');
                        }

                        if ($targetObject instanceof $type) {
                            $objectToUpdateId = -1;

                            // Suche nach vorhandenen assets anhand des Attributes 'sourcePath'
                            foreach ($currentValue as $key => $existingObject) {
                                if ($existingObject->getId() == $targetObject->getId()) {
                                    $objectToUpdateId = $key;
                                }
                            }

                            if ($objectToUpdateId > -1) {
                                $currentValue[$objectToUpdateId] = $targetObject;
                            } else {
                                $currentValue[] = $targetObject;
                            }
                        }
                    }
                }

                $value = $currentValue;
                break;
            case 'objectbricks':
                $brickContainer = $currentValue;
                if(!$brickContainer instanceof Pimcore\Model\DataObject\Objectbrick) {
                    $brickContainer = new Pimcore\Model\DataObject\Objectbrick();
                }

                $allowedTypes = $brickContainer->getAllowedBrickTypes();
                foreach((array)$value as $brickName => $itemData) {
                    if(in_array($brickName, $allowedTypes)) {
                        $brickGetter = 'get'.ucfirst($brickName);
                        $brickItem = $brickContainer->$brickGetter();
                        if(!$brickItem instanceof Pimcore\Model\DataObject\Objectbrick\Data\AbstractData) {
                            $brickClass = 'Pimcore\\Model\\DataObject\\Objectbrick\\Data\\'.ucfirst($brickName);
                            $brickItem = new $brickClass($dataObject);
                            $addMethod = 'set'.ucfirst($brickName);
                            $brickContainer->$addMethod($brickItem);
                        }
                        foreach($itemData as $field => $fieldValue) {
                            $setter = 'set'.ucfirst($field);
                            if(!method_exists($brickItem, $setter)) {
                                continue;
                            }

                            \call_user_func_array([$brickItem, $setter], (array)$fieldValue);
                        }
                    }
                }

                $value = $brickContainer;
                break;
            case 'multihrefMetadata':
            case 'advancedManyToManyRelation':
                $purgeitems = false;
                if (array_key_exists('purgeitems', $format)) {
                    $purgeitems = $format['purgeitems'] === true;
                }

                $value = (array)$value;

                if (!is_array($currentValue)) {
                    $currentValue = array();
                }

                // Bestehende Einträge nicht übernehmen, wenn purgeItems gesetzt ist
                if ($purgeitems) {
                    $currentValue = array();
                }

                $overwrite = false;
                if ($format && array_key_exists("overwrite", $format)) {
                    $overwrite = $format['overwrite'] === true;
                }

                foreach ($value as $assetSource) {
                    $assetToUpdateId = null;

                    // assetSource array durchlaufen und alle Einträge außer 'url' als Metadaten setzen
                    $metaData = array();
                    foreach ($assetSource as $key => $value) {
                        if (\in_array($key, ['url', 'query', 'filename'])){
                            continue;
                        }

                        if(\is_scalar($value)) {
                            $metaData[$key] = $value;
                        }
                    }

                    $newAsset = $this->getAsset($assetSource, ['overwrite' => $overwrite]);
                    if ($newAsset instanceof Asset) {
                        $assetSourceURI = $newAsset->getProperty('sourcePath');
                        if ($assetSourceURI) {
                            // Suche nach vorhandenen assets anhand des Attributes 'sourcePath'
                            foreach ($currentValue as $key => $currentAssetMeta) {
                                $currentAsset = $currentAssetMeta->getElement();
                                $sourcePath = $currentAsset->getProperty('sourcePath');
                                if ($sourcePath === $assetSourceURI) {
                                    $assetToUpdateId = $key;
                                }
                            }
                        }

                        $newAsset->save();

                        $metaElement = new ElementMetadata($fieldDefinition->getName(), array_keys($metaData), $newAsset);

                        foreach($metaData as $key => $value) {
                            $metaSetter = 'set'.ucfirst($key);
                            $metaElement->$metaSetter($value);
                        }

                        if ($assetToUpdateId > -1) {
                            $currentValue[$assetToUpdateId] = $metaElement;
                        } else {
                            $currentValue[] = $metaElement;
                        }
                    } else {
                        $objectIdentifier = $assetSource['query'];

                        $parts = explode(':', $objectIdentifier);
                        if (\count($parts) < 3) {
                            continue;
                        }

                        $type = $parts[0];
                        if(!class_exists($type)) {
                            $type = 'Pimcore\\Model\\DataObject\\'.ucfirst($type);
                        }
                        $method = 'getBy'.ucfirst($parts[1]);

                        try {
                            $targetObject = $type::$method($parts[2], 1);
                        } catch (\Exception $e) {
                            $this->logger->debug($e->getMessage().' Searched for '.$type.' with '.$parts[1].'="'.$parts[2].'"');
                        }

                        if ($targetObject instanceof $type) {
                            $objectToUpdateId = -1;

                            // Suche nach vorhandenen assets anhand des Attributes 'sourcePath'
                            foreach ($currentValue as $key => $metaElement) {
                                $existingObject = $metaElement->getElement();
                                if ($existingObject->getId() == $targetObject->getId()) {
                                    $objectToUpdateId = $key;
                                }
                            }

                            $metaElement = new ObjectMetadata($fieldDefinition->getName(), array_keys($metaData), $targetObject);

                            foreach($metaData as $key => $value) {
                                $metaSetter = 'set'.ucfirst($key);
                                $metaElement->$metaSetter($value);
                            }

                            if ($objectToUpdateId > -1) {
                                $currentValue[$objectToUpdateId] = $metaElement;
                            } else {
                                $currentValue[] = $metaElement;
                            }
                        }
                    }
                }

                $value = $currentValue;
                break;
            case 'href':
            case 'manyToOneRelation':
                if(!\is_string($value)) {
                    break;
                }

                $parts = explode(':', $value);
                if(\count($parts) === 2) {
                    $parts = [
                        $parts[0],
                        'path',
                        $parts[1],
                    ];
                }

                $type = $parts[0];
                if(!class_exists($type)) {
                    $type = '\Pimcore\\Model\\DataObject\\'.ucfirst($type);
                }
                $method = 'getBy'.ucfirst($parts[1]);

                if($parts[1] === 'path') {
                    $objectPathArray = explode('/', $parts[2]);
                    array_walk($objectPathArray, function (&$value) {
                        $value = Service::getValidKey($value, 'object');
                    });
                    $parts[2] = implode('/', $objectPathArray);
                }

                try {
                    $targetObject = $type::$method($parts[2], 1);
                } catch (\Exception $e) {
                    $this->logger->debug($e->getMessage().' Searched for '.$type.' with '.$parts[1].'="'.$parts[2].'"');
                }

                if ($targetObject instanceof $type) {
                    $value = $targetObject;
                } else {
                    $value = null;
                }
                break;
            case 'structuredTable':
                $value = new \Pimcore\Model\DataObject\Data\StructuredTable($rawValue);
                break;

            case 'input':
            case 'textarea':
                if (!is_string($value) && $value !== null) {
                    $value = serialize($value);
                }
                break;
            case 'classificationstore':
                if(!is_array($value)) {
                    break;
                }

                // get store dynamic by classificationStoreType or create a new store by type if not set
                $store = $dataObject->{'get' . ucfirst($value['classificationStoreType'])}();
                if (!$store instanceof Classificationstore) {
                    $store = new Classificationstore();
                    $store->setFieldName($value['classificationStoreType']);
                }

                foreach($value['values'] as $itemName => $itemValue){
                    // add support for item based store groups
                    if (empty($itemValue['CSgroup'])) {
                        $itemValue['CSgroup'] = $value['classificationStoreGroupId'];
                    }

                    if (!empty($itemValue['CSkey'])) {
                        if ($itemValue['type'] === 'MULTISELECT') {
                            if (isset($itemValue['seperator'])) {
                                $tmpArray = explode($itemValue['seperator'], $itemValue['value']);
                                $itemValue['value'] = [];
                                foreach ($tmpArray as $tmpValue) {
                                    $tmpValue = trim(strtolower($tmpValue));
                                    $itemValue['value'][] = (isset($itemValue['translationPrefix']) && !empty($tmpValue)) ?
                                        $itemValue['translationPrefix'] . str_replace(" ", "_", $tmpValue) :
                                        $tmpValue;
                                }
                            }
                        } elseif ($itemValue['type'] === 'SELECT') {
                            $tmpValue = trim($itemValue['value']);
                            if ('' !== $tmpValue) {
                                $itemValue['value'] = (isset($itemValue['translationPrefix']) && !empty($tmpValue)) ? $itemValue['translationPrefix'] . strtolower($tmpValue) : $tmpValue;
                            }
                        }

                        // check if group is already activated, set to active if not
                        $activeGroups = $store->getActiveGroups();
                        if (!isset($activeGroups[$itemValue['CSgroup']])
                            || false === $activeGroups[$itemValue['CSgroup']]
                        ) {
                            $activeGroups[$itemValue['CSgroup']] = true;
                            $store->setActiveGroups($activeGroups);
                        }

                        $store->setLocalizedKeyValue($itemValue['CSgroup'], $itemValue['CSkey'], $itemValue['value'], $value['locale']);

                        // set default value if given locale is the system config default language
                        if ($value['locale'] == \Pimcore\Config::getSystemConfig()->get('general')->get('defaultLanguage')) {
                            $store->setLocalizedKeyValue($itemValue['CSgroup'], $itemValue['CSkey'], $itemValue['value']);
                        }
                    }
                }

                $value = null;
                if ($store instanceof Classificationstore) {
                    $value = $store;
                }
                break;
            case 'geopoint':
                $geoPoint = $dataObject->{'get' . ucfirst($fieldDefinition->getName())}();
                if (!$geoPoint instanceof Geopoint) {
                    $geoPoint = new Geopoint();
                }

                $geoPoint->setLatitude($value['latitude']);
                $geoPoint->setLongitude($value['longitude']);

                $value = $geoPoint;
                break;
            case 'imageGallery':
                /** @var ImageGallery $imageGallery */
                $imageGallery = $dataObject->{'get' . ucfirst($fieldDefinition->getName())}();
                if (!$imageGallery instanceof ImageGallery) {
                    $imageGallery = new ImageGallery([]);
                }

                $hotspotImages = [];
                foreach (is_array($value) ? $value : [] as $imagePath) {
                    $image = Asset\Image::getByPath($imagePath);

                    if (null !== $image) {
                        $hotspotImage = new Hotspotimage();
                        $hotspotImage->setImage($image);
                        $hotspotImages[] = $hotspotImage;
                    }
                }

                $imageGallery->setItems($hotspotImages);
                $value = $imageGallery;
                break;
        }

        if(\is_string($value) && \preg_match('/^[^:]+:[^:]+(:[^:]+)?$/', $value)) {
            $parts = explode(':', $value);

            // Get class from parts ;)
            $type = $parts[0];
            if (!class_exists($type)) {
                $type = 'Pimcore\\Model\\DataObject\\' . ucfirst($type);
            }
            if (!class_exists($type)) {
                $type = 'Pimcore\\Model\\Object\\' . ucfirst($type);
            }
            if (!class_exists($type)) {
                return $value;
            }

            if(\count($parts) === 2) {
                $parts = [
                    $parts[0],
                    'path',
                    $parts[1],
                ];
            }

            if(\count($parts) === 3) {
                $method = 'getBy'.ucfirst($parts[1]);

                if(\strtolower($parts[1]) === 'path') {
                    $objectPathArray = explode('/', $parts[2]);
                    array_walk($objectPathArray, function (&$value) {
                        $value = Service::getValidKey($value, 'object');
                    });
                    $parts[2] = implode('/', $objectPathArray);
                }

                try {
                    $targetObject = $type::$method($parts[2], 1);
                } catch (\Exception $e) {
                    $this->logger->debug($e->getMessage().' Searched for '.$type.' with '.$parts[1].'="'.$parts[2].'"');
                }

                if ($targetObject instanceof $type) {
                    if($fieldDefinition->getName() === 'path') {
                        $getter = 'getFullPath';
                    } else {
                        $getter = 'get'.\ucfirst($fieldDefinition->getName());
                    }
                    return $targetObject->$getter();
                }
            }
        }

        return $value;
    }

    /**
     * Versucht, ein Asset vom Filesystem oder von einer URL zu laden
     * Falls das neue Asset nicht geladen werden kann, wird $currentValue zurückgegeben
     */
    private function getAsset($value, $config = null) {
        if (empty($value)) {
            return $value;
        }

        $overwrite = false;
        if ($config && array_key_exists("overwrite", $config)) {
            $overwrite = $config['overwrite'] === true;
        }

        if(\is_string($value)) {
            $value = ['url' => $value];
        }

        if(isset($value['url'])) {
            $url = trim($value['url']);

            if (filter_var($url, FILTER_VALIDATE_URL) !== false) {
                try {
                    $response = Tool::getHttpData($url);
                    if ($response) {
                        $data = $response;
                    } else {
                        $this->logger->error('Unable to load asset data from "' . $url . '"');
                        return null;
                    }

                    $filename = substr($url, strrpos($url, '/') + 1);
                } catch (\Exception $e) {
                    $this->logger->error("Unable to load uri '" . $url . "'");
                    return null;
                }
            } else {
                $url = "file://" . $this->assetSource . "/" . $url;
                if (is_file($url)) {
                    $data = file_get_contents($url);
                    $filename = substr($url, strrpos($url, '/') + 1);
                } else {
                    $this->logger->error("Unable to load uri '" . $url . "'");
                    return null;
                }
            }

            $pathPrefix = '/';
            if ($this->assetFolder instanceof Asset\Folder) {
                $pathPrefix = $this->assetFolder->getFullPath() . '/';
            }

            $filename = Service::getValidKey($filename, 'asset');
            $newFileName = $filename;
            $existing = null;
            $i = 1;
            do {
                $existing = $asset = Asset::getByPath($pathPrefix . $newFileName);
                if (true === $overwrite) {
                    break;
                }

                if ($existing instanceof Asset) {
                    $newFileName = $i . '-' . $filename;
                    $i++;
                }
            } while (!is_null($existing) && $i <= 10000);

            $parent = ($this->assetFolder instanceof Asset\Folder) ? $this->assetFolder->getId() : 1;
            if (true === $overwrite && $asset instanceof Asset) {
                $asset->setData($data);
            } else {
                $asset = Asset::create($parent, array(
                    'filename' => $newFileName,
                    'data' => $data,
                    'userOwner' => 0,
                    'userModification' => 0,
                ));
            }
            $asset->setProperty('sourcePath', 'Text', $url);
        } elseif(isset($value['query'])) {
            $parts = explode(':', $value['query']);
            if(count($parts) === 2) {
                \array_unshift($parts, 'Pimcore\\Model\\Asset');
            }
            $type = $parts[0];
            if(!class_exists($type)) {
                $type = 'Pimcore\\Model\\Asset\\'.ucfirst($type);
            }
            $method = 'getBy'.ucfirst($parts[1]);

            if(\strtolower($parts[1]) === 'path') {
                $objectPathArray = explode('/', $parts[2]);
                array_walk($objectPathArray, function (&$value) {
                    $value = Service::getValidKey($value, 'asset');
                });
                $parts[2] = implode('/', $objectPathArray);
            }

            try {
                $targetObject = $type::$method($parts[2], 1);
            } catch (\Exception $e) {
                $this->logger->debug($e->getMessage().' Searched for '.$type.' with '.$parts[1].'="'.$parts[2].'"');
            }

            if($targetObject instanceof Asset) {
                $asset = $targetObject;
            } else {
                $this->logger->error("Unable to find asset by query '" . $value['query'] . "'");
                return null;
            }
        } else {
            return null;
        }

        if(!empty($value['filename'])) {
            $asset->setFilename($value['filename']);
            $asset->setUserModification(0);
        }

        return $asset;
    }


    /**
     * @return bool
     * @throws \Exception
     */
    private function isModified(Concrete $newObject = null, Concrete $oldObject = null, array $fieldsToTest=null) {
        try {
            if ($newObject === null || $oldObject === null){
                return true;
            }
            if ($newObject->getPublished() !== $oldObject->getPublished()){
                return true;
            }
            if ($newObject->getFullPath() !== $oldObject->getFullPath()){
                return true;
            }

            // Dependencies
            // Haben sich Abhängigkeiten geändert, gilt das Objekt ebenfalls als geändert
            if ($newObject->getDependencies()->getRequires() != $oldObject->getDependencies()->getRequires()) {
                return true;
            }

            $fields = $newObject->getClass()->getFieldDefinitions();
            foreach ($fields as $fieldName => $definition) {
                if($definition instanceof ClassDefinition\Data\Localizedfields) {
                    foreach(Tool::getValidLanguages() as $language) {
                        foreach ($definition->getFieldDefinitions() as $lfd) {
                            if ($fieldsToTest !== null && !in_array($lfd->getName(), $fieldsToTest)){
                                continue;
                            }

                            $getter = 'get'.ucfirst($lfd->getName());
                            $v1 = $lfd->getVersionPreview($newObject->$getter($language));
                            $v2 = $lfd->getVersionPreview($oldObject->$getter($language));

                            if ($v1 != $v2) {
                                return true;
                            }
                        }
                    }
                } elseif($definition instanceof ClassDefinition\Data\Objectbricks) {
                    $bricks1 = $newObject->{'get'. ucfirst($fieldName)}();
                    $bricks2 = $oldObject->{'get'. ucfirst($fieldName)}();

                    if (!$bricks1 instanceof \Pimcore\Model\DataObject\Objectbrick &&
                        !$bricks2 instanceof \Pimcore\Model\DataObject\Objectbrick) {
                        continue;
                    }

                    foreach($definition->getAllowedTypes() as $allowedType) {
                        $collectionDef = Definition::getByKey($allowedType);

                        foreach ($collectionDef->getFieldDefinitions() as $lfd) {
                            if ($fieldsToTest !== null && !in_array($lfd->getName(),$fieldsToTest)){
                                continue;
                            }
                            $v1 = null;
                            if ($bricks1) {
                                $brick1Value = $bricks1->{'get'. $allowedType}();
                                if ($brick1Value) {
                                    $v1 = $lfd->getVersionPreview($brick1Value->{'get'. ucfirst($lfd->getName())}());
                                }
                            }
                            $v2 = null;
                            if ($bricks2) {
                                $brick2Value = $bricks2->{'get'. $allowedType}();
                                if ($brick2Value) {
                                    $v2 = $lfd->getVersionPreview($brick2Value->{'get'. ucfirst($lfd->getName())}());
                                }
                            }

                            if ($v1 != $v2) { return true; }
                        }
                    }
                } elseif ($definition instanceof ClassDefinition\Data\Classificationstore) {
                    if ($fieldsToTest !== null && !in_array($fieldName, $fieldsToTest)){
                        continue;
                    }
                    $getter = 'get'.ucfirst($fieldName);
                    if (method_exists($definition, 'getDiffVersionPreview')) {
                        $v1 = $definition->getDiffVersionPreview($newObject->$getter());
                        $v2 = $definition->getDiffVersionPreview($oldObject->$getter());
                    } else {
                        $v1 = $definition->getVersionPreview($newObject->$getter());
                        $v2 = $definition->getVersionPreview($oldObject->$getter());
                    }

                    if ($v1 !== $v2) { return true; }
                } else {
                    if ($fieldsToTest !== null && !in_array($fieldName, $fieldsToTest)){
                        continue;
                    }
                    $getter = 'get'.ucfirst($fieldName);
                    if (method_exists($definition, 'getDiffVersionPreview')) {
                        $v1 = $definition->getDiffVersionPreview($newObject->$getter());
                        $v2 = $definition->getDiffVersionPreview($oldObject->$getter());
                    } else {
                        $v1 = $definition->getVersionPreview($newObject->$getter());
                        $v2 = $definition->getVersionPreview($oldObject->$getter());
                    }

                    if ($v1 != $v2) { return true; }
                }
            }
        } catch (\Exception $e) {
            // In case of error assume the object was changed
            return true;
        }

        return false;
    }

    protected static function objectToArray($d) {
        if (is_object($d)) {
            $d = get_object_vars($d);
        }

        if (is_array($d)) {
            return array_map( array('Pim_Item_Importer','objectToArray') , $d);
        }
        return $d;
    }

    /**
     * Converts a string to a number
     * @param string $value the string to be converted
     * @param string $decimalSep decimal separator
     * @param string $groupingSep grouping separator
     * @return float the resulting number
     */
    protected function parseNumber($value, $decimalSep = null, $groupingSep = null) {
        if (!empty($groupingSep)) {
            $value = str_replace($groupingSep, '', $value);
        }

        if (!empty($decimalSep)) {
            $value = str_replace($decimalSep, '.', $value);
        }

        return (float)$value;
    }

    protected function createObject($className, $classId, AbstractObject $parent, $key) {
        $key = Service::getValidKey($key, 'object');

        $intendedPath = $parent->getFullPath() .'/'. $key;

        if (!Service::pathExists($intendedPath)) {
            try {
                $object = \Pimcore::getContainer()->get('pimcore.model.factory')->build('Pimcore\\Model\\DataObject\\' . ucfirst($className));
            } catch(\Pimcore\Loader\ImplementationLoader\Exception\UnsupportedException $e) {
                $object = \Pimcore::getContainer()->get('pimcore.model.factory')->build('Pimcore\\Model\\Object\\' . ucfirst($className));
            }

            if($object instanceof Concrete) {
                $object->setOmitMandatoryCheck(true); // allow to save the object although there are mandatory fields
            }
            $object->setClassId($classId);
            $object->setClassName($className);
            $object->setParentId($parent->getId());
            $object->setKey($key);
            $object->setCreationDate(time());
            $object->setUserOwner(0);
            $object->setUserModification(0);
            $object->setPublished(false);

            return $object;
        }

        $this->logger->error("prevented creating PIM object because object with same path+key [ $intendedPath ] already exists");
        return null;
    }

    private function getIdPrefix() {
        return $this->idPrefix;
    }

    /**
     * @return array
     */
    public function getChangedObjectIds() {
        return $this->changedObjectIds;
    }

    protected function hasObjectChanged($id) {
        return in_array($id, $this->changedObjectIds);
    }

    protected function markObjectAsChanged($id) {
        if (!in_array($id, $this->changedObjectIds)) {
            $this->changedObjectIds[] = $id;
        }
    }

    /**
     * @param int $dataportId
     */
    private function hasRawdataChanged($dataportId, array $rawItem, Concrete $item) : bool
    {
        $property = self::HASH_PROP_PREFIX . $dataportId;

        // No attribute present on the object,
        if (!$item->hasProperty($property)) {
            return true;
        }

        $hash = $item->getProperty($property);

        return $rawItem['hash'] !== $hash;
    }

    /**
     * @param string $path
     * @param array  $targetconfig
     *
     * @return string
     */
    private function sanitizePathString($path, array $targetconfig)
    {
        if (strpos($path, '/') !== 0) {
            $path = $targetconfig['itemFolder'] . '/' . $path;
        }

        $objectPathArray = explode('/', $path);
        array_walk(
            $objectPathArray,
            function (&$value) {
                $value = Service::getValidKey($value, 'object');
            }
        );
        return implode('/', $objectPathArray);
    }
}
