<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\Parser;

trait ResourceBasedParser
{
    private $removeFileAfterImport = false;

    /**
     * @param $source
     *
     * @return bool|null|string
     */
    private function getFileOrUrl($source) {
         if(\is_dir($source)) {
            $source = realpath($source);
            $directoryIterator = new \RecursiveDirectoryIterator($source, \RecursiveDirectoryIterator::SKIP_DOTS);

            /**
             * @param \SplFileInfo $file
             * @param mixed $key
             * @param \RecursiveCallbackFilterIterator $iterator
             * @return bool True if you need to recurse or if the item is acceptable
             */
            $filter = function ($file, $key, $iterator) use ($source) {
                if ($iterator->hasChildren() && $source.'/archive' !== $file->getPathname()) {
                    return true;
                }
                return $file->isFile();
            };
            $fileIterator = new \RecursiveIteratorIterator(new \RecursiveCallbackFilterIterator($directoryIterator, $filter));

            $fileIterator->rewind();
            if(!$fileIterator->valid()) {
                return null;
            }

            /** @var \SplFileInfo $file */
            $file = $fileIterator->current();
            return $file->getPathname();
        }

        if(strpos($source, '/') === 0 && !\file_exists($source)) {
            // relative HTTP route
            $mainDomain = \Pimcore\Config::getSystemConfig()->general->domain;
            if ($mainDomain) {
                return $mainDomain.$source;
            }
        }

        if(strpos($source, ' ') !== false) {
             // call to CLI script
             $arguments = \array_filter(explode(' ', $source));
             $script = \array_shift($arguments);
             $config = $this->config;
             $sourceResult = \call_user_func(function() use ($script, $arguments, $config) {
                 return include PIMCORE_PROJECT_ROOT.'/'.$script;
             });

             if($sourceResult) {
                 return $sourceResult;
             }
         }

        return $source;
    }

    private function archive($filePath, $source) {
        if(\is_dir($source)) {
            $source = realpath($source);
            $file = new \SplFileInfo($filePath);
            $archiveDirectory = $source.'/archive/'.\preg_replace('~^'.$source.'~', '', $file->getPath());
            if (!is_dir($archiveDirectory) && !mkdir($archiveDirectory, 0744, true)) {
                throw new \InvalidArgumentException('Could not create directory "'.$archiveDirectory.'"');
            }

            \copy($filePath, $archiveDirectory.'/'.$file->getFilename());
        }

        if($this->removeFileAfterImport && \is_file($filePath)) {
            unlink($filePath);
        }
    }

    public function setSourceFile($file) {
        if(!$file) {
            return;
        }
        $this->config['file'] = $file;
    }

    /**
     * @param bool $removeFileAfterImport
     */
    public function removeFileAfterImport($removeFileAfterImport = true)
    {
        $this->removeFileAfterImport = (bool)$removeFileAfterImport;
    }
}