<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\Controller;

use Blackbit\PimBundle\lib\Pim\Helper;
use Blackbit\PimBundle\lib\Pim\Item\ImporterInterface;
use Blackbit\PimBundle\lib\Pim\Item\Importmanager;
use Blackbit\PimBundle\lib\Pim\RawData;
use Blackbit\PimBundle\model\Dataport;
use Blackbit\PimBundle\model\ImportStatus;
use Blackbit\PimBundle\model\RawItem;
use Pimcore\Controller\FrontendController;
use Pimcore\Db;
use Pimcore\Tool\Console;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/**
 * Rohdatenimport (Datenquelle in DB) starten
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Medien GmbH, http://www.blackbit.de/
* @Route("/import")
 */
class ImportController extends FrontendController {
	public function importDataportAction(Request $request) {
		// Starte Import
		$importer = new RawData\Importmanager();
		$importer->importDataport($request->get('id'));


		$response = array(
			'success' => true
		);
		return new JsonResponse($response);
	}

    /**
     * @Route("/delete-rawdata")
     */
	public function deleteRawdataAction(Request $request) {
		$portId = (int)$request->get('dataportId');

		$rawItemRepository = new RawItem();
		$rawItemRepository->deleteWhere(['dataportId' => $portId]);

        Db::get()->delete('properties', ['name' => 'importhash_'.$portId]);

		return new JsonResponse([
            'success' => true,
            'port' => $portId
        ]);
	}


	public function importBackgroundAction(Request $request) {
		$cmd = 'php ' . __DIR__ . '/../cli/fullimport.php dataport=' . (int)$request->get('id') . ' > /dev/null 2>&1 &';

		exec($cmd);

		$response = array(
			'success' => true
		);

		return new JsonResponse($response);
	}

    /**
     * @Route("/get-status/{dataportId}")
     */
	public function getStatusAction(int $dataportId, Request $request) {
		$data = array(
			'success' => true,
			'status' => array()
		);

		$importStatusModel = new ImportStatus();
		$result = $importStatusModel->find([
			'dataportId = ?' => $dataportId
		], 'startDate DESC, `key` DESC', $request->get('limit'), $request->get('start'));


		$dataportModel = new Dataport();

		$status = [];

		$dataportNames = [];

		foreach ($result as $entry) {
			$percentage = 0;
			$total = (int)$entry['totalItems'];
			$done = (int)$entry['doneItems'];

			if ($total > 0) {
				$percentage = (int)($done / $total * 100);
			}

			$name = '';
			if (array_key_exists($dataportId, $dataportNames)) {
				$name = $dataportNames[$dataportId];
			} else {
				$dataport = $dataportModel->get($dataportId);
				if ($dataport) {
					$name = $dataport['name'];
					$dataportNames[$dataportId] = $name;
				}
			}

			$startDate = $entry['startDate'];
			$endDate = $entry['endDate'];

			if (!empty($startDate)) {
				$startDate = (new \DateTimeImmutable($startDate))->format('d.m.Y H:i');
			}

			if (!empty($endDate)) {
                $endDate = (new \DateTimeImmutable($endDate))->format('d.m.Y H:i');
            }

			$status[] = array(
				'id' => $entry['key'],
				'dataport' => $name,
				'startDate' => $startDate,
				'endDate' => $endDate,
				'status' => $entry['status'],
				'percentage' => $percentage,
				'type' => $entry['importType'],
			);
		}

		$data['status'] = $status;
		$data['total'] = $importStatusModel->countRows([
            'dataportId = ?' => $dataportId
        ]);
        return new JsonResponse($data);
	}

    /**
     * @Route("/manual-import")
     */
	public function manualImportAction(Request $request) {
		$response = array(
			'success' => true
		);

		try {
			$phpCliBin = (bool) Console::getPhpCli();
		} catch (\Exception $e) {
			$phpCliBin = false;
		}

		if ($phpCliBin) {
			$dataportId = (int)$request->get('dataportId');
			$dataportModel = new Dataport();
			$dataport = $dataportModel->get($dataportId);

			if ($dataport) {
				$type = $request->get('importType');
				if ($type === 'raw') {
					// Rohdatenimport
					$response['success'] = $this->startImport('importfile', $dataportId);
				} else if ($type === 'complete') {
					// Komplettimport
					$response['success'] = $this->startImport('importfile', $dataportId, true);
				} else if ($type === 'bmecat' && $dataport['sourcetype'] === 'bmecat') {
					$response['success'] = $this->startBmecatImport('importfile', $dataportId);
				} else {
					$response['success'] = false;
				}
			} else {
				$response['success'] = false;
			}
		} else {
			$response['success'] = false;
			$response['msg'] = 'pim.manual.error.noPhpCli';
		}

        return new JsonResponse($response);
	}

    /**
     * @Route("/upload-bmecat")
     */
	public function uploadBmecatAction(Request $request) {
		@ini_set("max_execution_time", 3600);
		set_time_limit(3600);

		$response = array(
			'success' => true,
			'isValid' => false,
			'file' => null,
		);

		$dataportId = (int)$request->get('dataportId');

		$file = null;
		$dataportModel = new Dataport();
		$dataport = $dataportModel->get($dataportId);
		$version = null;

		if ($dataport && !empty($dataport['sourceconfig'])) {
            $config = unserialize($dataport['sourceconfig']);

            if (array_key_exists('importfile', $_FILES) && !empty($_FILES['importfile']['tmp_name'])) {
                // Hochgeladene Datei importieren
                $fileData = $_FILES['importfile'];

                $hash = md5_file($fileData['tmp_name']);

                $importFileName = 'bmecat-'.$hash;
                $importFilePath = PIMCORE_TEMPORARY_DIRECTORY.DIRECTORY_SEPARATOR.$importFileName;
                if (!$fileData['error'] && move_uploaded_file($fileData['tmp_name'], $importFilePath)) {
                    $file = $importFilePath;
                    $config['file'] = $file;
                }

                $dataport['sourceconfig'] = serialize($config);
                $dataportModel->update($dataport, ['id' => $dataportId]);
            } elseif (array_key_exists('file', $config)) {
                $file = $config['file'];
            }

            if (array_key_exists('version', $config)) {
                $version = $config['version'];
            }
        }


		if (empty($file) || !file_exists($file) || !is_readable($file)) {
		    return new JsonResponse($response);
		}


		// Parse file
		Helper::parseBmecat($file, $version);

		$response['file'] = $file;
		$response['isValid'] = Helper::validateBmecat($file, $version);
		$response['fileInfo'] = Helper::getBmecatInfo($file, $version);

		$importStatusModel = new Importstatus();
		$lastImport = $importStatusModel->find(array(
			'dataportId = ?' => $dataportId,
			'status = ?' => Importstatus::STATUS_FINISHED,
		), 'endDate DESC', 1);


		$importKey = null;
		if (count($lastImport) === 1) {
			$importKey = $lastImport[0]['key'];
		}

		$response['comparison'] = Helper::compareToImport($file, $version, $importKey);

		return new JsonResponse($response);
	}

    /**
     * @Route("/manual-pim-import")
     */
	public function manualPimImportAction(Request $request) {
		$response = array(
			'success' => true
		);

		try {
			$phpCliBin = (bool) Console::getPhpCli();
		} catch (\Exception $e) {
			$phpCliBin = false;
		}

		if ($phpCliBin) {
			$dataportId = (int)$request->get('dataportId');
			$dataportModel = new Dataport();
			$dataport = $dataportModel->get($dataportId);

			if ($dataport) {
                $ignoreHashCheck = '';
                if($request->get('ignoreHashCheck')) {
                    $ignoreHashCheck = ' --ignore-hash-check';
                }
				$cmd = escapeshellcmd('"' . Console::getPhpCli() . '" ' .
                    realpath(PIMCORE_PROJECT_ROOT . DIRECTORY_SEPARATOR . 'bin' . DIRECTORY_SEPARATOR  . 'console') . ' import:pim ' . $dataportId.$ignoreHashCheck);
				Console::execInBackground($cmd);
			} else {
				$response['success'] = false;
			}
		} else {
			$response['success'] = false;
			$response['msg'] = 'pim.manual.error.noPhpCli';
		}

		return new JsonResponse($response);
	}

    /**
     * @param $fileFieldName
     * @param $dataportId
     * @param $command
     *
     * @return bool
     */
    private function executeImportCommand($fileFieldName, $dataportId, $command) : bool
    {
        $cmd = '"'.Console::getPhpCli().'" ' .
            realpath(PIMCORE_PROJECT_ROOT . DIRECTORY_SEPARATOR . 'bin' . DIRECTORY_SEPARATOR  . 'console') . ' ' . $command . ' ' . $dataportId;

        $importFileName = $this->getUploadedFile($fileFieldName);
        if (!empty($importFileName)) {
            $cmd .= ' ' . $importFileName;
        }

        Console::execInBackground(escapeshellcmd($cmd));

        return true;
    }

	/**
	 * @param string $fileFieldName
	 * @param integer $dataportId
	 * @param bool $complete
	 * @return bool
	 */
	private function startImport($fileFieldName, $dataportId, $complete = false) {
		$command = 'import:rawdata';
		if ($complete === true) {
			$command = 'import:complete';
		}

        return $this->executeImportCommand($fileFieldName, $dataportId, $command);
	}

	private function startBmecatImport($fileFieldName, $dataportId) {
//        return $this->executeCommand($fileFieldName, $dataportId, 'bmecatimport.php');
        throw new \Exception('Not implemented yet in Pimcore 5');
	}

	private function getUploadedFile($fileFieldName) {
		$importFileName = null;
		if (!array_key_exists($fileFieldName, $_FILES)) {
			return null;
		}

		if (array_key_exists($fileFieldName, $_FILES) && !empty($_FILES[$fileFieldName]['tmp_name'])) {
			// Hochgeladene Datei importieren
			$fileData = $_FILES[$fileFieldName];

			if (!array_key_exists('tmp_name', $fileData) || $fileData['error'] == true) {
				return null;
			}

			$importFileName = PIMCORE_TEMPORARY_DIRECTORY . DIRECTORY_SEPARATOR . uniqid('pimimport-');

			if (!move_uploaded_file($fileData['tmp_name'], $importFileName)) {
				return null;
			}
		}

		return $importFileName;
	}
}
