<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Blackbit\PimBundle\Controller;

use Blackbit\PimBundle\lib\Pim\Variant\Model\TreeItem;
use Blackbit\PimBundle\lib\Pim\Variant\VariantGenerator;
use Blackbit\PimBundle\lib\Pim\Variant\VariantHelper;
use Blackbit\PimBundle\model\ResourceFactory;
use Doctrine\DBAL\DBALException;
use Pimcore\Controller\FrontendController;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/**
 * Class VariantconfigController
 *
 * @package BlackbitPimBundle\Controller
 * @Route("/variantconfig")
 */
class VariantconfigController extends FrontendController
{
    const DEFAULT_SOURCETYPE = 'none';

    /**
     * @var ResourceFactory
     */
    private $resourceFactory;

    /**
     * @var VariantHelper
     */
    private $variantHelper;

    /**
     * @var VariantGenerator
     */
    private $variantGenerator;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * VariantconfigController constructor.
     *
     * @param ResourceFactory  $resourceFactory
     * @param VariantHelper    $variantHelper
     * @param VariantGenerator $variantGenerator
     * @param LoggerInterface  $logger
     */
    public function __construct(
        ResourceFactory $resourceFactory,
        VariantHelper $variantHelper,
        VariantGenerator $variantGenerator,
        LoggerInterface $logger
    ) {
        $this->resourceFactory = $resourceFactory;
        $this->variantHelper = $variantHelper;
        $this->variantGenerator = $variantGenerator;
        $this->logger = $logger;
    }

    /**
     * @Route("/get-variants")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function getVariantsAction(Request $request): JsonResponse
    {
        $selectedItem = $request->get('selectedItem');

        $treeItems[0] = (new TreeItem())->setRoot(true);
        foreach ($this->resourceFactory->variant()->find() as $variant) {
            $iconClass = 'pimcore_icon_variant';
            $type = 'item';
            $text = $variant['id'] . ': ' . $variant['name'];
            if ($variant['sourcetype'] === 'folder') {
                $iconClass = 'pimcore_icon_' . $variant['sourcetype'];
                $type = $variant['sourcetype'];
                $text = $variant['name'];
            }

            $treeItems[$variant['id']] = (new TreeItem())
                ->setId($variant['id'])
                ->setText($text)
                ->setName($variant['name'])
                ->setLeaf(true)
                ->setExpanded((intval($variant['id']) === intval($selectedItem)))
                ->setIconCls($iconClass)
                ->setParentId($variant['parent'])
                ->setType($type);
        }

        // build child structure
        foreach ($treeItems as $key => $treeItem) {
            if ($key > 0) {
                $treeItems[$treeItem->getParentId()]->addChild($treeItem);
                $treeItems[$treeItem->getParentId()]->setLeaf(false);
                // expand all parents
                if ($treeItem->isExpanded()) {
                    $parent = $treeItems[$treeItem->getParentId()];
                    while (false === $parent->isRoot()) {
                        $parent->setExpanded(true);
                        $parent = $treeItems[$parent->getParentId()];
                    }
                }
            }
        }

        return new JsonResponse($treeItems[0]->toArray()['children']);
    }

    /**
     * @Route("/get")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function getAction(Request $request): JsonResponse
    {
        $response = ['success' => false];

        $variant = $this->resourceFactory->variant()->get((int)$request->get('id'));
        if ($variant) {
            $response['data'] = $variant;
            $targetConfig = unserialize($response['data']['targetconfig']);

            $sourceConfig = unserialize($response['data']['sourceconfig']);
            if (!is_array($sourceConfig)) {
                $sourceConfig = [];
            }

            $response['data']['sourceconfig'] = $sourceConfig;
            $response['data']['fields'] = $this->variantHelper->getSourceConfig($variant['sourcetype']);
            $response['data']['mode'] = $targetConfig['mode'];

            $response['success'] = true;
        } else {
            $response['errorMessage'] = 'pim_variant_notfound';
        }

        return new JsonResponse($response);
    }

    /**
     * @Route("/add")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function addAction(Request $request): JsonResponse
    {
        $name = $this->variantHelper->sanitizeName($request->get('name'));
        $type = $request->get('type');
        $parentId = $request->get('parentId');

        if (!$this->variantHelper->checkDataportName($name)) {
            return new JsonResponse([
                'success' => false,
                'errorMessage' => 'pim_variant_name_taken',
            ]);
        }

        $variant = $this->resourceFactory->variant()->create([
            'name' => $name,
            'sourcetype' => (($type === 'folder') ? $type : self::DEFAULT_SOURCETYPE),
            'sourceconfig' => serialize([]),
            'targetconfig' => serialize([]),
            'parent' => $parentId,
        ]);

        $variant = $this->resourceFactory->variant()->update(
            ['targetconfig' => serialize([])],
            $variant['id']
        );

        $response = [
            'success' => true,
            'dataport' => $variant,
        ];

        return new JsonResponse($response);
    }

    /**
     * @Route("/delete/{variantId}")
     * @param Request $request
     * @param int     $variantId
     *
     * @return JsonResponse
     */
    public function deleteAction(Request $request, int $variantId): JsonResponse
    {
        try {
            // check if item exists
            $variant = $this->resourceFactory->variant()->findOne(['id = ?' => $variantId]);
            if (!is_array($variant) || !isset($variant['id'])) {
                throw new \Exception('Not deleted! Item not found');
            }
            // check if item has children
            if ($this->variantHelper->hasChildren($variantId)) {
                throw new \Exception('Not deleted! Item has child elements.');
            }

            $response = [
                'success' => true,
                'id' => $variantId,
                'parentId' => $variant['parent'],
            ];

            // delete generated objects
            $this->variantGenerator->delete($variantId);
            // delete raw items
            $this->resourceFactory->variantItemField()->deleteWhere(['dataportId' => $variantId]);
            // delete variant item
            $this->resourceFactory->variant()->delete($variantId);
        } catch (\Exception $exception) {
            $response['success'] = false;
            $response['errorMessage'] = $exception->getMessage();
        }

        return new JsonResponse($response);
    }

    /**
     * @Route("/update")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function updateAction(Request $request): JsonResponse
    {
        $response = ['success' => false];

        $variantId = (int)$request->get('id');
        $variant = $this->resourceFactory->variant()->get($variantId);

        if (!$variant) {
            $response['errorMessage'] = 'pim_variant_notfound';

            return new JsonResponse($response);
        }

        $name = $this->variantHelper->sanitizeName($request->get("name"));
        if (!$this->variantHelper->checkDataportName($name, $variantId)) {
            $response['errorMessage'] = 'pim_variant_name_taken';

            return new JsonResponse($response);
        }

        $sourcetype = $request->get("sourcetype");

        $data = [
            'name' => $name,
            'description' => $request->get("description"),
            'sourcetype' => $sourcetype,
        ];

        $objectData = (array)json_decode($request->get('objectData'));
        $rawitemData = json_decode($request->get("rawitemData"));

        $sourceConfig['objectData'] = $objectData;
        $sourceConfig['fields'] = [];

        if (!empty($variant['sourceconfig'])) {
            $config = unserialize($variant['sourceconfig']);
            if (!empty($config['fields'])) {
                $sourceConfig['fields'] = $config['fields'];
            }
        }

        $data['targetconfig'] = serialize([]);

        // Find existing fields
        $rawItemFields = [];
        $toBeDeleted = [];
        $existingFields = $this->resourceFactory->variantItemField()->find(['dataportId = ?' => $variantId]);
        foreach ($existingFields as $field) {
            $rawItemFields['field_' . $field['fieldNo']] = $field;
            $toBeDeleted['field_' . $field['fieldNo']] = $field;
        }

        // Create new fields and update existing ones
        if (is_array($rawitemData)) {
            $sourceConfig['fields'] = [];
            foreach ($rawitemData as $index => $setting) {
                $fieldNo = (int)$setting->fieldNo;
                $name = $setting->name;

                if ($fieldNo > 0) {
                    $key = 'field_' . $fieldNo;

                    if (empty($name)) {
                        continue;
                    }

                    if (array_key_exists($key, $rawItemFields)) {
                        $rawItemFields[$key]['name'] = $name;
                        $rawItemFields[$key]['priority'] = $index;
                    } else {
                        $rawItemFields[$key] = [
                            'dataportId' => $variantId,
                            'fieldNo' => $fieldNo,
                            'name' => $name,
                            'priority' => $index,
                        ];
                    }

                    unset($toBeDeleted[$key]);

                    // Custom field config
                    foreach ($this->variantHelper->getSourceConfigFields($sourcetype) as $config => $defaultValue) {
                        if (property_exists($setting, $config)) {
                            // A value for the config key has been submitted
                            $sourceConfig['fields'][$key][$config] = $setting->$config;
                        }
                    }
                }
            }
        }

        $data['sourceconfig'] = serialize($sourceConfig);

        try {
            $this->resourceFactory->variantItemField()->beginTransaction();
            if ($this->resourceFactory->variant()->update($data, ['id' => $variantId]) !== false) {
                $response['success'] = true;
            }

            foreach ($rawItemFields as $key => $field) {
                try {
                    $this->resourceFactory->variantItemField()->create($field);
                } catch (DBALException $exception) {
                    $this->resourceFactory->variantItemField()->update(
                        $field,
                        [
                            'dataportId' => $field['dataportId'],
                            'fieldNo' => $field['fieldNo'],
                        ]
                    );
                }
            }

            foreach ($toBeDeleted as $field) {
                // also remove the object if existing
                $this->variantGenerator->deleteOne($variantId, $data, $field);
                $this->resourceFactory->variantItemField()->deleteWhere(
                    [
                        'dataportId' => $field['dataportId'],
                        'fieldNo' => $field['fieldNo'],
                    ]
                );
            }

            $this->resourceFactory->variantItemField()->commit();
        } catch (\Exception $exception) {
            $this->resourceFactory->variantItemField()->rollback();
            $response['success'] = false;
            $response['errorMessage'] = 'Unable to save configuration: ' . $exception;
        }

        return new JsonResponse($response);
    }

    /**
     * @Route("/get-rawitemfield-config/{id}")
     * @param Request $request
     * @param int     $id
     *
     * @return JsonResponse
     */
    public function getRawitemfieldConfigAction(Request $request, int $id): JsonResponse
    {
        if (!$this->resourceFactory->variant()->get($id)) {
            return new JsonResponse([
                'success' => false,
                'total' => 0,
                'fields' => [],
            ]);
        }

        $data = $this->variantHelper->getRawitemfieldConfig($id);

        return new JsonResponse([
            'success' => true,
            'total' => count($data),
            'fields' => $data,
        ]);
    }

    /**
     * @Route("/get-variant-configs")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function getVariantConfigsAction(Request $request): JsonResponse
    {
        $result = [0 => ['id' => 'none', 'name' => 'NONE']];

        foreach ($this->resourceFactory->dataport()->find(['sourcetype = ?' => 'variant']) as $variantConfig) {
            $tmpName = explode('-', $variantConfig['name']);
            $result[] = [
                'id' => strtolower($tmpName[0]),
                'name' => $tmpName[0],
            ];
        }

        return new JsonResponse([
            'success' => true,
            'data' => $result,
        ]);
    }
}
