/*
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

pimcore.registerNS("pimcore.plugin.Pim.ImportConfig");
pimcore.plugin.Pim.DataportPanel = Ext.extend(Ext.TabPanel, {
    configPanel: null,
    previewPanel: null,
    manualPanel: null,
    dataport: null,
    dataportPanel: null,
    sourceConfigPanel: null,
    sourceConfigBasePanel: null,
    sourceConfigRawitemPanel: null,

    initComponent: function () {
        this.configPanel = Ext.create('Ext.Panel', {
            title: t('pim.dataport_configpanel'),
            iconCls: '',
            scrollable: true,
            buttons: [{
                text: t("save"),
                iconCls: "pimcore_icon_save",
                handler: this.save.bind(this)
            }]
        });

        this.manualPanel = Ext.create('pimcore.plugin.Pim.ManualImport', {
            title: t('pim.manual.title'),
            dataportId: this.dataportId,
            layout: {
                type: 'vbox',
                align: 'stretch',
                pack: 'start'
            }
        });

        this.previewPanel = new pimcore.plugin.Pim.DataportPreview({
            title: t('pim.dataport_previewpanel'),
            dataportId: this.dataportId
        });

        this.configPanel.on('afterrender', function (panel) {
            this.getEl().mask();
            this.dataportPanel = this.createDataportPanel();
            this.sourceConfigPanel = Ext.create('Ext.Panel', {
                flex: 1,
                border: false
            });

            panel.add(this.dataportPanel);
            panel.add(this.sourceConfigPanel);

            pimcore.layout.refresh();

            this.populateDataportForm();
        }.bind(this));

        Ext.apply(this, {
            title: t('pim.loading'),
            iconCls: 'pimcore_icon_loading',
            closable: true,
            activeTab: 0,
            items: [this.configPanel, this.manualPanel, this.previewPanel],
            listeners: {
                'tabchange': function () {
                    pimcore.layout.refresh();
                }
            }
        });

        pimcore.plugin.Pim.DataportPanel.superclass.initComponent.call(this);
    },

    populateDataportForm: function () {
        this.dataportPanel.getForm().load({
            url: '/admin/BlackbitPim/importconfig/get',
            params: {
                id: this.dataportId
            },
            failure: function (form, action) {
                this.getEl().unmask();
                pimcore.helpers.showNotification(t("error"), t(action.result.errorMessage), "error");
            }.bind(this),
            success: function (form, action) {
                this.getEl().unmask();
                this.dataport = action.result.data;

                this.setTitle(this.dataport.name);
                this.setIconCls('');

                this.renderSourceConfigPanel(this.dataport.sourcetype);
            }.bind(this)
        });
    },

    createDataportPanel: function () {
        var dataPortStore = Ext.create('Ext.data.JsonStore', {
            model: 'DataPortModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/importconfig/get-pimcore-classes',
                reader: {
                    type: 'json',
                    rootProperty: 'classes'
                }
            },
            autoLoad: true
        });

        var javascriptEngineStore = Ext.create('Ext.data.JsonStore', {
            model: 'JavascriptEngineModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/importconfig/get-javascript-engines',
                reader: {
                    type: 'json',
                    rootProperty: 'data'
                }
            },
            autoLoad: true
        });

        var form = Ext.create('Ext.form.Panel', {
            border: false,
            padding: 5,
            defaults: {
                width: 600
            },
            labelWidth: 300,
            layout: {
                type: 'vbox',
                align: 'streched'
            },
            items: [
                {
                    xtype: 'textfield',
                    name: 'name',
                    fieldLabel: t('pim.dataport_name'),
                    allowBlank: false
                },
                {
                    xtype: 'textfield',
                    name: 'description',
                    fieldLabel: t('pim.dataport_description')
                },
                {
                    xtype: 'combo',
                    hiddenName: 'sourcetype',
                    name: 'sourcetype',
                    fieldLabel: t('pim.dataport_sourcetype'),
                    forceSelection: true,
                    lazyInit: false,
                    editable: false,
                    triggerAction: 'all',
                    store: [
                        ['xml', 'XML'],
                        ['csv', 'CSV'],
                        ['excel', 'Excel'],
                        ['pimcore', 'Pimcore'],
                        ['variant', 'Variant'],
                        ['bmecat', 'BMEcat']
                    ],
                    value: 'xml',
                    listeners: {
                        'change': function(combo, newValue, oldValue) {
                            if(this.sourceConfigRawitemPanel && this.sourceConfigRawitemPanel.store.data.items.length > 0) {
                                Ext.MessageBox.show({
                                    title: t('pim.dataport_sourcetype.change.question'),
                                    msg: t('pim.dataport_sourcetype.change.warning'),
                                    buttons: Ext.MessageBox.YESNO,
                                    fn: function (e) {
                                        switch (e) {
                                            case 'yes':
                                                this.renderSourceConfigPanel(newValue);
                                                break;
                                            default:
                                                combo.suspendEvent('change');
                                                combo.select(oldValue);
                                                combo.resumeEvent('change');
                                        }
                                    }.bind(this),
                                    icon: Ext.MessageBox.QUESTION
                                });
                            } else {
                                this.renderSourceConfigPanel(newValue);
                            }
                        }.bind(this)
                    }
                },
                {
                    xtype: 'combo',
                    hiddenName: 'targetclass',
                    name: 'targetclass',
                    fieldLabel: t('pim.dataport_targetclass'),
                    selectOnFocus: true,
                    queryMode: 'local',
                    typeAhead: true,
                    forceSelection: true,
                    store: dataPortStore,
                    valueField: 'id',
                    displayField: 'name',
                    minChars: 1
                },
                {
                    xtype: 'panel',
                    collapsible: true,
                    collapsed: true,
                    title: t('pim.dataport_advancedOptions'),
                    defaults: {
                        width: 600
                    },
                    labelWidth: 300,
                    items: [
                        {
                            xtype: 'combo',
                            hiddenName: 'mode',
                            name: 'mode',
                            fieldLabel: t('pim.dataport_mode'),
                            selectOnFocus: true,
                            queryMode: 'local',
                            lazyInit: false,
                            editable: false,
                            forceSelection: true,
                            triggerAction: 'all',
                            store: [
                                ['3', t('pim.dataport_mode.create_and_edit')],
                                ['1', t('pim.dataport_mode.create_only')],
                                ['2', t('pim.dataport_mode.edit_only')]
                            ],
                            value: '3'
                        },
                        {
                            xtype: 'textfield',
                            name: 'targetfolder',
                            fieldLabel: t('pim.dataport_targetfolder'),
                            cls: "input_drop_target",
                            listeners: {
                                render: function (el) {
                                    var dd = new Ext.dd.DropZone(el.getEl().dom, {
                                        ddGroup: "element",

                                        getTargetFromEvent: function (e) {
                                            return this.getEl();
                                        },

                                        onNodeOver: function (target, dd, e, data) {
                                            data = data.records[0].data;

                                            if (data.elementType == "object" && data.type == "folder") {
                                                return Ext.dd.DropZone.prototype.dropAllowed;
                                            }
                                            return Ext.dd.DropZone.prototype.dropNotAllowed;
                                        },

                                        onNodeDrop: function (target, dd, e, data) {
                                            data = data.records[0].data;

                                            if (data.elementType == "object" && data.type == "folder") {
                                                this.setValue(data.path);
                                                return true;
                                            }
                                            return false;
                                        }.bind(this)
                                    });
                                }
                            }
                        },
                        {
                            xtype: 'textfield',
                            name: 'assetfolder',
                            fieldLabel: t('pim.dataport_assetfolder'),
                            cls: "input_drop_target",
                            listeners: {
                                render: function (el) {
                                    var dd = new Ext.dd.DropZone(el.getEl().dom, {
                                        ddGroup: "element",

                                        getTargetFromEvent: function (e) {
                                            return this.getEl();
                                        },

                                        onNodeOver: function (target, dd, e, data) {
                                            data = data.records[0].data;

                                            if (data.elementType == "asset" && data.type == "folder") {
                                                return Ext.dd.DropZone.prototype.dropAllowed;
                                            }
                                            return Ext.dd.DropZone.prototype.dropNotAllowed;
                                        },

                                        onNodeDrop: function (target, dd, e, data) {
                                            data = data.records[0].data;

                                            if (data.elementType == "asset" && data.type == "folder") {
                                                this.setValue(data.path);
                                                return true;
                                            }
                                            return false;
                                        }.bind(this)
                                    });
                                }
                            }
                        },
                        {
                            xtype: 'textfield',
                            name: 'importerClass',
                            fieldLabel: t('pim.dataport_importer_class'),
                            allowBlank: true
                        },
                        {
                            xtype: 'combo',
                            hiddenName: 'javascriptEngine',
                            name: 'javascriptEngine',
                            fieldLabel: t('pim.dataport_javascriptEngine'),
                            selectOnFocus: true,
                            queryMode: 'local',
                            lazyInit: false,
                            editable: false,
                            triggerAction: 'all',
                            store: javascriptEngineStore,
                            valueField: 'id',
                            displayField: 'name',
                            emptyText: t('pim.jsengine.none'),
                            listeners: {
                                select: function (comp, record, index) {
                                    if (comp.getValue() == "" || comp.getValue() == "&nbsp;") {
                                        comp.setValue(null);
                                    }
                                }
                            }

                        }, {
                            xtype: 'textfield',
                            name: 'idPrefix',
                            fieldLabel: t('pim.dataport_idprefix'),
                            allowBlank: true
                        }, {
                            xtype: 'textfield',
                            name: 'categoryClass',
                            fieldLabel: t('pim.dataport_categoryclass'),
                            allowBlank: true
                        }, {
                            xtype: 'textfield',
                            name: 'fieldnameProducts',
                            fieldLabel: t('pim.dataport_fieldnameProducts'),
                            allowBlank: true
                        }
                    ]
                }
            ]
        });

        return form;
    },

    renderSourceConfigPanel: function (sourcetype) {
        if(!this.dataport) {
            return;
        }

        this.sourceConfigPanel.removeAll();
        this.setActiveTab(0);
        this.sourceConfigBasePanel = null;
        this.sourceConfigRawitemPanel = null;

        switch (sourcetype) {
            case 'xml':
                this.renderXmlSourceConfigPanel();
                break;
            case 'csv':
                this.renderCsvSourceConfigPanel();
                break;
            case 'excel':
                this.renderExcelSourceConfigPanel();
                break;
            case 'pimcore':
                this.renderPimcoreSourceConfigPanel();
                break;
            case 'variant':
                this.renderVariantSourceConfigPanel();
                break;
            case 'bmecat':
                this.renderBmecatSourceConfigPanel();
                break;
        }
    },

    renderXmlSourceConfigPanel: function () {
        var additionalColumns = this.getColumnConfigForSourcetype('xml');

        // Rohdatenfeldkonfiguration
        if (typeof XmlImportconfigModel == 'undefined') {
            var readerFields = [
                {name: 'fieldNo', allowBlank: false, type: 'integer'},
                {name: 'name', allowBlank: false}
            ];

            Ext.each(additionalColumns, function (column) {
                readerFields.push({name: column.dataIndex});
            });

            readerFields.push({name: 'multiValues', type: 'boolean'});
            readerFields.push({name: 'data1'});
            readerFields.push({name: 'data2'});

            Ext.define('XmlImportconfigModel', {
                    extend: 'Ext.data.Model',
                    fields: readerFields,
                    idProperty: 'fieldNo',
                    root: 'fields'
                }
            );
        }

        var store = new Ext.data.JsonStore({
            model: 'XmlImportconfigModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/importconfig/get-rawitemfield-config/' + this.dataport.id,
                reader: {
                    type: 'json',
                    rootProperty: 'fields'
                }
            },
            autoLoad: true
        });

        var columnConfig = [];
        columnConfig.push({
            header: "#",
            width: 40,
            fixed: true,
            dataIndex: 'fieldNo',
            //editor: new Ext.form.NumberField({allowBlank: false})
            editor: {
                xtype: 'numberfield',
                allowBlank: false
            }
        });

        columnConfig.push({
            header: t('pim.dataport-fields-name'),
            width: 'auto',
            dataIndex: 'name',
            flex: 2,
            //editor: new Ext.form.TextField({allowBlank: false})
            editor: {
                xtype: 'textfield',
                allowBlank: false
            }
        });

        Ext.each(additionalColumns, function (column) {
            columnConfig.push(column);
        });

        columnConfig.push({
            header: t('pim.dataport-fields-multivalues'),
            width: 'auto',
            dataIndex: 'multiValues',
            xtype: 'checkcolumn'
        });

        columnConfig.push({
            header: t('pim.dataport-fields-ex1'),
            dataIndex: 'data1',
            width: 'auto',
            flex: 1
        });

        columnConfig.push({
            header: t('pim.dataport-fields-ex2'),
            dataIndex: 'data2',
            width: 'auto',
            flex: 1
        });

        var rowEditing = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToMoveEditor: 1,
            autoCancel: true
        });

        this.sourceConfigRawitemPanel = Ext.create('Ext.grid.Panel', {
            title: '',
            store: store,
            columns: columnConfig,
            flex: 1,
            border: false,
            frame: false,
            autoScroll: true,
            layout: 'fit',
            minHeight:300,
            dockedItems: [{
                xtype: 'toolbar',
                dock: 'top',
                items: [
                    {
                        text: t('add'),
                        handler: this.onAddDataport.bind(this),
                        iconCls: "pimcore_icon_add"
                    },
                    '-',
                    {
                        text: t('delete'),
                        handler: this.onDeleteDataport.bind(this),
                        iconCls: "pimcore_icon_delete"
                    }
                ]
            }],
            listeners: {
                edit: function (editor, e) {
                    this.onAddDataport();
                }.bind(this)
            },
            plugins: [rowEditing],
            viewConfig: {
                plugins: {
                    ptype: 'gridviewdragdrop',
                    dragText: t('pim.dataport.change-order-hint')
                }
            }
        });

        // Panel für Importkonfiguration
        this.sourceConfigBasePanel = new Ext.form.FormPanel({
            border: false,
            padding: 5,
            height: 160,
            defaults: {
                width: 600
            },
            labelWidth: 300,
            items: [{
                xtype: 'textfield',
                name: 'file',
                fieldLabel: t('pim.dataport.file.xml'),
                value: this.getSourceconfig('file')
            }, {
                xtype: 'textfield',
                name: 'assetSource',
                fieldLabel: t('pim.dataport.csv.assetSource'),
                value: this.getSourceconfig('assetSource')
            }, {
                xtype: 'textfield',
                name: 'itemxpath',
                fieldLabel: t('pim.dataport_xml_itemxpath'),
                value: this.getSourceconfig('itemxpath')
            }]
        });

        this.sourceConfigPanel.add(this.sourceConfigBasePanel, this.sourceConfigRawitemPanel);

        pimcore.layout.refresh();
    },


    renderCsvSourceConfigPanel: function () {
        var config = this.dataport.sourceconfig;
        var additionalColumns = this.getColumnConfigForSourcetype('csv');

        // Rohdatenfeldkonfiguration
        if (typeof CsvImportconfigModel == 'undefined') {
            var readerFields = [
                {name: 'fieldNo', allowBlank: false, type: 'integer'},
                {name: 'name', allowBlank: false}
            ];

            Ext.each(additionalColumns, function (column) {
                readerFields.push({name: column.dataIndex});
            });

            readerFields.push({name: 'data1'});
            readerFields.push({name: 'data2'});

            Ext.define('CsvImportconfigModel', {
                    extend: 'Ext.data.Model',
                    fields: readerFields,
                    idProperty: 'fieldNo',
                    root: 'fields'
                }
            );
        }

        var store = new Ext.data.JsonStore({
            model: 'CsvImportconfigModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/importconfig/get-rawitemfield-config/' + this.dataport.id,
                reader: {
                    type: 'json',
                    rootProperty: 'fields'
                }
            },
            autoLoad: true
        });

        var columnConfig = [];
        columnConfig.push({
            header: "#",
            width: 40,
            fixed: true,
            dataIndex: 'fieldNo',
            editor: {
                xtype: 'numberfield',
                allowBlank: false
            },
            sortable: false
        });

        columnConfig.push({
            header: t('pim.dataport-fields-name'),
            dataIndex: 'name',
            flex: 2,
            width: 'auto',
            editor: {
                xtype: 'textfield',
                allowBlank: false
            },
            sortable: false
        });

        Ext.each(additionalColumns, function (column) {
            columnConfig.push(column);
        });

        columnConfig.push({header: t('pim.dataport-fields-ex1'), dataIndex: 'data1', flex: 1, sortable: false});
        columnConfig.push({header: t('pim.dataport-fields-ex2'), dataIndex: 'data2', flex: 1, sortable: false});

        var rowEditing = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToMoveEditor: 1,
            autoCancel: true
        });

        this.sourceConfigRawitemPanel = Ext.create('Ext.grid.Panel', {
            title: '',
            store: store,
            columns: columnConfig,
            flex: 1,
            border: false,
            frame: false,
            autoScroll: true,
            layout: 'fit',
            minHeight:300,
            dockedItems: [{
                xtype: 'toolbar',
                dock: 'top',
                items: [
                    {
                        text: t('add'),
                        handler: this.onAddDataport.bind(this),
                        iconCls: "pimcore_icon_add"
                    },
                    '-',
                    {
                        text: t('delete'),
                        handler: this.onDeleteDataport.bind(this),
                        iconCls: "pimcore_icon_delete"
                    }]

            }],
            listeners: {
                edit: function (editor, e) {
                    this.onAddDataport();
                }.bind(this)
            },
            plugins: [rowEditing],
            viewConfig: {
                plugins: {
                    ptype: 'gridviewdragdrop',
                    dragText: t('pim.dataport.change-order-hint')
                }
            }
        });

        // Panel für Importkonfiguration
        this.sourceConfigBasePanel = new Ext.form.FormPanel({
            border: false,
            padding: 5,
            height: 240,
            defaults: {
                width: 600
            },
            labelWidth: 300,
            items: [
                {
                    xtype: 'textfield',
                    name: 'file',
                    fieldLabel: t('pim.dataport.file.csv'),
                    value: this.getSourceconfig('file')
                },
                {
                    xtype: 'textfield',
                    name: 'assetSource',
                    fieldLabel: t('pim.dataport.csv.assetSource'),
                    value: this.getSourceconfig('assetSource')
                },
                {
                    xtype: 'checkbox',
                    name: 'hasHeader',
                    fieldLabel: t('pim.dataport.csv.hasHeader'),
                    value: this.getSourceconfig('hasHeader'),
                },
                {
                    xtype: 'textfield',
                    name: 'separator',
                    fieldLabel: t('pim.dataport.csv.separator'),
                    value: this.getSourceconfig('separator'),
                    allowBlank: false
                },
                {
                    xtype: 'textfield',
                    name: 'quote',
                    fieldLabel: t('pim.dataport.csv.quote'),
                    value: this.getSourceconfig('quote'),
                    allowBlank: false
                }
            ]
        });

        this.sourceConfigPanel.add(this.sourceConfigBasePanel, this.sourceConfigRawitemPanel);

        pimcore.layout.refresh();
    },

    renderVariantSourceConfigPanel: function () {
        var config = this.dataport.sourceconfig;

        // Rohdatenfeldkonfiguration
        if (typeof VariantImportconfigModel == 'undefined') {
            var readerFields = [
                {name: 'fieldNo', allowBlank: false, type: 'integer'},
                {name: 'name', allowBlank: false},
                {name: 'type', allowBlank: false}
            ];

            Ext.define('VariantImportconfigModel', {
                    extend: 'Ext.data.Model',
                    fields: readerFields,
                    idProperty: 'fieldNo',
                    root: 'fields'
                }
            );
        }

        var store = new Ext.data.JsonStore({
            model: 'VariantImportconfigModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/importconfig/get-rawitemfield-config/' + this.dataport.id,
                reader: {
                    type: 'json',
                    rootProperty: 'fields'
                }
            },
            autoLoad: true
        });

        var columnConfig = [];
        columnConfig.push({
            header: "#",
            width: 40,
            fixed: true,
            dataIndex: 'fieldNo',
            editor: {
                xtype: 'numberfield',
                allowBlank: false
            },
            sortable: false
        });

        columnConfig.push({
            header: t('pim.dataport-fields-name'),
            dataIndex: 'name',
            flex: 2,
            width: 'auto',
            editor: {
                xtype: 'textfield',
                allowBlank: false
            },
            sortable: false
        });

        columnConfig.push({
            header: 'type',
            dataIndex: 'type',
            flex: 2,
            width: 'auto',
            editor: new Ext.form.ComboBox({
                displayField:"name",
                valueField: "name",
                store: [
                    ['textfield', 'Text'],
                    ['numberfield', 'Number'],
                ],
                triggerAction: 'all',
                queryMode: 'local',
                typeAhead: true,
                allowBlank: false
            })
        });

        var rowEditing = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToMoveEditor: 1,
            autoCancel: true
        });

        this.sourceConfigRawitemPanel = Ext.create('Ext.grid.Panel', {
            title: '',
            store: store,
            columns: columnConfig,
            flex: 1,
            border: false,
            frame: false,
            autoScroll: true,
            layout: 'fit',
            minHeight:300,
            dockedItems: [{
                xtype: 'toolbar',
                dock: 'top',
                items: [
                    {
                        text: t('add'),
                        handler: this.onAddDataport.bind(this),
                        iconCls: "pimcore_icon_add"
                    },
                    '-',
                    {
                        text: t('delete'),
                        handler: this.onDeleteDataport.bind(this),
                        iconCls: "pimcore_icon_delete"
                    }]
            }],
            listeners: {
                edit: function (editor, e) {
                    this.onAddDataport();
                }.bind(this)
            },
            plugins: [rowEditing],
            viewConfig: {
                plugins: {
                    ptype: 'gridviewdragdrop',
                    dragText: t('pim.dataport.change-order-hint')
                }
            }
        });

        // Panel für Importkonfiguration
        this.sourceConfigBasePanel = new Ext.form.FormPanel({
            border: false,
            padding: 5,
            height: 10,
            defaults: {
                width: 600
            },
            labelWidth: 300,
            items: []
        });

        this.sourceConfigPanel.add(this.sourceConfigBasePanel, this.sourceConfigRawitemPanel);

        pimcore.layout.refresh();
    },

    renderExcelSourceConfigPanel: function() {
        var config = this.dataport.sourceconfig;
        var additionalColumns = this.getColumnConfigForSourcetype('excel');

        // Rohdatenfeldkonfiguration
        if (typeof ExcelImportconfigModel == 'undefined') {
            var readerFields = [
                {name: 'fieldNo', allowBlank: false, type: 'integer'},
                {name: 'name', allowBlank: false}
            ];

            Ext.each(additionalColumns, function (column) {
                readerFields.push({name: column.dataIndex});
            });

            readerFields.push({name: 'data1'});
            readerFields.push({name: 'data2'});

            Ext.define('ExcelImportconfigModel', {
                    extend: 'Ext.data.Model',
                    fields: readerFields,
                    idProperty: 'fieldNo',
                    root: 'fields'
                }
            );
        }

        var store = new Ext.data.JsonStore({
            model: 'ExcelImportconfigModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/importconfig/get-rawitemfield-config/' + this.dataport.id,
                reader: {
                    type: 'json',
                    rootProperty: 'fields'
                }
            },
            autoLoad: true
        });

        var columnConfig = [];
        columnConfig.push({
            header: "#",
            width: 40,
            fixed: true,
            dataIndex: 'fieldNo',
            editor: {
                xtype: 'numberfield',
                allowBlank: false
            }
        });

        columnConfig.push({
            header: t('pim.dataport-fields-name'),
            dataIndex: 'name',
            flex: 2,
            width: 'auto',
            editor: {
                xtype: 'textfield',
                allowBlank: false
            }
        });

        Ext.each(additionalColumns, function(column) {
            columnConfig.push(column);
        });

        columnConfig.push({ header: t('pim.dataport-fields-ex1'), dataIndex: 'data1'});
        columnConfig.push({ header: t('pim.dataport-fields-ex2'), dataIndex: 'data2'});

        var rowEditing = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToMoveEditor: 1,
            autoCancel: true
        });

        this.sourceConfigRawitemPanel = Ext.create('Ext.grid.Panel', {
            title: '',
            store: store,
            columns: columnConfig,
            flex: 1,
            border: false,
            frame: false,
            autoScroll: true,
            layout: 'fit',
            minHeight:300,
            dockedItems: [{
                xtype: 'toolbar',
                dock: 'top',
                items: [
                    {
                        text: t('add'),
                        handler: this.onAddDataport.bind(this),
                        iconCls: "pimcore_icon_add"
                    },
                    '-',
                    {
                        text: t('delete'),
                        handler: this.onDeleteDataport.bind(this),
                        iconCls: "pimcore_icon_delete"
                    }
                ]
            }],
            listeners: {
                edit: function (editor, e) {
                    this.onAddDataport();
                }.bind(this)
            },
            plugins: [rowEditing],
            viewConfig: {
                plugins: {
                    ptype: 'gridviewdragdrop',
                    dragText: t('pim.dataport.change-order-hint')
                }
            }
        });

        // Panel für Importkonfiguration
        this.sourceConfigBasePanel = new Ext.form.FormPanel({
            border: false,
            padding: 5,
            height: 240,
            defaults: {
                width: 600
            },
            labelWidth: 300,
            items: [
                {
                    xtype: 'textfield',
                    name: 'file',
                    fieldLabel: t('pim.dataport.file.excel'),
                    value: this.getSourceconfig('file')
                },
                {
                    xtype: 'textfield',
                    name: 'assetSource',
                    fieldLabel: t('pim.dataport.excel.assetSource'),
                    value: this.getSourceconfig('assetSource')
                },
                {
                    xtype: 'checkbox',
                    name: 'hasHeader',
                    fieldLabel: t('pim.dataport.excel.hasHeader'),
                    value: true,
                    checked: this.getSourceconfig('hasHeader')
                },
                {
                    xtype: 'textfield',
                    name: 'sheet',
                    fieldLabel: t('pim.dataport.excel.sheet'),
                    value: this.getSourceconfig('sheet')
                }
            ]
        });

        this.sourceConfigPanel.add(this.sourceConfigBasePanel, this.sourceConfigRawitemPanel);

        pimcore.layout.refresh();
    },

    renderPimcoreSourceConfigPanel: function () {
        var additionalColumns = this.getColumnConfigForSourcetype('pimcore');

        // Rohdatenfeldkonfiguration
        if (typeof PimcoreImportconfigModel == 'undefined') {
            var readerFields = [
                {name: 'fieldNo', allowBlank: false, type: 'integer'},
                {name: 'name', allowBlank: false},
                {name: 'method', allowBlank: false},
            ];

            Ext.each(additionalColumns, function (column) {
                readerFields.push({name: column.dataIndex});
            });

            readerFields.push({name: 'data1'});
            readerFields.push({name: 'data2'});

            Ext.define('PimcoreImportconfigModel', {
                extend: 'Ext.data.Model',
                fields: readerFields,
                idProperty: 'fieldNo',
                root: 'fields'
            });
        }

        var store = new Ext.data.JsonStore({
            model: 'PimcoreImportconfigModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/importconfig/get-rawitemfield-config/' + this.dataport.id,
                reader: {
                    type: 'json',
                    rootProperty: 'fields'
                }
            },
            autoLoad: true
        });

        var dataPortStore = Ext.create('Ext.data.JsonStore', {
            model: 'DataPortModel',
            proxy: {
                type: 'ajax',
                url: '/admin/BlackbitPim/importconfig/get-pimcore-classes',
                reader: {
                    type: 'json',
                    rootProperty: 'classes'
                }
            },
            autoLoad: true
        });

        var fieldStore = new Ext.data.JsonStore({
            proxy: {
                type: 'ajax',
                url: '',
                reader: {
                    type: 'json',
                    rootProperty: 'methods'
                }
            },
            fields: ['name'],
            autoLoad: true
        });

        // Panel für Importkonfiguration
        this.sourceConfigBasePanel = new Ext.form.FormPanel({
            border: false,
            padding: 5,
            height: 240,
            defaults: {
                width: 600
            },
            labelWidth: 300,
            items: [
                {
                    xtype: 'combo',
                    name: 'sourceClass',
                    fieldLabel: t('pim.dataport_sourceClass'),
                    selectOnFocus: true,
                    mode: 'local',
                    lazyInit: false,
                    forceSelection: true,
                    editable: false,
                    triggerAction: 'all',
                    store: dataPortStore,
                    valueField: 'id',
                    displayField: 'name',
                    value: this.getSourceconfig('sourceClass') || this.dataportPanel.getForm().getFieldValues().targetclass,
                    listeners: {
                        afterrender: function() {
                            fieldStore.getProxy().setUrl('/admin/BlackbitPim/importconfig/get-source-class-methods/' + this.value);
                            fieldStore.load();
                        },
                        change: function() {
                            fieldStore.getProxy().setUrl('/admin/BlackbitPim/importconfig/get-source-class-methods/' + this.value);
                            fieldStore.load();
                        }
                    }
                }
            ]
        });

        var columnConfig = [];
        columnConfig.push({
            header: "#",
            width: 40,
            fixed: true,
            dataIndex: 'fieldNo',
            editor: {
                xtype: 'numberfield',
                allowBlank: false
            }
        });

        columnConfig.push({
            header: t('pim.dataport-fields-name'),
            width: 'auto',
            dataIndex: 'name',
            flex: 2,
            editor: {
                xtype: 'textfield'
            }
        });

        columnConfig.push({
            header: t('pim.dataport.pimcore.fields.method'),
            dataIndex: 'method',
            flex: 2,
            width: 'auto',
            editor: new Ext.form.ComboBox({
                displayField:"name",
                valueField: "name",
                store: fieldStore,
                triggerAction: 'all',
                queryMode: 'local',
                typeAhead: true,
                allowBlank: false
            })
        });

        Ext.each(additionalColumns, function (column) {
            columnConfig.push(column);
        });

        columnConfig.push({header: t('pim.dataport-fields-ex1'), dataIndex: 'data1', flex: 1});
        columnConfig.push({header: t('pim.dataport-fields-ex2'), dataIndex: 'data2', flex: 1});

        var rowEditing = Ext.create('Ext.grid.plugin.RowEditing', {
            clicksToMoveEditor: 1,
            autoCancel: true
        });

        this.sourceConfigRawitemPanel = Ext.create('Ext.grid.Panel', {
            title: '',
            store: store,
            columns: columnConfig,
            flex: 1,
            border: false,
            frame: false,
            autoScroll: true,
            layout: 'fit',
            minHeight:300,
            dockedItems: [{
                xtype: 'toolbar',
                dock: 'top',
                items: [
                    {
                        text: t('add'),
                        handler: this.onAddDataport.bind(this),
                        iconCls: "pimcore_icon_add"
                    },
                    '-',
                    {
                        text: t('delete'),
                        handler: this.onDeleteDataport.bind(this),
                        iconCls: "pimcore_icon_delete"
                    }
                ]
            }],
            listeners: {
                edit: function (editor, e) {
                    this.onAddDataport();
                }.bind(this)
            },
            plugins: [rowEditing],
            viewConfig: {
                plugins: {
                    ptype: 'gridviewdragdrop',
                    dragText: t('pim.dataport.change-order-hint')
                }
            }
        });

        this.sourceConfigPanel.add(this.sourceConfigBasePanel, this.sourceConfigRawitemPanel);

        pimcore.layout.refresh();
    },

    renderBmecatSourceConfigPanel: function () {
        var config = this.dataport.sourceconfig,
            self = this;

        var fieldcollectionStore = new Ext.data.JsonStore({
            url: '/admin/BlackbitPim/importconfig/get-fieldcollections-in-object',
            root: 'elements',
            idProperty: 'id',
            fields: ['id', 'name']
        });

        var multihrefStore = new Ext.data.JsonStore({
            url: '/admin/BlackbitPim/importconfig/get-multihrefs-in-object',
            root: 'elements',
            idProperty: 'id',
            fields: ['id', 'name']
        });

        var nameStore = new Ext.data.JsonStore({
            url: '/admin/BlackbitPim/importconfig/get-inputfields-in-object',
            root: 'elements',
            idProperty: 'id',
            fields: ['id', 'name']
        });

        var fieldcollectionProductStore = new Ext.data.JsonStore({
            url: '/admin/BlackbitPim/importconfig/get-fieldcollections-in-object',
            root: 'elements',
            idProperty: 'id',
            fields: ['id', 'name']
        });

        // Panel für Importkonfiguration
        this.sourceConfigBasePanel = new Ext.form.FormPanel({
            border: false,
            padding: 5,
            height: 300,
            defaults: {
                width: 300
            },
            labelWidth: 200,
            listeners: {
                afterrender: function () {
                    pimcore.layout.refresh();
                }
            },
            items: [{
                xtype: 'textfield',
                name: 'file',
                fieldLabel: t('pim.dataport.file.bmecat'),
                value: this.getSourceconfig('file')
            }, {
                xtype: 'combo',
                hiddenName: 'version',
                name: 'version',
                fieldLabel: t('pim.dataport.bmecat.version'),
                forceSelection: true,
                lazyInit: false,
                editable: false,
                allowBlank: false,
                triggerAction: 'all',
                store: [
                    ['1.2', '1.2'],
                    ['2005', '2005']
                ],
                value: this.getSourceconfig('version')
            }, {
                xtype: 'textfield',
                name: 'assetSource',
                fieldLabel: t('pim.dataport.bmecat.assetSource'),
                value: this.getSourceconfig('assetSource')
            }, {
                xtype: 'combo',
                hiddenName: 'categoryClassId',
                name: 'categoryClassId',

                fieldLabel: t('pim.dataport.bmecat.categoryClass'),
                forceSelection: true,
                lazyInit: false,
                editable: false,
                triggerAction: 'all',
                store: new Ext.data.JsonStore({
                    url: '/admin/BlackbitPim/importconfig/get-pimcore-classes',
                    root: 'classes',
                    idProperty: 'id',
                    fields: ['id', 'name']
                }),
                valueField: 'id',
                displayField: 'name',
                listeners: {
                    beforerender: function (combo) {
                        combo.getStore().load();
                    },
                    select: function (combo, record, index) {
                        multihrefStore.setBaseParam('classId', record.id);
                        multihrefStore.load();

                        fieldcollectionStore.setBaseParam('classId', record.id);
                        fieldcollectionStore.load();

                        nameStore.setBaseParam('classId', record.id);
                        nameStore.load();
                    }
                },
                value: this.getSourceconfig('categoryClassId')
            }, {
                xtype: 'combo',
                hiddenName: 'categoryNameElement',
                fieldLabel: t('pim.dataport.bmecat.categoryClass.name'),
                forceSelection: true,
                lazyInit: false,
                editable: false,
                triggerAction: 'all',
                store: nameStore,
                valueField: 'id',
                displayField: 'name',
                listeners: {
                    afterrender: function (combo) {
                        combo.getStore().setBaseParam('classId', self.getSourceconfig('categoryClassId'));
                        combo.getStore().load();
                    }
                },
                value: this.getSourceconfig('categoryNameElement')
            }, {
                xtype: 'combo',
                hiddenName: 'categoryProductElement',
                fieldLabel: t('pim.dataport.bmecat.categoryClass.productHref'),
                forceSelection: true,
                lazyInit: false,
                editable: false,
                triggerAction: 'all',
                store: multihrefStore,
                valueField: 'id',
                displayField: 'name',
                listeners: {
                    afterrender: function (combo) {
                        combo.getStore().setBaseParam('classId', self.getSourceconfig('categoryClassId'));
                        combo.getStore().load();
                    }
                },
                value: this.getSourceconfig('categoryProductElement')
            }, {
                xtype: 'combo',
                hiddenName: 'categoryAttributeElement',
                fieldLabel: t('pim.dataport.bmecat.categoryClass.attributeHref'),
                forceSelection: true,
                lazyInit: false,
                editable: false,
                triggerAction: 'all',
                store: fieldcollectionStore,
                valueField: 'id',
                displayField: 'name',
                listeners: {
                    afterrender: function (combo) {
                        combo.getStore().setBaseParam('classId', self.getSourceconfig('categoryClassId'));
                        combo.getStore().load();
                    }
                },
                value: this.getSourceconfig('categoryAttributeElement')
            }, {
                xtype: 'combo',
                hiddenName: 'itemAttributeElement',
                fieldLabel: t('pim.dataport.bmecat.categoryClass.itemAttributeElement'),
                forceSelection: true,
                lazyInit: false,
                editable: false,
                triggerAction: 'all',
                store: fieldcollectionProductStore,
                valueField: 'id',
                displayField: 'name',
                listeners: {
                    afterrender: function (combo) {
                        combo.getStore().setBaseParam('classId', self.dataportPanel.getForm().getFieldValues().targetclass);
                        combo.getStore().load();
                    }
                },
                value: this.getSourceconfig('itemAttributeElement')
            }, {
                xtype: 'textfield',
                name: 'categoryfolder',
                fieldLabel: t('pim.dataport.bmecat.categoryFolder'),
                cls: "input_drop_target",
                listeners: {
                    render: function (el) {
                        var dd = new Ext.dd.DropZone(el.getEl().dom.parentNode.parentNode, {
                            ddGroup: "element",

                            getTargetFromEvent: function (e) {
                                return this.getEl();
                            },

                            onNodeOver: function (target, dd, e, data) {
                                if ("object" == data.node.attributes.elementType && "folder" == data.node.attributes.type) {
                                    return Ext.dd.DropZone.prototype.dropAllowed;
                                }
                                return Ext.dd.DropZone.prototype.dropNotAllowed;
                            },

                            onNodeDrop: function (target, dd, e, data) {
                                if ("object" == data.node.attributes.elementType && "folder" == data.node.attributes.type) {
                                    this.setValue(data.node.attributes.path);
                                    return true;
                                }
                                return false;
                            }.bind(this)
                        });
                    }
                },
                value: this.getSourceconfig('categoryfolder')
            }, {
                xtype: 'button',
                text: t('pim.dataport.bmecat.startImport'),
                width: 200,
                padding: '20px 0 0',
                listeners: {
                    'click': function (e) {
                        this.save(function (e) {
                            if (e) {
                                console.log('Fehler beim starten des Imports');
                            } else {

                                var bmecatPanel = null;

                                this.items.each(function (child, index, total) {
                                    if (child instanceof Pim.BmecatImport && child.dataportId == self.dataport.id) {
                                        bmecatPanel = child;
                                    }
                                });
                                console.log('Wuff', bmecatPanel);

                                if (!bmecatPanel) {
                                    var tab = new Pim.BmecatImport({
                                        dataportId: this.dataport.id
                                    });
                                    this.add(tab);
                                    this.setActiveTab(tab);
                                } else {
                                    this.setActiveTab(bmecatPanel);
                                }
                            }
                        }.bind(self));
                    }.bind(self)
                }
            }]
        });

        this.sourceConfigPanel.add(this.sourceConfigBasePanel);

        pimcore.layout.refresh();
    },

    onAddDataport: function () {
        if (this.sourceConfigRawitemPanel instanceof Ext.grid.Panel) {
            var store = this.sourceConfigRawitemPanel.getStore();

            var lastRecord = this.sourceConfigRawitemPanel.getStore().getAt(this.sourceConfigRawitemPanel.getStore().getCount()-1);

            if(!lastRecord || lastRecord.get('name')) {
                var maxNum = 0;
                if(lastRecord) {
                    maxNum = lastRecord.get('fieldNo');
                }
                var u = new store.model();
                u.set('fieldNo', maxNum + 1);

                store.add(u);
            }
        }
    },

    onDeleteDataport: function () {
        if (this.sourceConfigRawitemPanel instanceof Ext.grid.Panel) {
            var rec = this.sourceConfigRawitemPanel.getSelectionModel().getSelection()[0];
            if (!rec) {
                return false;
            }

            Ext.Msg.confirm(
                t("Are you sure?"),
                t("Are you sure?"),
                function (button) {
                    if (button == 'yes') {
                        this.sourceConfigRawitemPanel.getStore().remove(rec);
                        this.save();
                    }
                }.bind(this)
            );
        }
    },

    getSourceconfig: function (key) {
        if (this.dataport.sourceconfig && key in this.dataport.sourceconfig) {
            return this.dataport.sourceconfig[key];
        }

        return null;
    },

    getColumnConfigForSourcetype: function (sourcetype) {
        var columns = [];

        switch (sourcetype) {
            case 'xml':
                columns.push({
                    header: t('pim.dataport-fields-xpath'),
                    dataIndex: 'xpath',
                    editor: new Ext.form.TextField({allowBlank: false}),
                    flex: 2,
                    sortable: false
                });
                break;
            case 'csv':
                columns.push({
                    header: t('pim.dataport.csv.fields.name'),
                    dataIndex: 'column',
                    editor: new Ext.form.TextField({allowBlank: false}),
                    flex: 2,
                    sortable: false
                });
                break;
            case 'excel':
                columns.push({
                    header: t('pim.dataport.excel.fields.name'),
                    dataIndex: 'column',
                    editor: new Ext.form.TextField({allowBlank: false}),
                    flex: 2,
                    sortable: false
                });
                break;
            case 'pimcore':
                columns.push({
                    header: t('pim.dataport.pimcore.fields.parameters'),
                    dataIndex: 'parameters',
                    editor: new Ext.form.TextField({allowBlank: true}),
                    flex: 2,
                    sortable: false
                });
                break;
            case 'variant':
                break;
        }

        return columns;
    },

    save: function (cb) {
        if (!Ext.isFunction(cb)) {
            cb = function () {
            };
        }

        if (this.dataportPanel.getForm().isValid() && this.sourceConfigBasePanel.getForm().isValid()) {
            this.getEl().mask();
            var baseData = this.dataportPanel.getForm().getFieldValues();
            var sourceData = Ext.util.JSON.encode(this.sourceConfigBasePanel.getForm().getFieldValues());

            var gridData = [];
            if (this.sourceConfigRawitemPanel instanceof Ext.grid.Panel) {
                this.sourceConfigRawitemPanel.getStore().each(function (record) {
                    var data = {};
                    var f = record.fields;

                    for (var i = 0, len = f.length; i < len; i++) {
                        var field = f[i];
                        data[field.name] = record.get(field.name);
                    }
                    gridData.push(data);
                });
            }

            var rawitemData = Ext.util.JSON.encode(gridData);

            baseData['id'] = this.dataport.id;
            baseData['sourceData'] = sourceData;
            baseData['rawitemData'] = rawitemData;

            Ext.Ajax.request({
                url: "/admin/BlackbitPim/importconfig/update",
                method: 'post',
                params: baseData,
                success: function (response) {
                    response = Ext.decode(response.responseText);

                    if (!(response && response.success)) {
                        this.getEl().unmask();
                        pimcore.helpers.showNotification(t("error"), t(response.errorMessage), "error");

                        cb(new Error(t(response.errorMessage)));
                    } else {
                        this.populateDataportForm();
                        this.previewPanel.rebuild();
                        cb();
                    }
                }.bind(this),
                failure: function (response) {
                    this.getEl().unmask();
                    response = Ext.decode(response.responseText);

                    pimcore.helpers.showNotification(t("error"), t(response.errorMessage), "error");
                    cb(new Error(t(response.errorMessage)));
                }.bind(this)
            });
        }
    }
});
