<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim\Item\Bmecat;

use Blackbit\PimBundle\Tools\Installer;
use Blackbit\PimBundle\lib\Pim\Feature;
use Blackbit\PimBundle\lib\Pim\Helper;
use Blackbit\PimBundle\model\CategoryLog;
use Blackbit\PimBundle\model\FeatureLog;
use Blackbit\PimBundle\model\ImportStatus;
use Blackbit\PimBundle\model\ProductLog;
use Pimcore\Cache;
use Pimcore\Db;
use Pimcore\Logger;
use Pimcore\Model\Asset;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\ClassDefinition;
use Pimcore\Model\DataObject\Concrete;

/**
 * Importiert eine BMEcat-Datei in Produktobjekte
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Medien GmbH, http://www.blackbit.de/
 */
class Importer {
    /** @var array */
	private $dataport;
	private $file;
	private $preJs;

	private $xp;
	private $xpathExpr;

	private $itemAttributeElement;

	private $itemClassName;
	private $referencesList;

	private $idPrefix = '';

	private $debug = false;

	/**
	 * @var string Identifier for current import process
	 */
	private $importKey;
	
	
	public function __construct($dataport, $file, $preJs = '') {
		Feature::checkFeature(Feature::BMECAT);

		$this->dataport = $dataport;
		$this->file = $file;
		$this->preJs = $preJs;

		$this->referencesList = array();
		AbstractObject::setHideUnpublished(false);

		$this->debug = \Pimcore::inDebugMode();
	}

	/**
	 * Importiert alle vorhandenen Rohdatensätze in die dazugehörigen Artikel und Produkte. Nicht vorhandene Artikel
	 * und Produkte werden angelegt.
	 */
	public function import() {
		$sourceconfig = unserialize($this->dataport['sourceconfig']);
		$targetconfig = unserialize($this->dataport['targetconfig']);

		if (array_key_exists('idPrefix', $targetconfig) && !empty($targetconfig['idPrefix'])) {
			$this->idPrefix = $targetconfig['idPrefix'];
		}

		// General config
		// Quelldatei einlesen
		$dom = Helper::getDomDocument($this->file);
		if (!($dom instanceof \DOMDocument)) {
			return;
		}
		$this->xp = new \DOMXPath($dom);

		$version = null;
		if (array_key_exists('version', $sourceconfig)) {
			$version = $sourceconfig['version'];
			$this->xpathExpr = Helper::getBmecatXpathForVersion($version);
		} else {
			Logger::error("Version of BMECat not set.");
			return;
		}

		if (empty($this->xpathExpr)) {
			Logger::error('Invalid BMEcat version "' . $version . '"');
			return;
		}

		$nodes = $this->xp->query(implode('/', array($this->xpathExpr['category']['base'], $this->xpathExpr['category']['root'])));

		// Item config
		$itemAttributeElement = $sourceconfig['itemAttributeElement'];
		$this->itemAttributeElement = $itemAttributeElement;


		$itemClassId = $targetconfig['itemClass'];
		$itemClass = ClassDefinition::getById($itemClassId);
		if (!($itemClass instanceof ClassDefinition)) {
			Logger::error("No class found for Item with id " . $itemClassId);
			return;
		}
		$this->itemClassName = $itemClassName = $itemClass->getName();


		$itemFolderPath = $targetconfig['itemFolder'];
		$itemFolder = Concrete::getByPath($itemFolderPath);
		if (!($itemFolder instanceof Concrete)) {
			$itemFolder = Concrete::getById(1);
		}
		$itemFolder = $itemFolder->getId();

		$assetFolder = Asset::getByPath($targetconfig['assetFolder']);
		if (!($assetFolder instanceof Asset\Folder)) {
			$assetFolder = Asset::getById(1);
		}

		$assetSource = $sourceconfig['assetSource'];

		// Category config
		$categoryClassId = $sourceconfig['categoryClassId'];
		$categoryProductElement = $sourceconfig['categoryProductElement'];
		$categoryAttributeElementName = $sourceconfig['categoryAttributeElement'];

		$categoryClass = ClassDefinition::getById($categoryClassId);
		if (!($categoryClass instanceof ClassDefinition)) {
			Logger::error("No class found for category with id " . $categoryClassId);
			return;
		}
		$categoryClassName = $categoryClass->getName();

		// Artikel der Quelldatei auslesen
		$sourceProducts = $this->getProductsFromFile($this->xp);

		$total = $sourceProducts->length;


		$dbNow = new \DateTime();
		$statusModel = new ImportStatus();
		$this->importKey = uniqid();

		$statusModel->create(array(
			'key' => $this->importKey,
			'dataportId' => $this->dataport['id'],
			'startDate' => $dbNow,
			'lastUpdate' => $dbNow,
			'importType' => Importstatus::TYPE_COMPLETE_PIM,
			'totalItems' => $total,
		));

		$categoryLogModel = new CategoryLog();
		$productLogModel = new ProductLog();

		$fileHash = md5_file($this->file);

		$categoryLogData = Helper::extractCategories($fileHash);
		$productLogData = Helper::extractProducts($fileHash, $version);

		foreach ($categoryLogData as $d) {
			$categoryLogModel->create(array(
				'importKey' => $this->importKey,
				'categoryId' => $d['id'],
				'hash' => $d['hash'],
			));
		}

		foreach ($productLogData['products'] as $d) {
			$productLogModel->create(array(
				'importKey' => $this->importKey,
				'productId' => $d['id'],
				'hash' => $d['hash'],
			));
		}


		$db = Db::get();
		foreach ($productLogData['features'] as $d) {
			$sql = 'REPLACE INTO ' . Installer::TABLE_BME_IMPORTLOG_FEATURES . ' (`importKey`, `name`) VALUES (?, ?)';
			$db->query($sql, array($this->importKey, $d));
		}

		if ($this->debug) {
			ini_set('display_errors', 1);
		}


		$doneItems = 0;
		$importResult = array();
		$client = new \GearmanClient();
		$client->addServer();
		$client->setCompleteCallback(function(\GearmanTask $task) use (&$importResult, &$doneItems, $statusModel, $dbNow, $total) {
			$importResult[$task->unique()] = array(
				'handle' => $task->jobHandle(),
				'data' => unserialize($task->data()),
			);

			$doneItems++;
			if ($doneItems % 100 == 0) {
				/**
				 * Ebenfalls Status und Endzeitpunkt zurücksetzen, falls dies zwischenzeitlich durch den Maintenance-Job als
				 * abgebrochener Prozess markiert wurde
				 */
				$statusModel->update(array(
					'status' => Importstatus::STATUS_RUNNING,
					'endDate' => null,
					'lastUpdate' => $dbNow,
					'doneItems' => $doneItems,
				), ['key' => $this->importKey]);

				if ($this->debug) {
					$this->debug('Done: ' . $doneItems . '/' . $total);
				}

				\Pimcore::collectGarbage();
			}
		});

		$itemCounter = 0;

		/**
		 * Iteriere über alle Artikel der Quelldatei
		 * @var $sourceProduct \DOMNode
		 */
		foreach ($sourceProducts as $sourceProduct) {
			$itemCounter++;

			// Für diese BMEcat eindeutiger Schlüssel des Artikels
			$keyList = $this->xp->query($this->xpathExpr['globalAttributes']['aid']['id'], $sourceProduct);
			if ($keyList->length == 0) {
				Logger::warning('BMECAT data has no SUPPLIER_AID for this item');
				continue;
			}
			$key = $keyList->item(0)->nodeValue;

			if ($this->debug) {
				$this->debug('Adding import task');
			}

			$importWorkload = array(
				'file' => $this->file,
				'version' => $version,
				'importKey' => $this->importKey,
				'sourceProductId' => $key,
				'itemStatusFormula' => $targetconfig['itemStatus'],
				'itemClassId' => $itemClassId,
				'itemClassName' => $itemClassName,
				'itemFolder' => $itemFolder,
				'idPrefix' => $this->idPrefix,
				'dataportId' => $this->dataport['id'],
				'dummyCategory' => $nodes->length == 0,
				'categoryClassName' => $categoryClassName,
				'categoryProductElement' => $categoryProductElement,
				'categoryAttributeElementName' => $categoryAttributeElementName,
				'itemAttributeElement' => $itemAttributeElement,
				'assetSource' => $assetSource,
				'assetFolderId' => $assetFolder->getId(),
			);
			$cacheKey = 'blackbit_' . $this->importKey . '_workload_' . uniqid();
			Cache::save($importWorkload, $cacheKey, array('bmecat_import_' . $this->importKey));
			$client->addTask('ImportItem', $cacheKey, null, 't-' . $this->importKey . '-' . $itemCounter);
		}

		$start = microtime(true);
		$client->runTasks();
		$totaltime = number_format(microtime(true) - $start, 2);

		if ($this->debug) {
			$this->debug('Gesamtzeit Artikelimport: ' . $totaltime);
		}

		// Combine results
		$bmeToPimId = array();
		foreach ($importResult as $result) {
			// Mappings BMEcat -> PIM
			if (!$result['data']['success']) {
				continue;
			}

			foreach ($result['data']['bmeToPimId'] as $bmeId => $pimId) {
				if (!array_key_exists($bmeId, $bmeToPimId)) {
					$bmeToPimId[$bmeId] = $pimId;
				}
			}
		}

		// Save BMEtoPIM-Mapping
		Cache::save($bmeToPimId, 'bme2pim_' . $this->importKey, array('bmecat_import_' . $this->importKey));


		$categoryMappings = array();
		$references = array();
		foreach ($importResult as $result) {
			if (!$result['data']['success']) {
				if (!empty($result['data']['message'])) {
					Logger::error($result['data']['message']);
				}
				continue;
			}

			// Kategoriemappings
			foreach ($result['data']['categoryMappings'] as $key => $mapping) {
				$categoryId = substr($key, 3); // Remove prefix "cat"
				if (!array_key_exists($categoryId, $categoryMappings)) {
					$categoryMappings[$categoryId] = array();
				}

				foreach ($mapping as $field => $itemIds) {
					if (!array_key_exists($field, $categoryMappings[$categoryId])) {
						$categoryMappings[$categoryId][$field] = array();
					}

					foreach ($itemIds as $itemId) {
						$categoryMappings[$categoryId][$field][] = $itemId;
					}
				}
			}

			// Referenzen
			foreach ($result['data']['referencesList'] as $ref) {
				$key = 'p' . $ref['artId'];
				if (!array_key_exists($key, $references)) {
					$references[$key] = array(
						'id' => $ref['artId'],
						'references' => array(),
					);
				}

				$fieldKey = $ref['fieldCollection'] . '#' . $ref['attributeName'];
				if (!array_key_exists($fieldKey, $references[$key]['references'])) {
					$references[$key]['references'][$fieldKey] = array(
						'fieldCollection' => $ref['fieldCollection'],
						'attributeName' => $ref['attributeName'],
						'items' => array(),
					);
				}


				foreach ($ref['references'] as $r) {
					if (!in_array($r, $references[$key]['references'][$fieldKey]['items'])) {
						$references[$key]['references'][$fieldKey]['items'][] = $r;
					}
				}
			}
		}

		// Save category mappings
		$totalCategories = count($categoryMappings);

		if ($this->debug) {
			$start = microtime(true);
			$this->debug('Kategorien ... (' . $totalCategories . ')');
		}

		foreach ($categoryMappings as $catId => $mappings){
			if ($this->debug) {
				$this->debug('Adding category mapping task for category ' . $catId);
			}

			$cacheKey = 'blackbit_' . $this->importKey . '_cmapping_' . uniqid();
			Cache::save(array('catId' => $catId, 'mappings' => $mappings), $cacheKey, array('bmecat_import_' . $this->importKey));
			$client->addTask('AssignCategory', $cacheKey);
		}


		// New callback for different kind of task
		$doneCategories = 0;
		$client->setCompleteCallback(function($task) use (&$doneCategories, $statusModel, $dbNow, $totalCategories) {
			$doneCategories++;
			if ($doneCategories % 10 == 0) {
				/**
				 * Ebenfalls Status und Endzeitpunkt zurücksetzen, falls dies zwischenzeitlich durch den Maintenance-Job als
				 * abgebrochener Prozess markiert wurde
				 */
				$statusModel->update(array(
					'status' => Importstatus::STATUS_RUNNING,
					'endDate' => null,
					'lastUpdate' => $dbNow,
				), ['key' => $this->importKey]);

				if ($this->debug) {
					$this->debug('Categories done: ' . $doneCategories . '/' . $totalCategories);
				}

				\Pimcore::collectGarbage();
			}
		});

		$client->runTasks();


		$totalReferences = count($references);
		if ($this->debug) {
			$this->debug('Kategorien gesamt: ' . (microtime(true) - $start));

			$start = microtime(true);
			$this->debug('Referenzen ... (' . $totalReferences . ')');
		}

		foreach ($references as $element) {
			if ($this->debug) {
				$this->debug('Adding reference task for item ' . $element['id']);
			}

			$workload = array(
				'data' => $element,
				'importKey' => $this->importKey,
				'itemAttributeElement' => $itemAttributeElement,
				'itemClassName' => $itemClassName,
				'file' => $this->file,
				'version' => $version,
				'dataportId' => $this->dataport['id'],
				'idPrefix' => $this->idPrefix,
			);

			$cacheKey = 'blackbit_' . $this->importKey . '_references_' . uniqid();
			Cache::save($workload, $cacheKey, array('bmecat_import_' . $this->importKey));
			$client->addTask('UpdateReferences', $cacheKey);
		}

		// New callback for different kind of task
		$doneReferences = 0;
		$client->setCompleteCallback(function($task) use (&$doneReferences, $statusModel, $dbNow, $totalReferences) {
			$doneReferences++;
			if ($doneReferences % 10 == 0) {
				/**
				 * Ebenfalls Status und Endzeitpunkt zurücksetzen, falls dies zwischenzeitlich durch den Maintenance-Job als
				 * abgebrochener Prozess markiert wurde
				 */
				$statusModel->update(array(
					'status' => ImportStatus::STATUS_RUNNING,
					'endDate' => null,
					'lastUpdate' => $dbNow,
				), ['key' => $this->importKey]);

				if ($this->debug) {
					$this->debug('References done: ' . $doneReferences . '/' . $totalReferences);
				}

				\Pimcore::collectGarbage();
			}
		});

		$client->runTasks();

		if ($this->debug) {
			$this->debug('Referenzen gesamt: ' . (microtime(true) - $start));
		}

		Cache::clearTag('bmecat_import_' . $this->importKey);
		\Pimcore::collectGarbage();

		$statusModel->update(array(
			'status' => ImportStatus::STATUS_FINISHED,
			'lastUpdate' => $dbNow,
			'endDate' => $dbNow,
			'doneItems' => $total,
		), ['key' => $this->importKey]);
	}

	/**
	 * @param $xp \DOMXPath
	 * @return \DOMNodeList
	 */
	private function getProductsFromFile($xp) {
		return $xp->query($this->xpathExpr['product']['all']);
	}

	/*
	 * Verarbeitung der JS Konfiguration 
	 */
	public static function processJS($formula, $value, $valueArray, $jsPrefix, $articleData = null, $currentValue = null, $currentObjectValues = null) {
		$result = $value;
		// JS processing
		if (!empty($formula) && class_exists("JSContext")) {
			try {
				$js = new \JSContext();
				$js->assign('value', $value);
				$js->assign('valueArray', $valueArray);
				$js->assign('articleData', $articleData);
				$js->assign('currentValue', $currentValue);
				$js->assign('currentObjectData', $currentObjectValues);

				$modificationDate = 0;
				if ($currentValue instanceof Asset) {
					$modificationDate = $currentValue->getModificationDate();
				}
				$js->assign('modificationDate', $modificationDate);

				$result = $js->evaluateScript($jsPrefix . "\n\n" . $formula);
				if (is_object($result)) {
					$result = self::objectToArray($result);
				}
			} catch (\Exception $ex) {
				Logger::error("Error while calculating attribute value: {$ex}");
			}
		}
		return $result;
	}

	private static function objectToArray($d): array {
		if (is_object($d)) {
			$d = get_object_vars($d);
		}

		if (is_array($d)) {
			return array_map(array('Pim_Item_Bmecat_Importer', 'objectToArray'), $d);
		}
        return $d;
	}

	private function debug($msg) {
		echo $msg . PHP_EOL;
	}
}
