<?php
/**
 * Copyright Blackbit digital Commerce GmbH <info@blackbit.de>
 *
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

namespace Blackbit\PimBundle\lib\Pim;

use Blackbit\PimBundle\Tools\Installer;
use Blackbit\PimBundle\model\BmeCategory;
use Blackbit\PimBundle\model\BmeFile;
use Blackbit\PimBundle\model\BmeProduct;
use Blackbit\PimBundle\model\BmeProductCategory;
use Blackbit\PimBundle\model\CategoryLog;
use Blackbit\PimBundle\model\Categorymapping;
use Blackbit\PimBundle\model\CategorymappingField;
use Blackbit\PimBundle\model\Dataport;
use Blackbit\PimBundle\model\FeatureLog;
use Blackbit\PimBundle\model\Fieldmapping;
use Blackbit\PimBundle\model\ProductLog;
use Pimcore\Db;
use Pimcore\Logger;
use Pimcore\Model\DataObject\ClassDefinition\Data\Input;
use Pimcore\Model\DataObject\AbstractObject;
use Pimcore\Model\DataObject\ClassDefinition;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\DataObject\Fieldcollection;
use Pimcore\Model\DataObject\Folder;
use Pimcore\Tool;
use Pimcore\Translation\Translator;

/**
 *
 *
 * @author Dennis Korbginski <dennis.korbginski@blackbit.de>
 * @copyright Blackbit neue Medien GmbH, http://www.blackbit.de/
 */

class Helper {
    private static $fileHandles = array();
    
    private static $mappingDefaults = array(
        'numeric' => array(
            'decimalSeparator' => '.',
            'groupingSeparator' => ',',
        ),
        'date' => array(
            'dateFormat' => 'd.m.Y'
        ),
        'datetime' => array(
            'dateFormat' => 'd.m.Y H:i:s'
        ),
        'multiselect' => array(
            'separator' => ','
        ),
        'multihref' => array(
            'purgeitems' => false,
            'overwrite' => false
        ),
        'manyToManyRelation' => array(
            'purgeitems' => false,
            'overwrite' => false
        ),
        'image' => array(
            'overwrite' => false
        ),
        'hotspotimage' => array(
            'overwrite' => false
        ),
        'fieldcollections' => array(
            'purgeitems' => false
        ),
        'objects' => array(
            'purgeitems' => false
        ),
        'manyToManyObjectRelation' => array(
            'purgeitems' => false
        ),
        'multihrefMetadata' => array(
            'purgeitems' => false
        ),
        'advancedManyToManyRelation' => array(
            'purgeitems' => false
        ),
        'objectsMetadata' => array(
            'purgeitems' => false
        ),
        'advancedManyToManyObjectRelation' => array(
            'purgeitems' => false
        )
    );
    
    private static $bmecatXpath = array(
        '1.2' => array(
            'validation' => array(
                '/BMECAT/T_NEW_CATALOG/ARTICLE',
            ),
            'info' => array(
                'mime' => '/BMECAT/T_NEW_CATALOG/ARTICLE/MIME_INFO/MIME/MIME_SOURCE',
            ),
            'category' => array(
                'all' => '/BMECAT/T_NEW_CATALOG/CATALOG_GROUP_SYSTEM/CATALOG_STRUCTURE',
                'base' => '/BMECAT/T_NEW_CATALOG/CATALOG_GROUP_SYSTEM',
                'byId' => 'CATALOG_STRUCTURE[GROUP_ID="%s"]',
                'root' => 'CATALOG_STRUCTURE[@type="root"]',
                'child' => 'CATALOG_STRUCTURE[PARENT_ID="%s"]',
                'catId' => 'GROUP_ID',
                'catName' => 'GROUP_NAME',
                'catType' => '@type',
                'parentId' => 'PARENT_ID',
                'parentForCategory' => 'CATALOG_STRUCTURE[GROUP_ID="%s"]/PARENT_ID',
            ),
            'mapping' => array(
                'allProductToCategory' => '/BMECAT/T_NEW_CATALOG/ARTICLE_TO_CATALOGGROUP_MAP',
                'productToCategory' => '/BMECAT/T_NEW_CATALOG/ARTICLE_TO_CATALOGGROUP_MAP[CATALOG_GROUP_ID="%s"]/ART_ID',
                'productToCategories' => '/BMECAT/T_NEW_CATALOG/ARTICLE_TO_CATALOGGROUP_MAP[%s]/ART_ID',
                'categoriesToProduct' => '/BMECAT/T_NEW_CATALOG/ARTICLE_TO_CATALOGGROUP_MAP[ART_ID="%s"]/CATALOG_GROUP_ID',
                'categoryId' => 'CATALOG_GROUP_ID',
                'productId' => 'ART_ID',
            ),
            'product' => array(
                'all' => '/BMECAT/T_NEW_CATALOG/ARTICLE',
                'id' => 'SUPPLIER_AID',
                'byIds' => '/BMECAT/T_NEW_CATALOG/ARTICLE[%s]',
                'byIdsCondition' => 'SUPPLIER_AID="%s"',
                'byId' => '/BMECAT/T_NEW_CATALOG/ARTICLE[SUPPLIER_AID="%s"]',
                'features' => 'ARTICLE_FEATURES/FEATURE',
                'featureByName' => 'ARTICLE_FEATURES/FEATURE[FNAME="%s"]',
                'featureName' => 'FNAME',
                'featureValues' => 'FVALUE',
                'priceDetails' => array (
                    'byType' => 'ARTICLE_PRICE_DETAILS[ARTICLE_PRICE/@price_type=%s]',
                    'startDate' => 'DATETIME[@type="valid_start_date"]/DATE',
                    'endDate' => 'DATETIME[@type="valid_end_date"]/DATE',
                    'priceByType' => 'ARTICLE_PRICE[@price_type=%s]',
                ),
                'prices' => 'ARTICLE_PRICE_DETAILS/ARTICLE_PRICE',
                'pricesByType' => 'ARTICLE_PRICE_DETAILS/ARTICLE_PRICE[@price_type="%s"]',
                'priceType' => '@price_type',
                'priceAmount' => 'PRICE_AMOUNT',
                'priceCurrency' => 'PRICE_CURRENCY',
                'priceTax' => 'TAX',
                'priceFactor' => 'PRICE_FACTOR',
                'priceLowerBound' => 'LOWER_BOUND',
                'priceTerritory' => 'TERRITORY',
                'mimeBase' => 'MIME_INFO/MIME',
                'mimeType' => 'MIME_TYPE',
                'mimeSource' => 'MIME_SOURCE',
                'mimeDescr' => 'MIME_DESCR',
                'mimeAlt' => 'MIME_ALT',
                'mimePurpose' => 'MIME_PURPOSE',
                'mimeByPurpose' => 'MIME_INFO/MIME[MIME_PURPOSE="%s"]',
                'reference' => 'ARTICLE_REFERENCE',
                'referenceType' => '@type',
                'referenceByType' => 'ARTICLE_REFERENCE[@type="%s"]',
                'referenceId' => 'ART_ID_TO',
            ),
            'globalAttributes' => array(
                'aid' => array(
                    'id' => 'SUPPLIER_AID',
                    //					'name' => 'pim.dataport.bmecat.globalattr.erpcode',
                    'name' => 'Artikelnummer',
                    'values' => array(),
                ),
                'shortDescription' => array(
                    'id' => 'ARTICLE_DETAILS/DESCRIPTION_SHORT',
                    //					'name' => 'pim.dataport.bmecat.globalattr.shortDescription',
                    'name' => 'Kurzbeschreibung',
                    'values' => array(),
                ),
                'longDescription' => array(
                    'id' => 'ARTICLE_DETAILS/DESCRIPTION_LONG',
                    //					'name' => 'pim.dataport.bmecat.globalattr.longDescription',
                    'name' => 'Langbeschreibung',
                    'values' => array(),
                ),
                'ean' => array(
                    'id' => 'ARTICLE_DETAILS/EAN',
                    //					'name' => 'pim.dataport.bmecat.globalattr.ean',
                    'name' => 'EAN',
                    'values' => array(),
                ),
                'mnf' => array(
                    'id' => 'ARTICLE_DETAILS/MANUFACTURER_NAME',
                    //					'name' => 'pim.dataport.bmecat.globalattr.manufacturerName',
                    'name' => 'Herstellername',
                    'values' => array(),
                ),
                'mnfAid' => array(
                    'id' => 'ARTICLE_DETAILS/MANUFACTURER_AID',
                    //					'name' => 'pim.dataport.bmecat.globalattr.manufacturerAid',
                    'name' => 'Hersteller-ID',
                    'values' => array(),
                ),
                'orderUnit' => array(
                    'id' => 'ARTICLE_ORDER_DETAILS/ORDER_UNIT',
                    //					'name' => 'pim.dataport.bmecat.globalattr.orderUnit',
                    'name' => 'Bestelleinheit',
                    'values' => array(),
                ),
                'contentUnit' => array(
                    'id' => 'ARTICLE_ORDER_DETAILS/CONTENT_UNIT',
                    //					'name' => 'pim.dataport.bmecat.globalattr.contentUnit',
                    'name' => 'Bestellmenge / Einheit',
                    'values' => array(),
                ),
            ),
        ),
        '2005' => array(
            'validation' => array(
                '/BMECAT/T_NEW_CATALOG/PRODUCT',
            ),
            'info' => array(
                'mime' => '/BMECAT/T_NEW_CATALOG/PRODUCT/MIME_INFO/MIME/MIME_SOURCE',
            ),
            'category' => array(
                'all' => '/BMECAT/T_NEW_CATALOG/CATALOG_GROUP_SYSTEM/CATALOG_STRUCTURE',
                'base' => '/BMECAT/T_NEW_CATALOG/CATALOG_GROUP_SYSTEM',
                'byId' => 'CATALOG_STRUCTURE[GROUP_ID="%s"]',
                'root' => 'CATALOG_STRUCTURE[@type="root"]',
                'child' => 'CATALOG_STRUCTURE[PARENT_ID="%s"]',
                'catId' => 'GROUP_ID',
                'catName' => 'GROUP_NAME',
                'catType' => '@type',
                'parentId' => 'PARENT_ID',
                'parentForCategory' => 'CATALOG_STRUCTURE[GROUP_ID="%s"]/PARENT_ID',
            ),
            'mapping' => array(
                'allProductToCategory' => '/BMECAT/T_NEW_CATALOG/PRODUCT_TO_CATALOGGROUP_MAP',
                'productToCategory' => '/BMECAT/T_NEW_CATALOG/PRODUCT_TO_CATALOGGROUP_MAP[CATALOG_GROUP_ID="%s"]/ART_ID',
                'productToCategories' => '/BMECAT/T_NEW_CATALOG/PRODUCT_TO_CATALOGGROUP_MAP[%s]/ART_ID',
                'categoriesToProduct' => '/BMECAT/T_NEW_CATALOG/PRODUCT_TO_CATALOGGROUP_MAP[ART_ID=%s]/CATALOG_GROUP_ID',
                'categoryId' => 'CATALOG_GROUP_ID',
                'productId' => 'ART_ID',
            ),
            'product' => array(
                'all' => '/BMECAT/T_NEW_CATALOG/PRODUCT',
                'id' => 'SUPPLIER_PID',
                'byIds' => '/BMECAT/T_NEW_CATALOG/PRODUCT[%s]',
                'byIdsCondition' => 'SUPPLIER_PID="%s"',
                'byId' => '/BMECAT/T_NEW_CATALOG/PRODUCT[SUPPLIER_PID="%s"]',
                'features' => 'PRODUCT_FEATURES/FEATURE',
                'featureByName' => 'PRODUCT_FEATURES/FEATURE[FNAME="%s"]',
                'featureName' => 'FNAME',
                'featureValues' => 'FVALUE',
                'priceDetails' => array (
                    'byType' => 'PRODUCT_PRICE_DETAILS[PRODUCT_PRICE/@price_type="%s"]',
                    'startDate' => 'DATETIME[@type="valid_start_date"]/DATE',
                    'endDate' => 'DATETIME[@type="valid_end_date"]/DATE',
                    'priceByType' => 'PRODUCT_PRICE[@price_type="%s"]',
                ),
                'prices' => 'PRODUCT_PRICE_DETAILS/ARTICLE_PRICE',
                'pricesByType' => 'PRODUCT_PRICE_DETAILS/ARTICLE_PRICE[@price_type="%s"]',
                'priceType' => '@price_type',
                'priceAmount' => 'PRICE_AMOUNT',
                'priceCurrency' => 'PRICE_CURRENCY',
                'priceTax' => 'TAX',
                'priceFactor' => 'PRICE_FACTOR',
                'priceLowerBound' => 'LOWER_BOUND',
                'priceTerritory' => 'TERRITORY',
                'mimeBase' => 'MIME_INFO/MIME',
                'mimeType' => 'MIME_TYPE',
                'mimeSource' => 'MIME_SOURCE',
                'mimeDescr' => 'MIME_DESCR',
                'mimeAlt' => 'MIME_ALT',
                'mimePurpose' => 'MIME_PURPOSE',
                'mimeByPurpose' => 'MIME_INFO/MIME[MIME_PURPOSE="%s"]',
                'reference' => 'PRODUCT_REFERENCE',
                'referenceType' => '@type',
                'referenceByType' => 'PRODUCT_REFERENCE[@type="%s"]',
                'referenceId' => 'PROD_ID_TO',
            ),
            'globalAttributes' => array(
                'aid' => array(
                    'id' => 'SUPPLIER_PID',
                    //					'name' => 'pim.dataport.bmecat.globalattr.erpcode',
                    'name' => 'Artikelnummer',
                    'values' => array(),
                ),
                'shortDescription' => array(
                    'id' => 'PRODUCT_DETAILS/DESCRIPTION_SHORT',
                    //					'name' => 'pim.dataport.bmecat.globalattr.shortDescription',
                    'name' => 'Langbeschreibung',
                    'values' => array(),
                ),
                'longDescription' => array(
                    'id' => 'PRODUCT_DETAILS/DESCRIPTION_LONG',
                    //					'name' => 'pim.dataport.bmecat.globalattr.longDescription',
                    'name' => 'Langbeschreibung',
                    'values' => array(),
                ),
                'ean' => array(
                    'id' => 'PRODUCT_DETAILS/EAN',
                    //					'name' => 'pim.dataport.bmecat.globalattr.ean',
                    'name' => 'EAN',
                    'values' => array(),
                ),
                'mnf' => array(
                    'id' => 'PRODUCT_DETAILS/MANUFACTURER_NAME',
                    //					'name' => 'pim.dataport.bmecat.globalattr.manufacturerName',
                    'name' => 'Herstellername',
                    'values' => array(),
                ),
                'mnfAid' => array(
                    'id' => 'PRODUCT_DETAILS/MANUFACTURER_PID',
                    //					'name' => 'pim.dataport.bmecat.globalattr.manufacturerAid',
                    'name' => 'Hersteller-ID',
                    'values' => array(),
                ),
                'orderUnit' => array(
                    'id' => 'PRODUCT_ORDER_DETAILS/ORDER_UNIT',
                    //					'name' => 'pim.dataport.bmecat.globalattr.orderUnit',
                    'name' => 'Bestelleinheit',
                    'values' => array(),
                ),
                'contentUnit' => array(
                    'id' => 'PRODUCT_ORDER_DETAILS/CONTENT_UNIT',
                    //					'name' => 'pim.dataport.bmecat.globalattr.contentUnit',
                    'name' => 'Bestellmenge / Einheit',
                    'values' => array(),
                ),
            )
        )
    );
    
    private static $bmecatLegacyXpath = array(
        '2005' => array(
            'mapping' => array(
                'test' => '/BMECAT/T_NEW_CATALOG/ARTICLE_TO_CATALOGGROUP_MAP',
                'productToCategory' => '/BMECAT/T_NEW_CATALOG/ARTICLE_TO_CATALOGGROUP_MAP[CATALOG_GROUP_ID="%s"]/ART_ID',
                'productToCategories' => '/BMECAT/T_NEW_CATALOG/ARTICLE_TO_CATALOGGROUP_MAP[%s]/ART_ID',
                'categoriesToProduct' => '/BMECAT/T_NEW_CATALOG/ARTICLE_TO_CATALOGGROUP_MAP[ART_ID=%s]/CATALOG_GROUP_ID',
                'categoryId' => 'CATALOG_GROUP_ID',
            )
        ),
    );
    
    private static $bmecatXpathCache = array();
    
    public static function validateBmecat($file, $version): boolean {
        $isValid = true;
        
        $xpathExpr = self::getBmecatXpathForVersion($version);
        if (!array_key_exists('validation', $xpathExpr)) {
            return false;
        }
        
        try {
            $bmecatXpath = $xpathExpr['validation'];
            
            $dom = self::getDomDocument($file);
            if (!($dom instanceof \DOMDocument)) {
                return false;
            }
            
            $xp = new \DOMXPath($dom);
            
            foreach ($bmecatXpath as $xpathExpr) {
                $elements = $xp->query($xpathExpr);
                if (!$elements || $elements->length <= 0) {
                    $isValid = false;
                }
            }
        } catch (\Exception $e) {
            $isValid = false;
            Logger::error('Unable to check bmecat file "' . $file . '"');
        }
        
        return $isValid;
    }
    
    public static function getBmecatInfo($file, $version) {
        $xpathExpr = self::getBmecatXpathForVersion($version);
        if (!array_key_exists('validation', $xpathExpr)) {
            return null;
        }
        
        
        $dom = self::getDomDocument($file);
        if (!($dom instanceof \DOMDocument)) {
            return false;
        }
        
        $xp = new \DOMXPath($dom);
        
        
        $mimes = $xp->query($xpathExpr['info']['mime']);
        
        $assets = array(
            'image' => 0,
            'pdf' => 0,
            'other' => 0,
        );
        
        foreach ($mimes as $mime) {
            $assets[self::getAssetType($mime->nodeValue)]++;
        }
        
        $featureNameNodes = $xp->query(implode('/', array($xpathExpr['product']['all'], $xpathExpr['product']['features'], $xpathExpr['product']['featureName'])));
        
        $features = array();
        foreach ($featureNameNodes as $node) {
            if (!in_array($node->nodeValue, $features)) {
                $features[] = $node->nodeValue;
            }
        }
        
        
        $info = array(
            'categories' => $xp->query($xpathExpr['category']['all'])->length,
            'products' => $xp->query($xpathExpr['product']['all'])->length,
            'attributes' => count($features),
            'assetImage' => $assets['image'],
            'assetPdf' => $assets['pdf'],
            'assetOther' => $assets['other'],
        );
        
        return $info;
    }
    
    public static function getAssetType($filename) {
        $type = substr($filename, strrpos($filename, '.') + 1);
        
        switch (strtolower($type)) {
            case 'jpg':
            case 'jpeg':
            case 'gif':
            case 'tif':
            case 'tiff':
            case 'png':
            case 'bmp':
                return 'image';
                break;
            case 'pdf':
                return 'pdf';
                break;
            default:
                return 'other';
                break;
        }
    }
    
    /**
     * @param string $source
     * @return \DOMDocument|null
     */
    public static function getDomDocument($source) {
        if (!is_file($source) || !is_readable($source)) {
            return null;
        }
        if (array_key_exists($source, self::$fileHandles)) {
            return self::$fileHandles[$source];
        }
        
        $hash = md5_file($source);
        $tempFileName = PIMCORE_TEMPORARY_DIRECTORY . DIRECTORY_SEPARATOR . 'domdoc-' . $hash . '.xml';
        
        $dom = new \DOMDocument();
        if (is_file($tempFileName) && is_readable($tempFileName)) {
            $dom->load($tempFileName);
        } else {
            $dom->load($source);
            
            if ($dom == null) {
                Logger::error("XML file \"{$source}\" could not be read");
                return null;
            }
            
            // Remove namespace
            $root = $dom->documentElement;
            $root->removeAttributeNS($root->getAttributeNode("xmlns")->nodeValue, "");
            
            // hack hack, cough cough, hack hack
//  		$dom->loadXML($dom->saveXML($dom));
            $dom->save($tempFileName);
            $dom->load($tempFileName);
        }
        
        self::$fileHandles[$source] = $dom;
        
        return $dom;
    }
    
    /**
     * Lädt Mappingvorschriften für aus der Quelldatei für die angegebene Kategorie im Kontext des Datenports (dieser
     * definiert die BMEcat-Version, die zum Parsing der Datei herangezogen wird).
     * @param $dataportId
     * @param $sourceCategoryId
     * @param $hash
     * @param bool $includeRootNodes
     * @return array
     */
    public static function findMappings($dataportId, $sourceCategoryId, $hash, $includeRootNodes = false) {
        $dataportModel = new Dataport();
        $dataport = $dataportModel->get($dataportId);
        $db = Db::get();
        
        if (!$dataport) {
            return array();
        }
        
        $sourceConfig = unserialize($dataport['sourceconfig']);
        if (!is_array($sourceConfig)) {
            return array();
        }
        
        $categoryClassId = $sourceConfig['categoryClassId'];
        $clazz = ClassDefinition::getById($categoryClassId);
        
        if (!($clazz instanceof ClassDefinition)) {
            return array();
        }
        
        $mappingModel = new Categorymapping();
        
        $version = '1.2';
        if ($dataport && !empty($dataport['sourceconfig'])) {
            $config = unserialize($dataport['sourceconfig']);
            if (array_key_exists('version', $config)) {
                $version = $config['version'];
            }
        }
        
        $xpathExpr = self::getBmecatXpathForVersion($version);
        if (empty($xpathExpr)) {
            return array();
        }
        
        
        $mappings = array();
        
        $sourceCategories = $bmecatParents = self::findBmecatParentsCache($sourceCategoryId, $hash);
        $sourceCategories[] = $sourceCategoryId;
        
        // Root Node-IDs finden
        $rootNodeIds = array();
        
        if (!$includeRootNodes) {
            $rootNodes = $db->query('SELECT `categoryId` FROM ' . Installer::TABLE_BME_CATEGORY . ' WHERE `file` = ? AND `parentId` IS NULL', array($hash));
            
            foreach ($rootNodes as $rootNode) {
                $rootNodeIds[] = $rootNode['categoryId'];
            }
        }
        
        foreach ($sourceCategories as $source) {
            // Root-Nodes nicht beachten. Sie vererben zwar Mappingvorschriften, Artikel werden hier aber nicht
            // zugeordnet. Ausnahme: Die Quelldatei hat nur eine einzige Kategorie
            if (empty($bmecatParents) || !in_array($source, $rootNodeIds, true)) {
                $categoryMappings = $mappingModel->find(array(
                    'dataportId = ?' => $dataportId,
                    'sourceCategoryId = ?' => $source,
                ));
                
                foreach ($categoryMappings as $row) {
                    $mappings[$row->targetCategoryId] = $row['sourceCategoryId'];
                }
            }
        }
        
        $data = array();
        
        foreach ($mappings as $targetCategoryId => $source) {
            /** @var Concrete\ $category */
            $category = AbstractObject::getById($targetCategoryId);
            
            if ($category instanceof Concrete && $category->getClass()->getId() == $clazz->getId()) {
                $data[$targetCategoryId] = self::getRecursiveCategoryMapping($dataport['id'], $source, $targetCategoryId, $clazz, $sourceConfig['categoryfolder'], $hash);
            }
        }
        
        return $data;
    }
    
    /**
     * Ermittelt die Zuordnung von Quell- zu Zielkategorien unter Berücksichtigung der Mappings von Elternkategorien,
     * die an die Kinder vererbt werden. Tiefer im Baum liegende Mappingvorschriften sind spezifischer und überschreiben
     * Mappings für gleiche Felder weiter oben in der Hierarchie.
     *
     * @param $dataportId
     * @param $sourceCategoryId
     * @param $targetCategoryId
     * @param ClassDefinition $clazz
     * @param $categoryfolder
     * @param string $hash
     * @return array
     */
    private static function getRecursiveCategoryMapping($dataportId, $sourceCategoryId, $targetCategoryId, ClassDefinition $clazz, $categoryfolder, $hash) {
        $data = array();
        
        $pimParents = self::findCategoryParents($targetCategoryId, $clazz, $categoryfolder);
        $pimParents[] = $targetCategoryId;
        $bmecatParents = self::findBmecatParentsCache($sourceCategoryId, $hash);
        $bmecatParents[] = $sourceCategoryId;
        
        $mappingModel = new Categorymapping();
        
        /**
         * Ob zuerst über bmecat oder pim iteriert wird, bestimmt die Spezifität (und damit, welche Mappings
         * überschrieben werden, wenn diese rekursiv aufgelöst werden).
         */
        foreach ($bmecatParents as $bmecatParentId) {
            foreach ($pimParents as $pimParentId) {
                
                $mapping = $mappingModel->findOne(array(
                    'dataportId = ?' => $dataportId,
                    'sourceCategoryId = ?' => $bmecatParentId,
                    'targetCategoryId = ?' => $pimParentId,
                ));
                
                if (!empty($mapping)) {
                    $m = array();
                    
                    $categoryMappingFieldRepository = new CategorymappingField();
                    $result = $categoryMappingFieldRepository->find(['id = ?' => $mapping['id']]);
                    foreach ($result as $mapping) {
                        if (!array_key_exists($mapping['fieldCollection'], $m)) {
                            $m[$mapping['fieldCollection']] = array();
                        }
                        
                        
                        $key = $mapping['field'];
                        if (!empty($mapping['locale'])) {
                            $key .= '#' . $mapping['locale'];
                        }
                        
                        
                        $format = null;
                        
                        if (!empty($mapping['format'])) {
                            $format = unserialize($mapping['format']);
                        }
                        
                        $m[$mapping->fieldCollection][$key] = array(
                            'locale' => $mapping['locale'],
                            'xpath' => $mapping['xpath'],
                            'keyMapping' => $mapping['keyMapping'],
                            'format' => $format,
                            'calculation' => $mapping['calculation'],
                            'inherited' => $pimParentId != $targetCategoryId || $bmecatParentId != $sourceCategoryId,
                        );
                    }
                    
                    $data = Helper::array_merge_recursive_distinct($data, $m);
                }
                
            }
        }
        
        return $data;
    }
    
    
    /**
     * @param $targetCategoryId
     * @param ClassDefinition $clazz
     * @param $categoryfolder
     * @param $fieldCollectionProperty
     * @return array
     */
    public static function getRecursiveFieldcollectionFields($targetCategoryId, ClassDefinition $clazz, $categoryfolder, $fieldCollectionProperty) {
        $data = array();
        if (empty($fieldCollectionProperty)) {
            return array();
        }
        
        if (!($clazz->getFieldDefinition($fieldCollectionProperty) instanceof ClassDefinition\Data\Fieldcollections)) {
            return array();
        }
        
        $pimParents = self::findCategoryParents($targetCategoryId, $clazz, $categoryfolder);
        $pimParents[] = $targetCategoryId;
        
        foreach ($pimParents as $pimParentId) {
            $category = Concrete::getById($pimParentId);
            if ($category instanceof Concrete && $category) {
                try {
                    $getter = 'get' . ucfirst($fieldCollectionProperty);
                    $fieldCollections = $category->$getter();
                    
                    if ($fieldCollections instanceof Fieldcollection) {
                        /**
                         * @var $fieldCollection
                         */
                        foreach ($fieldCollections as $fieldCollection) {
                            if ($fieldCollection instanceof Fieldcollection\Data\AbstractData) {
                                $data[$fieldCollection->getType()] = array();
                                
                                $definitions = $fieldCollection->getDefinition()->getFieldDefinitions();
                                
                                if (is_array($definitions)) {
                                    foreach ($definitions as $def) {
                                        $data[$fieldCollection->getType()][] = self::getClassFieldData($def);
                                    }
                                }
                            }
                        }
                    }
                } catch (\Exception $e) {
                    // @todo log error or notify anywhere else because an error in this try-catch-block will not appear anywhere
//					echo $e;
                }
            }
        }
        
        return $data;
    }
    
    /**
     * @param Fieldcollection\Definition $fieldCollectionDefinition
     * @return array
     */
    public static function getFieldcollectionFields(Fieldcollection\Definition $fieldCollectionDefinition) {
        $data = array();
        
        if ($fieldCollectionDefinition instanceof Fieldcollection\Definition) {
            
            $definitions = $fieldCollectionDefinition->getFieldDefinitions();
            
            if (is_array($definitions)) {
                foreach ($definitions as $def) {
                    $data[] = self::getClassFieldData($def);
                }
            }
        }
        
        return $data;
    }
    
    /**
     * Lädt Elternkategorien der angegebenen Kategorie bis zur Wurzel des Baumes oder bis die Rekursion auf ein
     * Elternelement trifft, das weder Ordner noch von der konfigurierten Kategorieklasse ist.
     * @param int $categoryId ID der Startkategorie
     * @param ClassDefinition $categoryClass Klasse der Kategorieobjekte
     * @param string $root Pfad des Root-Objekts im Kategorieteilbaum
     * @return array
     */
    public static function findCategoryParents(int $categoryId, ClassDefinition $categoryClass, string $root) {
        $originalCategoryId = $categoryId;
        $parents = array();
        
        $running = true;
        $category = Concrete::getById($categoryId);
        
        do {
            if ( ($category instanceof Folder || ($category instanceof Concrete && $category->getClassId() == $categoryClass->getId()))
                && $root != $category->getFullPath()) {
                
                if ($category->getId() != $originalCategoryId) {
                    $parents[] = $category->getId();
                }
                $category = $category->getParent();
            } else {
                $running = false;
            }
        } while ($running);
        
        return array_reverse($parents);
    }
    
    /**
     * @param string $categoryId
     * @param \DOMXPath $xp
     * @param array $xpathExpr
     * @return array
     */
    public static function findBmecatParents(string $categoryId, \DOMXPath $xp, array $xpathExpr) {
        Feature::checkFeature(Feature::BMECAT);
        
        $data = array();
        
        $running = true;
        
        do {
            $xpath = implode('/', array($xpathExpr['category']['base'], sprintf($xpathExpr['category']['parentForCategory'], $categoryId)));
            
            $parentIdNode = $xp->query($xpath)->item(0);
            if ($parentIdNode instanceof \DOMElement) {
                $type = null;
                $parentId = $parentIdNode->nodeValue;
                
                $typeNode = $xp->query($xpathExpr['category']['catType'], $parentIdNode)->item(0);
                
                if ($typeNode instanceof \DOMElement) {
                    $type = $typeNode->nodeValue;
                }
                
                
                if (!$parentId || $type === 'root') {
                    $running = false;
                } else {
                    $data[] = $parentId;
                    $categoryId = $parentId;
                }
                
            } else {
                $running = false;
            }
        } while ($running);
        
        
        return array_reverse($data);
    }
    
    /**
     * @param string $categoryId
     * @param string $hash
     * @return array
     */
    public static function findBmecatParentsCache($categoryId, $hash) {
        Feature::checkFeature(Feature::BMECAT);
        
        $repository = new BmeCategory();
        $parents = array();
        
        do {
            $categoryId = $repository->findOne(['categoryId = ?' => $categoryId, 'file = ?' => $hash]);
            
            if (!empty($categoryId)) {
                $parents[] = $categoryId;
            }
        } while (!empty($categoryId));
        
        return $parents;
    }
    
    public static function getClassFields($class) {
        $fields = array();
        if (!($class instanceof ClassDefinition)) {
            return array();
        }
        
        $fieldDefinitions = $class->getFieldDefinitions();
        $languages = \Pimcore\Tool::getValidLanguages();
        
        foreach ($fieldDefinitions as $def) {
            if ($def instanceof ClassDefinition\Data\Localizedfields) {
                
                $localizedFields = $def->getFieldDefinitions();
                foreach ($localizedFields as $locField) {
                    foreach ($languages as $language) {
                        $fields[] = self::getClassFieldData($locField, $language);
                    }
                }
            } else {
                $fields[] = self::getClassFieldData($def);
            }
        }
        
        return $fields;
    }
    
    /**
     * @param ClassDefinition\Data $field Felddefinition
     * @param string $locale Sprachkürzel
     * @return array
     */
    private static function getClassFieldData($field, $locale = null) {
        $data = array(
            'name' => $field->getTitle() . ($locale != null ? ' (' . $locale . ')' : ''),
            'key' => $field->getName() . ($locale != null ? '#' . $locale : ''),
            'type' => $field->getFieldtype(),
            'tooltip' => $field->getTooltip(),
        );
        
        return $data;
    }
    
    /**
     * @param array $dataport
     * @return array
     */
    public static function createFieldMappings(array $dataport, Translator $translator) {
        $mappings = array();

        $fieldDefinition = new Input();
        $fieldDefinition->setTitle($translator->trans('pim.mapping.key', [], 'admin'));
        $fieldDefinition->setName('key');
        $mappings[] = self::createMapping($fieldDefinition, $dataport['id']);

        $fieldDefinition = new Input();
        $fieldDefinition->setTitle($translator->trans('pim.mapping.path', [], 'admin'));
        $fieldDefinition->setName('path');
        $mappings[] = self::createMapping($fieldDefinition, $dataport['id']);

        $fieldDefinition = new Input();
        $fieldDefinition->setTitle($translator->trans('pim.mapping.fullpath', [], 'admin'));
        $fieldDefinition->setName('fullpath');
        $mappings[] = self::createMapping($fieldDefinition, $dataport['id'], [], ['writeProtected' => true]);

        $fieldDefinition = new \Pimcore\Model\DataObject\ClassDefinition\Data\Checkbox();
        $fieldDefinition->setTitle($translator->trans('pim.dataport_importstatus', [], 'admin'));
        $fieldDefinition->setName('published');
        $mappings[] = self::createMapping($fieldDefinition, $dataport['id']);
        
        $fieldDefinitions = array();
        
        $targetconfig = unserialize($dataport['targetconfig']);
        if (is_array($targetconfig)) {
            $itemClassId = $targetconfig['itemClass'];
            
            $class = ClassDefinition::getById($itemClassId);
            if ($class instanceof ClassDefinition) {
                $fieldDefinitions = $class->getFieldDefinitions();
            }
        }

        /**
         * @var $fieldDefinitions \Pimcore\Model\DataObject\ClassDefinition\Data[]
         */
        foreach ($fieldDefinitions as $def) {
            $mappings = \array_merge($mappings, self::handleDefinitionElement($def, $dataport['id']));
        }
        
        return $mappings;
    }
    
    /**
     * Container mit Kindelementen rekursiv auflösen
     *
     * @param ClassDefinition\Layout|ClassDefinition\Data $element
     * @param int $dataportId
     * @param array $mappingOptions
     * @return array
     */
    private static function handleDefinitionElement($def, $dataportId, array $mappingOptions = []) {
        $mappings = [];
        if ($def instanceof ClassDefinition\Layout) {
            foreach ($def->getChildren() as $child) {
                $childMappings = self::handleDefinitionElement($child, $dataportId, $mappingOptions);
                $mappings = array_merge($mappings, $childMappings);
            }
        } elseif ($def instanceof ClassDefinition\Data\Localizedfields) {
            $languages = Tool::getValidLanguages();
            foreach ($def->getFieldDefinitions() as $child) {
                foreach ($languages as $language) {
                    $childMappings = self::handleDefinitionElement($child, $dataportId, \array_replace_recursive($mappingOptions, ['locale' => $language]));
                    $mappings = array_merge($mappings, $childMappings);
                }
            }
        } elseif($def instanceof ClassDefinition\Data\Objectbricks) {
            $allowedBricks = $def->getAllowedTypes();
            foreach($allowedBricks as $allowedBrick) {
                $brickDefinition = \Pimcore\Model\DataObject\Objectbrick\Definition::getByKey($allowedBrick);

                foreach ($brickDefinition->getFieldDefinitions() as $brickField) {
                    $childMappings = self::handleDefinitionElement($brickField, $dataportId, \array_replace_recursive($mappingOptions, ['targetBrickField' => $def->getName(), 'brickName' => $allowedBrick]));
                    $mappings = array_merge($mappings, $childMappings);
                }
            }
        } else {
            $mappings[] = self::createMapping($def, $dataportId, $mappingOptions);
        }
        
        return $mappings;
    }

    /**
     * @param ClassDefinition\Data $def Fielddefinition
     * @param int                  $dataportId
     * @param array                $mappingOptions
     * @param array                $formatOptions
     *
     * @return array
     */
    private static function createMapping(
        ClassDefinition\Data $def,
        $dataportId,
        array $mappingOptions = [],
        array $formatOptions = []
    ) {
        $fieldName = $def->getName();
        
        $key = $fieldName;
        if(!empty($mappingOptions['locale'])) {
            $key .= '#' . $mappingOptions['locale'];
        }

        $name = $def->getTitle();
        if(!empty($mappingOptions['locale'])) {
            $name .= '#' . $mappingOptions['locale'];
        }
        if(!empty($mappingOptions['brickName'])) {
            $name = $mappingOptions['brickName'].'/'.$name;
        }

        $mapping = array(
            'attribute' => array(
                'name' => $name,
                'key' => $key,
            ),
            'attributeKey' => $key,
            'attributeName' => $name,
            'type' => $def->getFieldtype(),
            
            // Mapping
            'field' => null,
            'settings' => array()
        );

        $mapping = array_replace_recursive($mapping, $mappingOptions);
        
        $conditions = array(
            'dataportId = ?' => $dataportId,
            'fieldName = ?' => $fieldName,
        );

        if(!empty($mappingOptions['locale'])) {
            $conditions['locale = ?'] = $mappingOptions['locale'];
        }
        
        $mappingTable = new Fieldmapping();
        $row = $mappingTable->findOne($conditions);
        
        if ($row) {
            $format = '';
            
            if (!empty($row['format'])) {
                $format = unserialize($row['format']);
            }
            $format = array_replace_recursive($format, $formatOptions);
            
            $mapping['field'] = $row['fieldNo'];
            $mapping['settings'] = self::createSettings($def->getFieldtype(),$row['keyMapping'] == 1, $format, $row['calculation']);
        } else {
            $mapping['settings'] = self::createSettings($def->getFieldtype(), false, $formatOptions);
        }
        
        return $mapping;
    }
    
    /**
     * @param $type
     * @param bool $keyMapping
     * @param array $format
     * @param string $calculcation
     * @return array
     */
    private static function createSettings($type, $keyMapping = false, $format = array(), $calculcation = '') {
        if (!is_array($format)) {
            $format = array();
        }
        
        if (!is_bool($keyMapping)) {
            $keyMapping = false;
        }
        
        $currentFormat = $format;
        $defaults = self::getMappingDefaults();
        $format = [];
        if (array_key_exists($type, $defaults)) {
            $format = $defaults[$type];
            foreach ($currentFormat as $attr => $value) {
                $format[$attr] = $value;
            }
        }
        $format['writeProtected'] = $currentFormat['writeProtected'];
        
        return array(
            'keyMapping' => $keyMapping,
            'format' => $format,
            'calculation' => $calculcation,
        );
    }
    
    /**
     * array_merge_recursive does indeed merge arrays, but it converts values with duplicate
     * keys to arrays rather than overwriting the value in the first array with the duplicate
     * value in the second array, as array_merge does. I.e., with array_merge_recursive,
     * this happens (documented behavior):
     *
     * array_merge_recursive(array('key' => 'org value'), array('key' => 'new value'));
     *     => array('key' => array('org value', 'new value'));
     *
     * array_merge_recursive_distinct does not change the datatypes of the values in the arrays.
     * Matching keys' values in the second array overwrite those in the first array, as is the
     * case with array_merge, i.e.:
     *
     * array_merge_recursive_distinct(array('key' => 'org value'), array('key' => 'new value'));
     *     => array('key' => 'new value');
     *
     * Parameters are passed by reference, though only for performance reasons. They're not
     * altered by this function.
     *
     * @param array $array1
     * @param mixed $array2
     * @author daniel@danielsmedegaardbuus.dk
     * @return array
     * @see http://danielsmedegaardbuus.dk/2009-03-19/phps-array_merge_recursive-as-it-should-be/
     */
    public static function &array_merge_recursive_distinct(array &$array1, &$array2 = null) {
        $merged = $array1;
        
        if (is_array($array2)) {
            foreach ($array2 as $key => $val) {
                if (is_array($array2[$key])) {
                    $merged[$key] = is_array($merged[$key]) ? self::array_merge_recursive_distinct($merged[$key], $array2[$key]) : $array2[$key];
                } else {
                    $merged[$key] = $val;
                }
            }
        }
        
        return $merged;
    }
    
    /**
     * @return array
     */
    public static function getMappingDefaults() {
        return self::$mappingDefaults;
    }
    
    public static function getBmecatXpath($version, $file) {
        Feature::checkFeature(Feature::BMECAT);
        
        if (!array_key_exists($version, self::$bmecatXpath)) {
            Logger::warn('Unsupported BMEcat version "' . $version . '"');
            return array();
        }
        
        $hash = md5_file($file);
        if (array_key_exists($hash, self::$bmecatXpathCache)) {
            return self::$bmecatXpathCache[$hash];
        }
        
        $xp = self::$bmecatXpath[$version];
        
        $filename = Installer::getConfigPath() . DIRECTORY_SEPARATOR . Installer::BMECAT_CONFIG_FILE;
        $config = simplexml_load_file($filename);
        $attrs = array();
        
        if ($version == '1.2' && isset($config->bmecat12) ) {
            $attrs = $config->bmecat12->toArray();
        } else if ($version == '2005' && isset($config->bmecat2005)) {
            $attrs = $config->bmecat2005->toArray();
        }
        
        if (array_key_exists($version, self::$bmecatLegacyXpath)) {
            // Test which type of mapping tags are used in the file
            $dom = self::getDomDocument($file);
            if ($dom instanceof \DOMDocument) {
                $xpath = new \DOMXPath($dom);
                
                $legacyXp = self::$bmecatLegacyXpath[$version];
                foreach ($legacyXp as $key => $legacy) {
                    if (array_key_exists('test', $legacy)) {
                        $result = $xpath->query($legacy['test']);
                        if ($result instanceof \DOMNodeList && $result->length > 0) {
                            $xp[$key] = $legacy;
                        }
                    }
                }
            }
        }
        
        if (is_array($attrs)) {
            foreach ($attrs as $attr) {
                $xp['globalAttributes'][] = array(
                    'id' => $attr['xpath'],
                    'name' => $attr['name'],
                    'values' => array(),
                );
            }
        }
        
        self::$bmecatXpathCache[$hash] = $xp;
        
        return $xp;
    }
    
    public static function getBmecatXpathForVersion($version) {
        if (array_key_exists($version, self::$bmecatXpath)) {
            return self::$bmecatXpath[$version];
        }
        
        return null;
    }
    
    public static function getFileForDataport($dataportId) {
        $dataportModel = new Dataport();
        $dataport = $dataportModel->get($dataportId);
        
        if (!$dataport) {
            return null;
        }
        
        $sourceConfig = unserialize($dataport['sourceconfig']);
        if (!is_array($sourceConfig) || !array_key_exists('file', $sourceConfig)) {
            return null;
        }
        
        return $sourceConfig['file'];
    }
    
    public static function extractCategories($hash) {
        $data = array();
        
        $repository = new BmeCategory();
        $allCategories = $repository->find(['file = ?' => $hash]);
        foreach ($allCategories as $category) {
            $data[] = array(
                'id' => $category['categoryId'],
                'hash' => $category['hash'],
            );
        }
        
        return $data;
    }
    
    public static function extractProducts($hash, $version) {
        $products = array();
        $features = array();
        
        $xpathExpr = self::getBmecatXpathForVersion($version);
        if (empty($xpathExpr)) {
            return array();
        }
        
        $repository = new BmeProduct();
        $allProducts = $repository->find(['file = ?' => $hash]);
        
        foreach ($allProducts as $product) {
            $productData = array(
                'id' => $product['productId'],
            );
            
            if (empty($productData['id'])) {
                continue;
            }
            
            try {
                $xml = @new \SimpleXMLElement($product['xml']);
                
                $featureNodes = $xml->xpath($xpathExpr['product']['features']);
                
                foreach ($featureNodes as $featureNode) {
                    $featureName = self::getSimpleXMLValue($featureNode, $xpathExpr['product']['featureName']);
                    if (!in_array($featureName, $features)) {
                        $features[] = $featureName;
                    }
                }
                
                $productData['hash'] = $product['hash'];
                $products[] = $productData;
            } catch (\Exception $e) {
                Logger::error('Unable to parse product xml for product "' . $product['productId'] . '" in "' . $hash . '"');
            }
        }
        
        return array(
            'features' => $features,
            'products' => $products,
        );
    }
    
    /**
     * Vergleicht die angegebene Datei mit dem Import mit der ID $importKey und gibt das Ergebnis für Kategorien,
     * Artikel und Attribute zurück.
     * @param $file
     * @param $version
     * @param $importKey
     * @return array
     */
    public static function compareToImport($file, $version, $importKey) {
        Feature::checkFeature(Feature::BMECAT);
        
        $result = array(
            'categories' => array(
                'new' => array(),
                'missing' => array(),
                'changed' => array(),
            ),
            'products' => array(
                'new' => array(),
                'missing' => array(),
                'changed' => array(),
            ),
            'features' => array(
                'new' => array(),
                'missing' => array(),
            ),
        );
        
        if (empty($importKey)) {
            return $result;
        }
        
        $categoryLogModel = new CategoryLog();
        $productLogModel = new ProductLog();
        $featureLogModel = new FeatureLog();
        
        // Sicherstellen, dass das XML bereits geparsed wurde
        self::parseBmecat($file, $version);
        $fileHash = md5_file($file);
        
        // Kategorien
        $oldCategories = array();
        
        $categoryRows = $categoryLogModel->find(array(
            'importKey = ?' => $importKey,
        ));
        
        foreach ($categoryRows as $row) {
            $oldCategories[$row['categoryId']] = $row['hash'];
        }
        
        $newCategories = array();
        $newCategoryData = self::extractCategories($fileHash);
        $changedCategories = array();
        
        foreach ($newCategoryData as $cat) {
            $newCategories[$cat['id']] = $cat['hash'];
            if (array_key_exists($cat['id'], $oldCategories) && $cat['hash'] != $oldCategories[$cat['id']]) {
                $changedCategories[] = $cat['id'];
            }
        }
        
        $result['categories']['new'] = array_keys(array_diff_key($newCategories, $oldCategories));
        $result['categories']['missing'] = array_keys(array_diff_key($oldCategories, $newCategories));
        $result['categories']['changed'] = $changedCategories;
        
        
        // Produkte
        $productData = self::extractProducts($fileHash, $version);
        $oldProducts = array();
        $newProducts = array();
        $changedProducts = array();
        
        $productRows = $productLogModel->find(array(
            'importKey = ?' => $importKey,
        ));
        
        foreach ($productRows as $row) {
            $oldProducts[$row['productId']] = $row['hash'];
        }
        
        foreach ($productData['products'] as $data) {
            $newProducts[$data['id']] = $data['hash'];
            if (array_key_exists($data['id'], $oldProducts) && $data['hash'] != $oldProducts[$data['id']]) {
                $changedProducts[] = $data['id'];
            }
        }
        
        $result['products']['new'] = array_keys(array_diff_key($newProducts, $oldProducts));
        $result['products']['missing'] = array_keys(array_diff_key($oldProducts, $newProducts));
        $result['products']['changed'] = $changedProducts;
        
        
        // Features
        $oldFeatures = array();
        
        $featureRows = $featureLogModel->find(array(
            'importKey = ?' => $importKey,
        ));
        
        foreach ($featureRows as $row) {
            $oldFeatures[] = $row['name'];
        }
        
        $newFeatureNames = array_diff($productData['features'], $oldFeatures);
        sort($newFeatureNames); // Entfernt alte Indexzuordnung
        $missingFeatureNames = array_diff($oldFeatures, $productData['features']);
        sort($missingFeatureNames); // Entfernt alte Indexzuordnung
        
        $result['features']['new'] = $newFeatureNames;
        $result['features']['missing'] = $missingFeatureNames;
        
        $valuesToString = function($el) {
            if (is_array($el)) {
                foreach ($el as $key => $value) {
                    $el[$key] = (string)$value;
                }
            }
            
            return $el;
        };
        
        $result['categories'] = array_map($valuesToString, $result['categories']);
        $result['products'] = array_map($valuesToString, $result['products']);
        $result['features'] = array_map($valuesToString, $result['features']);
        
        return $result;
    }
    
    public static function getNodeValue($node) {
        if ($node instanceof \DOMNodeList) {
            $node = $node->item(0);
        }
        
        if ($node instanceof \DOMNode) {
            return $node->nodeValue;
        }
        
        return null;
    }
    
    public static function getAllNodeValues($node) {
        $values = null;
        
        if ($node instanceof \DOMNodeList) {
            $values = array();
            
            foreach ($node as $n) {
                if ($n instanceof \DOMNode) {
                    $values[] = $n->nodeValue;
                }
            }
        } else if ($node instanceof \DOMNode) {
            $values = array($node->nodeValue);
        }
        
        return $values;
    }
    
    public static function parseBmecat($file, $version) {
        Feature::checkFeature(Feature::BMECAT);
        
        $dom = self::getDomDocument($file);
        if (!($dom instanceof \DOMDocument)) {
            return false;
        }
        
        $xpathExpr = self::getBmecatXpathForVersion($version);
        if (empty($xpathExpr)) {
            return false;
        }
        
        $bmeFileRepository = new BmeFile();
        $db = Db::get();
        $xp = new \DOMXPath($dom);
        $hash = md5_file($file);
        
        
        // DEBUG
//		$db->query('DELETE FROM ' . Pim_Plugin::TABLE_BME_FILE . ' WHERE `hash` = ?', array($hash));
        
        $existing = $bmeFileRepository->countRows(['hash = ?' => $hash]);
        if ($existing > 0) {
            return true;
        }
        
        $bmeFileRepository->create(['hash' => $hash, 'created' => new \DateTime()]);
        
        // Categories
        $allCategories = $xp->query($xpathExpr['category']['all']);
        if ($allCategories instanceof \DOMNodeList) {
            $bmeCategoryRepository = new BmeCategory();
            foreach ($allCategories as $category) {
                $categoryId = self::getNodeValue($xp->query($xpathExpr['category']['catId'], $category));
                $name = self::getNodeValue($xp->query($xpathExpr['category']['catName'], $category));
                $parent = self::getNodeValue($xp->query($xpathExpr['category']['parentId'], $category));
                $type = self::getNodeValue($xp->query($xpathExpr['category']['catType'], $category));
                
                if (empty($categoryId)) {
                    Logger::warn('Skipping category without id');
                    continue;
                }
                
                try {
                    $hashData = array(
                        'name' => self::getNodeValue($xp->query($xpathExpr['category']['catName'], $category)),
                        'type' => self::getNodeValue($xp->query($xpathExpr['category']['catType'], $category)),
                        'parent' => self::getNodeValue($xp->query($xpathExpr['category']['parentId'], $category)),
                    );
                    
                    sort($hashData);
                    $categoryHash = md5(serialize($hashData));
                    
                    $bmeCategoryRepository->create([
                        'file' => $hash,
                        'category_id' => $categoryId,
                        'name' => $name,
                        'parentId' => empty($parent) || $type === 'root' ? null : $parent,
                        'hash' => $categoryHash]);
                } catch (\Exception $e) {
                    Logger::error('Unable to parse category from BMEcat');
                }
            }
        }
        
        
        // Products
        $allProducts = $xp->query($xpathExpr['product']['all']);
        if ($allProducts instanceof \DOMNodeList) {
            $bmeProductRepository = new BmeProduct();
            /**
             * @var $product \DOMElement
             */
            foreach ($allProducts as $product) {
                $productId = self::getNodeValue($xp->query($xpathExpr['product']['id'], $product)->item(0));
                
                if (empty($productId)) {
                    Logger::warn('Skipping product without id');
                    continue;
                }
                
                $productHashData = array(
                    'features' => array()
                );
                
                foreach ($xpathExpr['globalAttributes'] as $key => $def) {
                    $productHashData[$key] = self::getNodeValue($xp->query($def['id'], $product)->item(0));
                }
                
                $featureNodes = $xp->query($xpathExpr['product']['features'], $product);
                
                foreach ($featureNodes as $featureNode) {
                    $featureName = self::getNodeValue($xp->query($xpathExpr['product']['featureName'], $featureNode));
                    if (empty($featureName)) {
                        $featureName = 'unknown';
                    }
                    
                    $productHashData['features'][$featureName] = self::getAllNodeValues($xp->query($xpathExpr['product']['featureValues'], $featureNode));
                    sort($productHashData['features'][$featureName]);
                }
                
                
                ksort($productHashData);
                ksort($productHashData['features']);
                
                $productHash = md5(serialize($productHashData));
                
                
                $xml = $product->ownerDocument->saveXML($product);
                
                try {
                    $bmeProductRepository->create([
                        'file' => $hash,
                        'productId' => $productId,
                        'xml' => $xml,
                        'hash' => $productHash
                    ]);
                } catch (\Exception $e) {
                    Logger::error('Unable to parse product from BMEcat');
                }
            }
        }
        
        
        // Products to categories
        $allRelations = $xp->query($xpathExpr['mapping']['allProductToCategory']);
        if ($allRelations instanceof \DOMNodeList) {
            $bmeProductCategoryRepository = new BmeProductCategory();
            /**
             * @var $product \DOMElement
             */
            foreach ($allRelations as $relation) {
                $productId = self::getNodeValue($xp->query($xpathExpr['mapping']['productId'], $relation)->item(0));
                $categoryId = self::getNodeValue($xp->query($xpathExpr['mapping']['categoryId'], $relation)->item(0));
                
                if (empty($productId) || empty($categoryId)) {
                    Logger::warn('Skipping invalid category mapping');
                    continue;
                }
                
                try {
                    $bmeProductCategoryRepository->create([
                        'file' => $hash,
                        'productId' => $productId,
                        'categoryId' => $categoryId
                    ]);
                } catch (\Exception $e) {
                    Logger::error('Unable to parse product to category mapping from BMEcat');
                }
            }
        }
        
        return true;
    }
    
    public static function getSimpleXMLValue($el, $xpath) {
        if ($el instanceof \SimpleXMLElement) {
            $result = $el->xpath($xpath);
            if (is_array($result) && !empty($result)) {
                return (string)$result[0];
            }
        }
        
        return null;
    }
    
    public static function getAllSimpleXMLValues($el, $xpath) {
        $values = null;
        
        if ($el instanceof \SimpleXMLElement) {
            $values = array();
            
            $result = $el->xpath($xpath);
            if (is_array($result)) {
                foreach ($result as $r) {
                    $values[] = (string)$r;
                }
            }
        }
        
        return $values;
    }
    
    /**
     * @param \DOMElement $root
     * @return array|string
     */
    
    public static function  nodeToArray(\DOMElement $root) {
        $result = array();
        
        if ($root->hasAttributes()) {
            $attrs = $root->attributes;
            foreach ($attrs as $attr) {
                $result['@attributes'][$attr->name] = $attr->value;
            }
        }
        
        if ($root->hasChildNodes()) {
            $children = $root->childNodes;
            if ($children->length == 1) {
                $child = $children->item(0);
                if ($child->nodeType == XML_TEXT_NODE) {
                    $result['_value'] = $child->nodeValue;
                    return count($result) == 1
                        ? $result['_value']
                        : $result;
                }
            }
            $groups = array();
            foreach ($children as $child) {
                if ($child->nodeName == "#text") {
                    continue;
                }
                if (!isset($result[$child->nodeName])) {
                    $result[$child->nodeName] = self::nodeToArray($child);
                } else {
                    if (!isset($groups[$child->nodeName])) {
                        $result[$child->nodeName] = array($result[$child->nodeName]);
                        $groups[$child->nodeName] = 1;
                    }
                    $result[$child->nodeName][] = self::nodeToArray($child);
                }
            }
        }
        
        return $result;
    }
    
}