<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Blackbit\PimBundle\lib\Pim\Variant\Generator;

use Blackbit\PimBundle\lib\Pim\Variant\VariantHelper;
use Blackbit\PimBundle\lib\Pim\Variant\VariantMapper;
use Pimcore\Model\DataObject;
use Pimcore\Model\DataObject\AbstractObject;

/**
 * Class TemplateGenerator
 *
 * @package blackbit\PimBundle\lib\Pim\Variant\Generator
 */
class TemplateGenerator implements GeneratorInterface
{
    /**
     * @var VariantHelper
     */
    private $variantHelper;
    /**
     * @var VariantMapper
     */
    private $variantMapper;

    /**
     * TemplateGenerator constructor.
     *
     * @param VariantHelper   $variantHelper
     * @param VariantMapper   $variantMapper
     */
    public function __construct(
        VariantHelper $variantHelper,
        VariantMapper $variantMapper
    ) {
        $this->variantHelper = $variantHelper;
        $this->variantMapper = $variantMapper;
    }

    /**
     * @param int   $variantId
     * @param array $variantConfig
     *
     * @throws \Exception
     */
    public function generate(int $variantId, array $variantConfig)
    {
        $sourceConfig = unserialize($variantConfig['sourceconfig']);
        $objectData = $sourceConfig['objectData'];
        $sourceObjects = [];
        foreach ($objectData as $object) {
            $sourceObjects[] = DataObject::getById($object->id);
        }
        $sourceObject = $sourceObjects[0];

        $mapperConfig = $this->variantHelper->getMapperConfig($variantConfig['sourcetype'], 'Products');

        $targetConfig = unserialize($mapperConfig['targetconfig']);
        $targetItemList = $this->variantHelper->getRawitemfieldConfig($variantId);

        $targetFolder = $targetConfig['itemFolder'] . '/' . $variantConfig['name'];
        $folder = DataObject\Folder::getByPath($targetFolder);
        if (null === $folder) {
            // generate Folder
            $folder = new DataObject\Folder();
            $folder->setKey($variantConfig['name']);
            $folder->setPath($targetConfig['itemFolder']);
            $folder->setParent(DataObject\Folder::getByPath($targetConfig['itemFolder']));
            $folder->save();
        }

        foreach ($targetItemList as $targetItem) {
            $targetName = \Pimcore\File::getValidFilename($variantConfig['name'] . '-' . $targetItem['name']);
            $copyMainContent = true;
            $existingObject = DataObject::getByPath($targetFolder . '/' . $targetName);
            $targetObject = $existingObject;
            $objectService = new DataObject\Service($sourceObject);
            if (null === $existingObject) {
                $copyMainContent = false;
                $targetObject = $objectService->copyAsChild($folder, $sourceObject);
            }

            if (true === $copyMainContent) {
                $targetObject = $objectService->copyContents($existingObject, $sourceObject);
            }

            $targetObject->setKey($targetName);
            $this->variantMapper->applyMapping(
                $this->variantHelper->getMapperId($variantConfig['sourcetype'], 'Products'),
                $sourceConfig,
                $targetConfig,
                $targetItem,
                $targetObject,
                $variantConfig['name']
            );

            $targetObject->save();

            foreach ($sourceObject->getChildren([AbstractObject::OBJECT_TYPE_VARIANT], true) as $sourceVariant) {
                $copyContent = true;
                $existingVariant = DataObject::getByPath($targetFolder . '/' . $targetName . '/' . $sourceVariant->getKey());
                $variantObject = $existingVariant;
                $objectService = new DataObject\Service($sourceVariant);
                if (null === $existingVariant) {
                    $copyContent = false;
                    $variantObject = $objectService->copyAsChild($targetObject, $sourceVariant);
                }

                if (true === $copyContent) {
                    $variantObject = $objectService->copyContents($existingVariant, $sourceVariant);
                }

                // run import mapper
                $this->variantMapper->applyMapping(
                    $this->variantHelper->getMapperId($variantConfig['sourcetype'], 'Variants'),
                    $sourceConfig,
                    $targetConfig,
                    $targetItem,
                    $variantObject,
                    $variantConfig['name']
                );

                $variantObject->save();
            }
        }
    }

    /**
     * @param int   $variantId
     * @param array $variantConfig
     *
     * @throws \Exception
     */
    public function delete(int $variantId, array $variantConfig)
    {
        $mapperConfig = $this->variantHelper->getMapperConfig($variantConfig['sourcetype'], 'Products');
        // removing the folder should remove and unpublish all containing items
        $targetConfig = unserialize($mapperConfig['targetconfig']);
        $targetFolder = $targetConfig['itemFolder'] . '/' . $variantConfig['name'];
        $folder = DataObject\Folder::getByPath($targetFolder);
        if (null !== $folder) {
            $folder->delete();
        }
    }

    /**
     * @param int   $variantId
     * @param array $variantConfig
     * @param array $fieldConfig
     *
     * @throws \Exception
     */
    public function deleteOne(int $variantId, array $variantConfig, array $fieldConfig)
    {
        $mapperConfig = $this->variantHelper->getMapperConfig($variantConfig['sourcetype'], 'Products');
        $targetConfig = unserialize($mapperConfig['targetconfig']);
        $targetFolder = $targetConfig['itemFolder'] . '/' . $variantConfig['name'];
        $targetName = \Pimcore\File::getValidFilename($variantConfig['name'] . '-' . $fieldConfig['name']);
        $existingObject = DataObject::getByPath($targetFolder . '/' . $targetName);
        if (null !== $existingObject) {
            $existingObject->delete();
        }
    }
}
