<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Blackbit\PimBundle\lib\Pim\Variant;

use Blackbit\PimBundle\lib\Pim\Variant\Generator\GeneratorFactory;
use Blackbit\PimBundle\model\ResourceFactory;

/**
 * Class VariantGenerator
 *
 * @package Blackbit\PimBundle\lib\Pim\Variant
 */
class VariantGenerator
{
    /**
     * @var GeneratorFactory
     */
    private $generatorFactory;
    /**
     * @var ResourceFactory
     */
    private $resourceFactory;

    /**
     * VariantGenerator constructor.
     *
     * @param GeneratorFactory $generatorFactory
     * @param ResourceFactory  $resourceFactory
     */
    public function __construct(
        GeneratorFactory $generatorFactory,
        ResourceFactory $resourceFactory
    ) {
        $this->generatorFactory = $generatorFactory;
        $this->resourceFactory = $resourceFactory;
    }

    /**
     * Generate all items from plugin_pim_variant_generator
     */
    public function generateAll()
    {
        // get all from plugin_pim_variant_generator
        foreach ($this->resourceFactory->variantGenerator()->find([]) as $variant) {
            try {
                // generate
                $this->generate($variant['variantId']);

                // remove from list
                $this->resourceFactory->variantGenerator()->deleteWhere(['variantId' => $variant['variantId']]);
            } catch (\Exception|\Error $exception) {
                // mark as failed
                $this->resourceFactory->variantGenerator()->update(
                    [
                        'state' => \Blackbit\PimBundle\model\VariantGenerator::STATE_FAILED,
                    ],
                    ['variantId' => $variant['variantId']]
                );
            }
        }
    }

    /**
     * @param int $variantId
     *
     * @throws \Exception
     */
    public function generate(int $variantId)
    {
        $variantConfig = $this->resourceFactory->variant()->get($variantId);
        // TODO: make generator type configurable
        $this->generatorFactory->getGeneratorByType('template')->generate($variantId, $variantConfig);
    }

    /**
     * @param int $variantId
     *
     * @throws \Exception
     */
    public function delete(int $variantId)
    {
        $variantConfig = $this->resourceFactory->variant()->get($variantId);

        if (isset($variantConfig['sourcetype']) && !in_array($variantConfig['sourcetype'], ['none', 'folder'])) {
            // TODO: make generator type configurable
            $this->generatorFactory->getGeneratorByType('template')->delete($variantId, $variantConfig);
        }
    }

    /**
     * @param int   $variantId
     * @param array $variantConfig
     * @param array $fieldConfig
     *
     * @throws \Exception
     */
    public function deleteOne(int $variantId, array $variantConfig, array $fieldConfig)
    {
        if (isset($variantConfig['sourcetype']) && !in_array($variantConfig['sourcetype'], ['none', 'folder'])) {
            // TODO: make generator type configurable
            $this->generatorFactory->getGeneratorByType('template')->deleteOne($variantId, $variantConfig, $fieldConfig);
        }
    }
}
