<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Pimcore\AssignProductImagesBundle\EventListener;

use Lifestyle\Pimcore\AssignProductImagesBundle\Configuration\ActiveConfig;
use Lifestyle\Pimcore\AssignProductImagesBundle\Configuration\PictureTypesList;
use Lifestyle\Pimcore\AssignProductImagesBundle\Events\AssignProductImagesEvents;
use Lifestyle\Pimcore\AssignProductImagesBundle\Model\AssetInfo;
use Lifestyle\Pimcore\AssignProductImagesBundle\Services\AssetTypeHandler\AssetTypeHandlerFactory;
use Pimcore\Event\Model\AssetEvent;
use Pimcore\Model\Asset;
use Pimcore\Model\Asset\Image;
use Pimcore\Model\Asset\Unknown;
use Pimcore\Model\Factory as ModelFactory;
use Pimcore\Tests\Helper\Model;
use Symfony\Component\EventDispatcher\EventDispatcher;

/**
 * Class AddAssetListener
 *
 * @package Lifestyle\Pimcore\AssignProductImagesBundle\EventListener
 */
class AddAssetListener
{
    /**
     * @var AssetTypeHandlerFactory
     */
    protected $assetTypeHandlerFactory;
    /**
     * @var EventDispatcher
     */
    protected $eventDispatcher;
    /**
     * @var PictureTypesList
     */
    protected $pictureTypesList;
    /**
     * @var ModelFactory;
     */
    protected $modelFactory;
    /**
     * @var bool
     */
    protected $active = false;

    /**
     * AddAssetListener constructor.
     *
     * @param AssetTypeHandlerFactory $assetTypeHandlerFactory
     * @param EventDispatcher         $eventDispatcher
     * @param PictureTypesList        $pictureTypesList
     * @param ModelFactory            $modelFactory
     * @param ActiveConfig            $activeConfig
     */
    public function __construct(
        AssetTypeHandlerFactory $assetTypeHandlerFactory,
        EventDispatcher $eventDispatcher,
        PictureTypesList $pictureTypesList,
        ModelFactory $modelFactory,
        ActiveConfig $activeConfig
    ) {
        $this->assetTypeHandlerFactory = $assetTypeHandlerFactory;
        $this->eventDispatcher = $eventDispatcher;
        $this->pictureTypesList = $pictureTypesList;
        $this->modelFactory = $modelFactory;
        $this->active = $activeConfig->isActive();
    }

    /**
     * @param AssetEvent $assetEvent
     *
     * @throws \Exception
     */
    public function onAssetAdded(AssetEvent $assetEvent)
    {
        if (false === $this->active) {
            return;
        }

        $assetInfo = new AssetInfo($assetEvent->getAsset());

        if ($this->pictureTypesList->hasPictureType($assetInfo->getPictureType())
            && ($assetEvent->getAsset() instanceof Image || $assetEvent->getAsset() instanceof Unknown)
        ) {
            $pictureTypeConfig = $this->pictureTypesList->getPictureTypeConfig($assetInfo->getPictureType());
            if (false !== stripos($assetEvent->getAsset()->getFullPath(), '/' . $pictureTypeConfig->getProcessingPath())) {
                $assetId = $assetEvent->getAsset()->getId();
                $oldPath = $assetEvent->getAsset()->getFullPath();
                // run update after create...
                $assetEvent->getAsset()->save();

                // remove processing image if still in processing folder (update case) | do not delete for unprocessed case
                $processingAsset = Asset::getById($assetId, true);
                if (null !== $processingAsset
                    && $processingAsset->getFullPath() === $oldPath
                ) {
                    $assetCount = $this->modelFactory
                        ->build('Pimcore\\Model\\Asset\\Listing')
                        ->setCondition('filename = ?', [trim($assetInfo->getOriginalName())])
                        ->count();

                    if ($assetCount > 1) {
                        $processingAsset->delete();
                    }
                }
            }
        }
    }

    /**
     * @param AssetEvent $assetEvent
     */
    public function onAssetUpdated(AssetEvent $assetEvent)
    {
        if (false === $this->active) {
            return;
        }

        $assetInfo = new AssetInfo($assetEvent->getAsset());

        if ($this->pictureTypesList->hasPictureType($assetInfo->getPictureType())
            && ($assetEvent->getAsset() instanceof Image || $assetEvent->getAsset() instanceof Unknown)
        ) {
            $pictureTypeConfig = $this->pictureTypesList->getPictureTypeConfig($assetInfo->getPictureType());

            if (false === stripos($assetEvent->getAsset()->getFullPath(), '/' . $pictureTypeConfig->getProcessingPath())) {
                $success = $this->assetTypeHandlerFactory->getByType($pictureTypeConfig->getType())->process(
                    $assetEvent,
                    $pictureTypeConfig
                );

                if (true === $success) {
                    // dispatch post assign event
                    $this->eventDispatcher->dispatch(AssignProductImagesEvents::POST_ASSIGN, $assetEvent);
                }
            }
        }
    }

    /**
     * @param AssetEvent $assetEvent
     */
    public function onBeforeAssetAdded(AssetEvent $assetEvent)
    {
        if (false === $this->active) {
            return;
        }

        $assetInfo = new AssetInfo($assetEvent->getAsset());

        if ($this->pictureTypesList->hasPictureType($assetInfo->getPictureType())
            && ($assetEvent->getAsset() instanceof Image || $assetEvent->getAsset() instanceof Unknown)
        ) {
            // add some metadata
            $assetEvent->getAsset()->addMetadata('title', 'Input', $assetEvent->getAsset()->getFilename());
            $assetEvent->getAsset()->addMetadata('position', 'Input', $assetInfo->getPicturePosition());
            foreach ($this->pictureTypesList->getPictureTypeConfig($assetInfo->getPictureType())->getMetaData() as $metaDataName => $metaDataValue) {
                $assetEvent->getAsset()->addMetadata($metaDataName, 'Input', $metaDataValue);
            }
        }
    }

    /**
     * @param AssetEvent $assetEvent
     */
    public function onBeforeAssetUpdated(AssetEvent $assetEvent)
    {
        if (false === $this->active) {
            return;
        }

        $assetInfo = new AssetInfo($assetEvent->getAsset());

        if ($this->pictureTypesList->hasPictureType($assetInfo->getPictureType())
            && ($assetEvent->getAsset() instanceof Image || $assetEvent->getAsset() instanceof Unknown)
        ) {
            $pictureTypeConfig = $this->pictureTypesList->getPictureTypeConfig($assetInfo->getPictureType());
            $modelObject = $this->modelFactory
                ->build('Pimcore\\Model\\DataObject\\' . $pictureTypeConfig->getObjectName() . '\\Listing')
                ->setCondition(
                    'o_key IN(?, ?, ?)',
                    [
                        trim($assetInfo->getObjectKey()),
                        strtoupper(trim($assetInfo->getObjectKey())),
                        strtolower(trim($assetInfo->getObjectKey())),
                    ]
                )
                ->current();

            if (false !== $modelObject) {
                if (null === $assetEvent->getAsset()->getParent()
                    || false !== stripos($assetEvent->getAsset()->getFullPath(), '/' . $pictureTypeConfig->getProcessingPath())
                ) {
                    // build target path
                    $targetPath = '/';
                    $pathArray = [
                        $pictureTypeConfig->getTargetPath(),
                    ];

                    foreach ($pictureTypeConfig->getSubfolderConfig() as $subfolderConfig) {
                        $pathArray[] = $modelObject->{'get' . ucfirst($subfolderConfig)}();
                    }

                    foreach ($pathArray as $pathValue) {
                        $parentPath = $targetPath;
                        $targetPath .= $pathValue . '/';
                        $folder = $this->getFolderCreateIfNotExists($parentPath, $pathValue);
                    }

                    $existingAsset = Asset::getByPath($targetPath . $assetEvent->getAsset()->getFilename());
                    if (null !== $existingAsset && $existingAsset instanceof Image) {
                        if ($assetEvent->getAsset()->getId() !== $existingAsset->getId()) {
                            // change content
                            $existingAsset->setData($assetEvent->getAsset()->getData());
                            $existingAsset->setDataChanged(true);
                            $existingAsset->save();

                            // pre return as we do not want to change the source asset in update case (duplicate path)
                            return;
                        }
                    }

                    $assetEvent->getAsset()->setParent($folder);
                    $assetEvent->getAsset()->setPath($targetPath);
                }
            }

            // add some metadata
            $assetEvent->getAsset()->addMetadata('title', 'Input', $assetEvent->getAsset()->getFilename());
            $assetEvent->getAsset()->addMetadata('position', 'Input', $assetInfo->getPicturePosition());
            foreach ($this->pictureTypesList->getPictureTypeConfig($assetInfo->getPictureType())->getMetaData() as $metaDataName => $metaDataValue) {
                $assetEvent->getAsset()->addMetadata($metaDataName, 'Input', $metaDataValue);
            }
        }
    }

    /**
     * @param $parentPath
     * @param $fileName
     *
     * @return Asset\Folder
     * @throws \Exception
     */
    private function getFolderCreateIfNotExists($parentPath, $fileName): Asset\Folder
    {
        $folder = Asset::getByPath($parentPath . '/' . $fileName);
        if (!$folder instanceof Asset\Folder) {
            $folder = (new Asset\Folder())
                ->setFilename($fileName)
                ->setPath($parentPath)
                ->setParent(Asset::getByPath($parentPath));
            $folder->getDao()->create();
            $folder->setType('folder');
            $folder->setCreationDate(microtime(false));
            $folder->setModificationDate(microtime(false));
            $folder->setUserOwner(1);
            $folder->setUserModification(1);
            $folder->setVersionCount(1);
            $folder->setHasMetaData(false);
            $folder->getDao()->update();
        }

        return $folder;
    }
}
