<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Pimcore\AssignProductImagesBundle\Command;

use Lifestyle\Pimcore\AssignProductImagesBundle\Configuration\ActiveConfig;
use Lifestyle\Pimcore\AssignProductImagesBundle\Services\MoveProcessingImages;
use Pimcore\Event\AssetEvents;
use Pimcore\Event\Model\AssetEvent;
use Pimcore\Model\Asset;
use Pimcore\Model\DataObject\AbstractObject;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Command\LockableTrait;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\EventDispatcher\EventDispatcher;

/**
 * Class ImageProcessingCommand
 *
 * @package Lifestyle\Pimcore\AssignProductImagesBundle\Command
 */
class ImageProcessingCommand extends Command
{
    const LOCK_ID = 'imageProcessingCommand';

    use LockableTrait;

    /**
     * @var string
     */
    protected static $defaultName = 'lifestyle:image:processing';

    /**
     * @var ActiveConfig
     */
    protected $activeConfig;
    /**
     * @var EventDispatcher
     */
    protected $eventDispatcher;
    /**
     * @var MoveProcessingImages
     */
    protected $imageMover;

    /**
     * @inheritDoc
     */
    public function __construct(
        ActiveConfig $activeConfig,
        EventDispatcher $eventDispatcher,
        MoveProcessingImages $moveProcessingImages
    ) {
        $this->activeConfig = $activeConfig;
        $this->eventDispatcher = $eventDispatcher;
        $this->imageMover = $moveProcessingImages;

        parent::__construct();
    }

    /**
     * @inheritDoc
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        if ($this->lock(self::LOCK_ID)) {
            AbstractObject::setHideUnpublished(false);
            $processingPath = Asset::getByPath('/' . $this->activeConfig->getProcessingPath());
            if (null !== $processingPath) {
                $this->handleAsset($processingPath, $output);
            }

            // release lock
            $this->release();
        }
    }

    /**
     * @param Asset $asset
     */
    protected function handleAsset(Asset $asset, OutputInterface $output)
    {
        foreach ($asset->getChildren() as $childAsset) {
            /** @var Asset $childAsset */
            $output->writeln($childAsset->getFilename());
            if ($childAsset instanceof Asset\Image) {
                /*
                 * image is moved (or content copied & deleted) to its target location
                 * image is saved after moving, so the update asset events are triggered for further processing
                 */
                $this->imageMover->move($childAsset);
            }

            if ($childAsset instanceof Asset\Folder) {
                $this->handleAsset($childAsset, $output);
                // reload folder to check if still items avaliable, if not delete
                $checkAsset = Asset::getById($childAsset->getId(), true);
                if (count($checkAsset->getChildren()) < 1 && $checkAsset->getPath() != '/') {
                    $checkAsset->delete();
                }
            }
        }
    }
}
