<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Pimcore\AssignProductImagesBundle\Services\AssetTypeHandler;

use Lifestyle\Pimcore\AssignProductImagesBundle\Configuration\PictureTypeConfig;
use Lifestyle\Pimcore\AssignProductImagesBundle\Model\AssetInfo;
use Pimcore\Event\Model\AssetEvent;
use Pimcore\Model\DataObject;
use Pimcore\Model\DataObject\Data\Hotspotimage;
use Pimcore\Model\Factory as ModelFactory;

/**
 * Class AssetTypeGalleryHandler
 *
 * @package Lifestyle\Pimcore\AssignProductImagesBundle\Services\AssetTypeHandler
 */
class AssetTypeGalleryHandler implements AssetTypeHandlerInterface
{
    /**
     * @var ModelFactory
     */
    protected $modelFactory;

    /**
     * AssetTypeGalleryHandler constructor.
     *
     * @param ModelFactory $modelFactory
     */
    public function __construct(ModelFactory $modelFactory)
    {
        $this->modelFactory = $modelFactory;
    }

    /**
     * @inheritDoc
     */
    public function process(AssetEvent $assetEvent, PictureTypeConfig $pictureTypeConfig): bool
    {
        $result = false;
        $assetInfo = new AssetInfo($assetEvent->getAsset());

        $modelObject = $this->getModelObject($pictureTypeConfig->getObjectName(), $assetInfo->getObjectKey());
        $pictureGalleryAccessor = 'get' . ucfirst($pictureTypeConfig->getPropertyName());

        if (false !== $modelObject
            && method_exists($modelObject, $pictureGalleryAccessor)
        ) {
            // rebuild gallery with positions
            $galleryImages = [];
            foreach ($modelObject->{$pictureGalleryAccessor}()->getItems() as $existingImage) {
                if (null !== $existingImage
                    && null !== $existingImage->getImage()
                ) {
                    $galleryImages[intval($existingImage->getImage()->getMetadata('position'))] = $existingImage;
                }
            }
            $galleryImages[intval($assetInfo->getPicturePosition())] = (new Hotspotimage())->setImage($assetEvent->getAsset());
            ksort($galleryImages);

            $modelObject->{$pictureGalleryAccessor}()->setItems($galleryImages);
            if (true === $pictureTypeConfig->isAutoPublish()) {
                $modelObject->setPublished(true);
            }
            $modelObject->save();

            $result = true;
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function processDelete(AssetEvent $assetEvent, PictureTypeConfig $pictureTypeConfig): bool
    {
        $result = false;
        $assetInfo = new AssetInfo($assetEvent->getAsset());

        $modelObject = $this->getModelObject($pictureTypeConfig->getObjectName(), $assetInfo->getObjectKey());
        $pictureGalleryAccessor = 'get' . ucfirst($pictureTypeConfig->getPropertyName());

        if (false !== $modelObject
            && method_exists($modelObject, $pictureGalleryAccessor)
        ) {
            // rebuild gallery with positions
            $galleryImages = [];
            foreach ($modelObject->{$pictureGalleryAccessor}()->getItems() as $existingImage) {
                if (null !== $existingImage) {
                    $galleryImages[intval($existingImage->getImage()->getMetadata('position'))] = $existingImage;
                }
            }

            // remove the image from gallery
            if (isset($galleryImages[intval($assetInfo->getPicturePosition())])
                && $galleryImages[intval($assetInfo->getPicturePosition())]->getImage()->getFilename() === $assetEvent->getAsset()->getFilename()
            ) {
                unset($galleryImages[intval($assetInfo->getPicturePosition())]);
            }

            ksort($galleryImages);

            $modelObject->{$pictureGalleryAccessor}()->setItems($galleryImages);
            $modelObject->save();

            $result = true;
        }

        return $result;
    }

    /**
     * @param $objectName
     * @param $objectKey
     *
     * @return mixed
     */
    private function getModelObject($objectName, $objectKey)
    {
        /** @var DataObject\Concrete $object */
        $object = $this->modelFactory
            ->build('Pimcore\\Model\\DataObject\\' . $objectName . '\\Listing')
            ->setCondition(
                'o_key IN (?, ?, ?)',
                [
                    trim($objectKey),
                    strtoupper(trim($objectKey)),
                    strtolower(trim($objectKey)),
                ]
            )
            ->current();

        $latestVersion = $object->getLatestVersion(true);
        if (null !== $latestVersion) {
            return $latestVersion->getData();
        }

        return $object;
    }
}
