<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Pimcore\AssignProductImagesBundle\Command;

use Lifestyle\Pimcore\AssignProductImagesBundle\Configuration\ActiveConfig;
use Lifestyle\Pimcore\AssignProductImagesBundle\Exception\ShutdownException;
use Lifestyle\Pimcore\AssignProductImagesBundle\Model\MaxItems;
use Lifestyle\Pimcore\AssignProductImagesBundle\Services\MoveProcessingImages;
use Pimcore\Model\Asset;
use Pimcore\Model\DataObject\AbstractObject;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Command\LockableTrait;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * Class ImageProcessingCommand
 *
 * @package Lifestyle\Pimcore\AssignProductImagesBundle\Command
 */
class ImageProcessingCommand extends Command
{
    use LockableTrait;

    private const LOCK_ID = 'imageProcessingCommand';

    /**
     * @var string
     */
    protected static $defaultName = 'lifestyle:image:processing';

    /**
     * @var ActiveConfig
     */
    protected $activeConfig;

    /**
     * @var MoveProcessingImages
     */
    protected $imageMover;

    /**
     * @inheritDoc
     */
    public function __construct(
        ActiveConfig $activeConfig,
        MoveProcessingImages $moveProcessingImages
    ) {
        $this->activeConfig = $activeConfig;
        $this->imageMover = $moveProcessingImages;

        parent::__construct();
    }

    /**
     * @inheritdoc
     */
    protected function configure()
    {
        $this->addOption(
            'limit',
            'l',
            InputOption::VALUE_OPTIONAL,
            'Size of Workpackages',
            500
        );
    }

    /**
     * @inheritDoc
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        if ($this->lock(self::LOCK_ID)) {
            if (true === $this->activeConfig->isProcessUnpublished()) {
                AbstractObject::setHideUnpublished(false);
            }

            $processingPath = Asset::getByPath('/' . $this->activeConfig->getProcessingPath());
            if (null !== $processingPath) {
                try {
                    $itemLimit = new MaxItems($input->getOption('limit'));
                    $this->handleAsset($processingPath, $output, $itemLimit);
                } catch (ShutdownException $shutdownException) {
                    $output->writeln('Max item limit reached. Process finished');
                }
            }

            // release lock
            $this->release();
        }
    }

    /**
     * @param Asset           $asset
     * @param OutputInterface $output
     * @param MaxItems        $itemLimit
     *
     * @throws ShutdownException
     */
    protected function handleAsset(Asset $asset, OutputInterface $output, MaxItems $itemLimit)
    {
        foreach ($asset->getChildren() as $childAsset) {
            /** @var Asset $childAsset */
            $output->writeln($childAsset->getFilename());
            if ($childAsset instanceof Asset\Image) {
                /*
                 * image is moved (or content copied & deleted) to its target location
                 * image is saved after moving, so the update asset events are triggered for further processing
                 */
                if (true === $this->imageMover->move($childAsset)) {
                    $itemLimit->decrease();
                }
            }

            if ($childAsset instanceof Asset\Folder) {
                $this->handleAsset($childAsset, $output, $itemLimit);
                // reload folder to check if still items avaliable, if not delete
                $checkAsset = Asset::getById($childAsset->getId(), true);
                if (count($checkAsset->getChildren()) < 1 && $checkAsset->getPath() != '/') {
                    $checkAsset->delete();
                }
                $checkAsset->setStream(null);
            }

            $childAsset->setStream(null);
            if (0 === $itemLimit->getCount()) {
                throw new ShutdownException();
            }
        }
    }
}
