<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Pimcore\AssignProductImagesBundle\Services;

use Lifestyle\Pimcore\AssignProductImagesBundle\Configuration\PictureTypesList;
use Lifestyle\Pimcore\AssignProductImagesBundle\Model\AssetInfo;
use Pimcore\Model\Asset;
use Pimcore\Model\Asset\Image;
use Pimcore\Model\Asset\Unknown;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\Factory as ModelFactory;

/**
 * Class MoveProcessingImages
 *
 * @package Lifestyle\Pimcore\AssignProductImagesBundle\Services
 */
class MoveProcessingImages
{
    /**
     * @var PictureTypesList
     */
    protected $pictureTypesList;
    /**
     * @var ModelFactory;
     */
    protected $modelFactory;

    /**
     * AddAssetListener constructor.
     *
     * @param PictureTypesList        $pictureTypesList
     * @param ModelFactory            $modelFactory
     */
    public function __construct(
        PictureTypesList $pictureTypesList,
        ModelFactory $modelFactory
    ) {
        $this->pictureTypesList = $pictureTypesList;
        $this->modelFactory = $modelFactory;
    }

    /**
     * @param Asset $asset
     *
     * @return bool
     * @throws \Exception
     */
    public function move(Asset $asset)
    {
        $assetInfo = new AssetInfo($asset);

        if ($this->pictureTypesList->hasPictureType($assetInfo->getPictureType())
            && ($asset instanceof Image || $asset instanceof Unknown)
        ) {
            $pictureTypeConfig = $this->pictureTypesList->getPictureTypeConfig($assetInfo->getPictureType());
            $modelObject = $this->modelFactory
                ->build('Pimcore\\Model\\DataObject\\' . $pictureTypeConfig->getObjectName() . '\\Listing')
                ->setCondition(
                    'o_key IN(?, ?, ?)',
                    [
                        trim($assetInfo->getObjectKey()),
                        strtoupper(trim($assetInfo->getObjectKey())),
                        strtolower(trim($assetInfo->getObjectKey())),
                    ]
                )
                ->current();

            if (is_object($modelObject) && $modelObject instanceof Concrete) {
                $latestVersion = $modelObject->getLatestVersion(true);
                if (null !== $latestVersion) {
                    $modelObject = $latestVersion->getData();
                }
            }

            if (is_object($modelObject) && $modelObject instanceof Concrete) {
                if (null === $asset->getParent()
                    || false !== stripos($asset->getFullPath(), '/' . $pictureTypeConfig->getProcessingPath())
                ) {
                    // build target path
                    $targetPath = '/';
                    $pathArray = [
                        $pictureTypeConfig->getTargetPath(),
                    ];

                    foreach ($pictureTypeConfig->getSubfolderConfig() as $subfolderConfig) {
                        $pathArray[] = $modelObject->{'get' . ucfirst($subfolderConfig)}();
                    }

                    foreach ($pathArray as $pathValue) {
                        $parentPath = $targetPath;
                        $targetPath .= $pathValue . '/';
                        $folder = $this->getFolderCreateIfNotExists($parentPath, $pathValue);
                    }

                    $existingAsset = Asset::getByPath($targetPath . $asset->getFilename());
                    if (null !== $existingAsset && $existingAsset instanceof Image) {
                        if ($asset->getId() !== $existingAsset->getId()) {
                            // change content
                            $existingAsset->setData($asset->getData());
                            $existingAsset->setDataChanged(true);
                            $existingAsset->save();
                            $existingAsset->setStream(null);

                            $asset->delete();

                            return true;
                        }
                    }

                    $asset->setParent($folder);
                    $asset->setPath($targetPath);
                    $asset->save();

                    return true;
                }
            }
        }

        return false;
    }

    /**
     * @param $parentPath
     * @param $fileName
     *
     * @return Asset\Folder
     * @throws \Exception
     */
    private function getFolderCreateIfNotExists($parentPath, $fileName): Asset\Folder
    {
        $folder = Asset::getByPath($parentPath . '/' . $fileName);
        if (!$folder instanceof Asset\Folder) {
            $folder = (new Asset\Folder())
                ->setFilename($fileName)
                ->setPath($parentPath)
                ->setParent(Asset::getByPath($parentPath));
            $folder->getDao()->create();
            $folder->setType('folder');
            $folder->setCreationDate(microtime(false));
            $folder->setModificationDate(microtime(false));
            $folder->setUserOwner(1);
            $folder->setUserModification(1);
            $folder->setVersionCount(1);
            $folder->setHasMetaData(false);
            $folder->getDao()->update();
        }

        return $folder;
    }
}
