<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace ExportArticleFamily\Tests\ValueMapper;

use ExportArticleFamily\Config\Client;
use ExportArticleFamily\Tests\TestBase;
use ExportArticleFamily\ValueMapper\NumberFormat;
use Pimcore\Model\Object\ArtikelFamilie;

/**
 * Test Value Mappers
 *
 * @package ExportArticleFamily\Tests
 */
class NumberFormatTest extends TestBase
{

    /**
     *
     */
    public function testMapDe()
    {

        $this->mapTest('de', 30.5, '30,5 mm');
    }

    /**
     * @param $language
     * @param $value
     * @param $expected
     */
    private function mapTest($language, $value, $expected)
    {

        $client = $this->getClient();
        $objectMapper = $this->getObjectMapper();
        $configProperty = $this->getConfigProperty();

        $formatter = new NumberFormat($this->apiM(), $client);

        $objectMapper->getObject()->setDatenblattWassersaeule($value);

        $formattedValue = $formatter->map($objectMapper, $configProperty, $language);

        $this->assertEquals($expected, $formattedValue);
    }

    /**
     * @return Client
     */
    private function getClient()
    {

        return $this->apiM()->clients()->configured()->first();
    }

    /**
     * @return \ExportArticleFamily\Mapper\ArtikelFamilie
     */
    private function getObjectMapper()
    {

        $datenblatt = new ArtikelFamilie();
        $datenblatt->setDatenblattDicke(1000);

        return $this->apiM()->mapper()->artikelFamilie($datenblatt);
    }

    /**
     *
     * @return \ExportArticleFamily\Config\Objects\Property
     */
    private function getConfigProperty()
    {

        $xml = <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<Property>
    <Id>datenblattWassersaeule</Id>
    <Label>__wassersaeule</Label>
    <Format>%s mm</Format>
    <Mapper>numberFormat</Mapper>
    <UnitValue>mm</UnitValue>
    <BaseUnit>mm</BaseUnit>
</Property>
XML;

        $property = $this->apiM()->serializer()->deserialize($xml, 'ExportArticleFamily\Config\Objects\Property', 'xml');

        return $property;
    }

    /**
     *
     */
    public function testMapEn()
    {

        $this->mapTest('en', 30.5, '30.5 mm');
    }

    /**
     *
     */
    public function testMapEs()
    {

        $this->mapTest('es', 30.5, '30,5 mm');
    }

    /**
     *
     */
    public function testMapFr()
    {

        $this->mapTest('fr', 30.5, '30,5 mm');
    }

    /**
     *
     */
    public function testMapIt()
    {

        $this->mapTest('it', 30.5, '30,5 mm');
    }

    /**
     *
     */
    public function testMapNl()
    {

        $this->mapTest('nl', 30.5, '30,5 mm');
    }

    /**
     *
     */
    public function testMapper()
    {

        $configProperty = $this->getConfigProperty();

        $this->assertEquals('numberFormat', $configProperty->getMapper());
    }

    /**
     *
     */
    public function testMultiMapDe()
    {

        $this->formatMapperTest('de', [8.111, 10.45], '8,111 10,45');
    }

    /**
     *
     */
    public function testMultiMapEn()
    {

        $this->formatMapperTest('en', [8.111, 10.45], '8.111 10.45');
    }

    /**
     * @param       $language
     * @param array $value
     * @param       $expected
     */
    public function formatMapperTest($language, array $value, $expected)
    {

        $client = $this->getClient();
        $objectMapper = $this->getObjectMapper();
        $configProperty = $this->getMultiConfigProperty();

        $formatter = new NumberFormat($this->apiM(), $client);

        $objectMapper->getObject()->setDatenblattInhalt($value[0]);
        $objectMapper->getObject()->setDatenblattInhaltEinheit($value[1]);

        $formattedValue = $formatter->map($objectMapper, $configProperty, $language);

        $this->assertEquals($expected, $formattedValue);
    }

    /**
     *
     * @return \ExportArticleFamily\Config\Objects\Property
     */
    private function getMultiConfigProperty()
    {

        $xml = <<<'XML'
<?xml version="1.0" encoding="UTF-8"?>
<Property>
    <Id>datenblattInhalt</Id>
    <Format>%s %s</Format>
    <Value>datenblattInhalt</Value>
    <Value>datenblattInhaltEinheit</Value>
</Property>
XML;

        $property = $this->apiM()->serializer()->deserialize($xml, 'ExportArticleFamily\Config\Objects\Property', 'xml');

        return $property;
    }
}
