<?php
/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportArticleFamily\ValueMapper;

use ExportArticleFamily\Api\Manager as ApiManager;
use ExportArticleFamily\Config\Client;
use ExportArticleFamily\Config\Objects\Property as ConfigProperty;
use ExportArticleFamily\Mapper\ArtikelFamilie as ObjectMapper;

/**
 * Unit mapper base class
 *
 * @package ExportArticleFamily\ValueMapper
 * @author  Oliver Friedrich <of@life-style.de>
 */
abstract class Base implements MapperInterface
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var array
     */
    protected $validUnits = array();

    /**
     * Power constructor.
     *
     * @param ApiManager $apiM
     * @param Client     $client
     */
    public function __construct(ApiManager $apiM, Client $client)
    {

        $this->apiM = $apiM;
        $this->client = $client;
    }

    /**
     * Get list of valid units
     *
     * @return array
     */
    public function getValidUnits()
    {

        return array_keys($this->validUnits);
    }

    /**
     * @param string $unit
     *
     * @return string
     * @throws InvalidUnitException
     */
    protected function getUnit($unit)
    {

        if (!isset($this->validUnits[$unit])) {
            throw new InvalidUnitException();
        }

        return $this->validUnits[$unit];
    }

    /**
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     *
     * @return string
     */
    protected function getObjectValue(ObjectMapper $objectMapper, ConfigProperty $property)
    {

        return $property->getValues()->isEmpty() ? '' : $this->getObjectProperty($objectMapper, $property->getValues()->first());
    }

    /**
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     *
     * @return string
     */
    protected function getObjectUnit(ObjectMapper $objectMapper, ConfigProperty $property)
    {

        return $property->getUnit() ? $this->getObjectProperty($objectMapper, $property->getUnit()) : $property->getUnitValue();
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param              $propertyName
     *
     * @return mixed
     * @throws InvalidPropertyException
     * @throws InvalidValueException
     */
    private function getObjectProperty(ObjectMapper $objectMapper, $propertyName)
    {

        // Property should not be empty
        if (!strlen($propertyName)) {
            throw new InvalidPropertyException('Cannot get object property! Property is empty. Please check your config.xml of ExportArticleFamily-plugin for missing unit/unitValue-tags.');
        }

        $value = $objectMapper->getValue($propertyName);

        // Value should not be an object
        if (is_object($value)) {
            throw new InvalidValueException('Cannot map object property! Value should be a scalar but is object for property ' . $propertyName);
        }

        return $value;
    }

    /**
     * @param string $value
     *
     * @return float|integer
     */
    protected function cleanUpValue($value)
    {

        return trim(str_replace(',', '.', preg_replace('/[^\d^,^\.]/', '', $value)), '.') + 0;
    }
}
