<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Publikat\Pimcore\ExportArticleFamilyBundle\ValueMapper;

use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Client;
use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Objects\Property as ConfigProperty;
use Publikat\Pimcore\ExportArticleFamilyBundle\Mapper\ArtikelFamilie as ObjectMapper;

/**
 * Class Temperature (°C, °F)
 *
 * @package Publikat\Pimcore\ExportArticleFamilyBundle\ValueMapper
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Temperature extends Base
{
    /**
     * @var string
     */
    private $baseUnit;

    /**
     * Temperature constructor.
     *
     * @param Client $client
     *
     * @throws InvalidUnitException
     */
    public function __construct(Client $client)
    {
        parent::__construct($client);
        $this->validUnits = array(
            '°C' => 'C',
            'C' => 'C',
            '°F' => 'F',
            'F' => 'F',
        );
        $this->baseUnit = $this->getUnit($client->getUnits()->getTemperature()->getBase());
    }

    /**
     * Map value
     *
     * Y = 9/5X +32 (Y= Fahrenheit; X = Celsius)
     * X = 5/9(Y - 32) (Y = Fahrenheit; X = Celsius)
     *
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     * @param string         $language
     *
     * @return float|int|mixed|string
     * @throws InvalidPropertyException
     * @throws InvalidUnitException
     * @throws InvalidValueException
     */
    public function map(ObjectMapper $objectMapper, ConfigProperty $property, $language)
    {
        $value = trim($this->getObjectValue($objectMapper, $property));

        // Do not map empty values
        if (!strlen($value)) {
            return '';
        }

        $cleanValue = $this->cleanUpValue($value);
        $cleanUnit = $this->getUnit($this->getObjectUnit($objectMapper, $property));

        if ($this->baseUnit == $cleanUnit) {
            return $cleanValue;
        }

        // Convert Celsius into Fahrenheit
        if ('F' == $this->baseUnit) {
            return 9 / 5 * $cleanValue + 32;
        }
        // Convert Fahrenheit into Celsius
        return 5 / 9 * ($cleanValue - 32);
    }
}
