<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Publikat\Pimcore\ExportArticleFamilyBundle\Webservice;

use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Client as ClientConfig;
use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Service as ServiceConfig;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Request\RequestInterface;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Response\ResponseInterface;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Response\Errors\Errors as ErrorResponse;
use Publikat\Pimcore\ExportArticleFamilyBundle\Pimcore\Config as ExportArticleFamilyPimcoreConfig;
use GuzzleHttp\Psr7\Response as WebserviceResponse;
use JMS\Serializer\Serializer;
use JMS\Serializer\SerializerBuilder;
use Pimcore\Log\ApplicationLogger;

/**
 * Webservice-Response
 *
 * @package Publikat\Pimcore\ExportArticleFamilyBundle\Webservice
 */
class Response
{
    /**
     * @var Serializer
     */
    protected $serializer;

    /**
     * @var ApplicationLogger
     */
    protected $logger;

    /**
     * @var ExportArticleFamilyPimcoreConfig
     */
    protected $exportArticleFamilyPimcoreConfig;

    /**
     * Response constructor.
     *
     * @param ApplicationLogger $logger
     */
    public function __construct(ApplicationLogger $logger, ExportArticleFamilyPimcoreConfig $exportArticleFamilyPimcoreConfig)
    {
        $this->logger = $logger;
        $this->exportArticleFamilyPimcoreConfig = $exportArticleFamilyPimcoreConfig;
        $this->serializer = SerializerBuilder::create()->build();
    }

    /**
     * Receive success response
     *
     * @param ClientConfig       $client
     * @param ServiceConfig      $service
     * @param RequestInterface   $request
     * @param WebserviceResponse $webserviceResponse
     *
     * @return ResponseInterface
     * @throws InvalidResponseException
     */
    public function receive(ClientConfig $client, ServiceConfig $service, RequestInterface $request, WebserviceResponse $webserviceResponse)
    {
        // Received an error
        if (200 != $webserviceResponse->getStatusCode()) {
            return $this->error($client, $service, $webserviceResponse);
        }

        // Invalid content type
        if (!$this->validateContentType($service->getContentType(), $webserviceResponse->getHeader('Content-Type')[0])) {
            $this->logger->debug('Response ' . $webserviceResponse->getBody()->getContents());
            throw new InvalidResponseException(sprintf(
                'Error while updating object: ' . PHP_EOL . 'Webservice-response should be type of %s but is %s',
                $service->getContentType(),
                $webserviceResponse->getHeader('Content-Type')[0]
            ));
        }

        // Receive body
        $body = $webserviceResponse->getBody()->getContents();

        $this->logger->info(sprintf('Object export: received success response for client "%s"', $client->getId()));
        $this->logger->debug($body);

        return $this->serializer->deserialize($body, $request->getResponseType(), $service->getFormat());
    }

    /**
     * Receive error response
     *
     * @param ClientConfig       $client
     * @param ServiceConfig      $service
     * @param WebserviceResponse $webserviceResponse
     *
     * @throws InvalidResponseException
     */
    private function error(ClientConfig $client, ServiceConfig $service, WebserviceResponse $webserviceResponse)
    {
        // Receive body
        $body = $webserviceResponse->getBody()->getContents();

        $this->logger->critical(sprintf('Object export: received error response status(%d) for client "%s"', $webserviceResponse->getStatusCode(), $client->getId()));
        $this->logger->error($body);

        // Invalid content type
        if (!$this->validateContentType($service->getContentType(), $webserviceResponse->getHeader('Content-Type')[0])) {
            throw new InvalidResponseException(sprintf(
                'Error while updating object: ' . PHP_EOL . 'Webservice-response should be %s but is %s',
                $service->getContentType(),
                $webserviceResponse->getHeader('Content-Type')[0]));
        }

        $response = $this->serializer->deserialize($body, 'Publikat\Pimcore\ExportArticleFamilyBundle\Model\Response\Errors\Errors', $service->getFormat());
        throw new InvalidResponseException('Error while updating objects: ' . PHP_EOL . implode(PHP_EOL, $this->getErrors($response)));
    }

    /**
     * Returns true, if content types matches
     *
     * @param string $contentTypeA
     * @param string $contentTypeB
     *
     * @return bool
     */
    private function validateContentType($contentTypeA, $contentTypeB)
    {
        return 0 === strcasecmp(preg_replace('/;.*$/', '', $contentTypeA), preg_replace('/;.*$/', '', $contentTypeB));
    }

    /**
     * @param ErrorResponse $response
     *
     * @return array
     * @throws \Exception
     */
    private function getErrors(ErrorResponse $response)
    {
        $errors = [];
        foreach ($response->getError() as $error) {

            // Add error message
            $errors[] = $error->getMessage();

            // Add additional information if in debug mode
            if ($this->exportArticleFamilyPimcoreConfig->isDebugMode()) {
                if ($error->getCode()) {
                    $errors[] = 'Error code: ' . $error->getCode();
                }
                if ($error->getDebug()) {
                    $errors[] = 'Debug message: ' . $error->getDebug();
                }
                if ($error->getException()) {
                    $errors[] = 'Exception message: ' . $error->getException();
                }
            }
        }

        if (!count($errors)) {
            $errors[] = 'An unknown error has occured in webservice-communication.';
        }
        return $errors;
    }
}
