<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Publikat\Pimcore\ExportArticleFamilyBundle\Webservice;

use GuzzleHttp\Client as GuzzleClient;
use JMS\Serializer\Serializer;
use JMS\Serializer\SerializerBuilder;
use MuleMessageBufferBundle\Service\MuleMessageBufferDispatcher;
use Pimcore\Log\ApplicationLogger;
use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Client as ClientConfig;
use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Service as ServiceConfig;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Request\RequestInterface;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Response\ResponseInterface;
use Publikat\Pimcore\ExportArticleFamilyBundle\Webservice\Response as Response;

/**
 * Webservice-Request
 *
 * @package Publikat\Pimcore\ExportArticleFamilyBundle\Webservice
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Request
{
    /**
     * @var Serializer
     */
    protected $serializer;

    /**
     * @var ApplicationLogger
     */
    protected $logger;

    /**
     * @var GuzzleClient
     */
    protected $guzzleClient;

    /**
     * @var \Publikat\Pimcore\ExportArticleFamilyBundle\Webservice\Response
     */
    protected $response;

    /**
     * @var MuleMessageBufferDispatcher|null
     */
    protected $muleDispatcher;

    /**
     * Request constructor.
     *
     * @param ApplicationLogger                                               $logger
     * @param GuzzleClient                                                    $guzzleClient
     * @param \Publikat\Pimcore\ExportArticleFamilyBundle\Webservice\Response $response
     * @param MuleMessageBufferDispatcher|null                                $muleDispatcher
     */
    public function __construct(ApplicationLogger $logger, GuzzleClient $guzzleClient, Response $response, ?MuleMessageBufferDispatcher $muleDispatcher)
    {
        $this->logger = $logger;
        $this->guzzleClient = $guzzleClient;
        $this->response = $response;
        $this->muleDispatcher = $muleDispatcher;
        $this->serializer = SerializerBuilder::create()->build();
    }

    /**
     * @param RequestInterface $request
     * @param ClientConfig     $client
     * @param ServiceConfig    $service
     *
     * @return ResponseInterface
     * @throws \Exception
     */
    public function send(RequestInterface $request, ClientConfig $client, ServiceConfig $service)
    {
        // Prepare request
        $data = $this->serializer->serialize($request->getRequest(), $service->getFormat());
        $options = [
            'headers' => [
                'Accept' => $service->getAcceptContentType(),
                'Content-Type' => $service->getContentType()
            ],
            'body' => $data
        ];

        $this->logger->info(sprintf('Object export: sending request to client "%s" url %s', $client->getId(), $service->getUrl()));
        $this->logger->debug($options['body']);

        // Send request
        try {
            $webserviceResponse = $this->guzzleClient->post($service->getUrl(), $options);
        } catch (\GuzzleHttp\Exception\RequestException $exception) {
            $this->logger->error($exception->getMessage());
            // Received an error response
            return $this->response->receive($client, $service, $request, $exception->getResponse());
        } catch (\GuzzleHttp\Exception\ClientException $exception) {
            $this->logger->error($exception->getMessage());
            // Received an error response
            return $this->response->receive($client, $service, $request, $exception->getResponse());
        }

        if (null !== $this->muleDispatcher) {
            $this->muleDispatcher->dispatch($data, 'ArticleFamily');
        } else {
            $this->logger->warning('Mule Message Buffer service is not available, Data is not put on the Buffer.');
        }

        // Received response
        return $this->response->receive($client, $service, $request, $webserviceResponse);
    }
}
