<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author    Franz Weisflug
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Publikat\Pimcore\ExportArticleFamilyBundle\Worker;

use MuleMessageBufferBundle\Service\MuleMessageBufferDispatcher;
use Pimcore\Event\Model\DataObjectEvent;
use Pimcore\Log\ApplicationLogger;
use Pimcore\Model\Object\ArtikelFamilie;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Mule\ArticleFamilyChange;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * Class Mule
 *
 * @package Publikat\Pimcore\ExportArticleFamilyBundle\Worker
 */
class Mule
{
    /**
     * @var SerializerInterface
     */
    protected $serializer;

    /**
     * @var null|MuleMessageBufferDispatcher
     */
    private $muleDispatcher;

    /**
     * @var ApplicationLogger
     */
    protected $logger;

    /**
     * Mule constructor.
     *
     * @param SerializerInterface              $serializer
     * @param MuleMessageBufferDispatcher|null $muleDispatcher
     * @param ApplicationLogger                $logger
     */
    public function __construct(
        SerializerInterface $serializer,
        ?MuleMessageBufferDispatcher $muleDispatcher,
        ApplicationLogger $logger
    ) {
        $this->serializer = $serializer;
        $this->muleDispatcher = $muleDispatcher;
        $this->logger = $logger;
    }

    public function run(DataObjectEvent $event)
    {
        if (null === $this->muleDispatcher) {
            $this->logger->debug('Mule Message Buffer not available, can not send Familientext changes');
            return;
        }

        /** @var ArtikelFamilie $object */
        $object = $event->getObject();
        if (!$object->isPublished() || $event->hasArgument('saveVersionOnly')) {
            return;
        }

        /** @var ArtikelFamilie $previousObject */
        $versions = $object->getVersions();
        $versionCount = count($versions);
        $previousVersion = $versions[$versionCount - 2];
        $previousObject = $previousVersion === null ? null : $previousVersion->getData();
        $articleFamilyChange = (new ArticleFamilyChange())
            ->setArticleFamilyId($object->getDisplayCode())
            ->setTextAvailable(!empty($object->getArtikelFamilieLangtextStylefile('de')));

        // check for previous version, if none assume text changed
        if ($versionCount < 2 || null === $previousObject) {
            $this->dispatchMuleEvent($articleFamilyChange);
            return;
        }

        if ($object->getArtikelFamilieLangtextStylefile('de') !== $previousObject->getArtikelFamilieLangtextStylefile('de')) {
            $this->dispatchMuleEvent($articleFamilyChange);
            return;
        }
    }

    /**
     * @param ArticleFamilyChange $eventData
     */
    private function dispatchMuleEvent(ArticleFamilyChange $eventData)
    {
        $data = $this->serializer->serialize($eventData, 'json');
        $this->logger->debug('Dispatching ArticleFamilyChange Mule Message Buffer data: ' . $data);
        $this->muleDispatcher->dispatch($data, 'ArticleFamilyChange');
    }
}
