<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Publikat\Pimcore\ExportArticleFamilyBundle\ValueMapper;

use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Client;
use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Objects\Property as ConfigProperty;
use Publikat\Pimcore\ExportArticleFamilyBundle\Mapper\ArtikelFamilie as ObjectMapper;

/**
 * Class Mass unit mapper (g, kg, t)
 *
 * @package Publikat\Pimcore\ExportArticleFamilyBundle\ValueMapper
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Mass extends Base
{
    /**
     * @var string
     */
    private $baseUnit;

    /**
     * Mass constructor.
     *
     * @param Client $client
     *
     * @throws InvalidUnitException
     */
    public function __construct(Client $client)
    {
        parent::__construct($client);
        $this->validUnits = array(
            'mg' => 'mg',
            'g' => 'g',
            'kg' => 'kg',
            't' => 't',
            'lb' => 'lb',
        );
        $this->baseUnit = $this->getUnit($client->getUnits()->getMass()->getBase());
    }

    /**
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     * @param string         $language
     *
     * @return float|int|mixed|string
     * @throws InvalidPropertyException
     * @throws InvalidUnitException
     * @throws InvalidValueException
     */
    public function map(ObjectMapper $objectMapper, ConfigProperty $property, $language)
    {
        $value = trim($this->getObjectValue($objectMapper, $property));

        // Do not map empty values
        if (!strlen($value)) {
            return '';
        }

        $cleanValue = $this->cleanUpValue($value);
        $cleanUnit = $this->getUnit($this->getObjectUnit($objectMapper, $property));

        if ($this->baseUnit == $cleanUnit) {
            return $cleanValue;
        }
        return $this->scaleToBase($this->scaleToDefault($cleanValue, $cleanUnit));
    }

    /**
     * @param float|int $value
     * @param string    $unit
     *
     * @return float|int
     */
    private function scaleToDefault($value, $unit)
    {
        switch ($unit) {
            case 'mg':
                $value /= 1000;
                break;
            default:
            case 'g':
                break;
            case 'kg':
                $value *= 1000;
                break;
            case 't':
                $value *= 1000000;
                break;
            case 'lb':
                $value *= 453.59237;
                break;
        }
        return $value;
    }

    /**
     * @param float|int $value
     *
     * @return float|int
     */
    private function scaleToBase($value)
    {
        switch ($this->baseUnit) {
            case 'mg':
                $value *= 1000;
                break;
            default:
            case 'g':
                break;
            case 'kg':
                $value /= 1000;
                break;
            case 't':
                $value /= 1000000;
                break;
            case 'lb':
                $value /= 453.59237;
                break;
        }
        return $value;
    }
}
