<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Publikat\Pimcore\ExportArticleFamilyBundle\Worker;

use Publikat\Pimcore\ExportArticleFamilyBundle\Config\Client;
use Publikat\Pimcore\ExportArticleFamilyBundle\Client\Manager as ClientManager;
use Publikat\Pimcore\ExportArticleFamilyBundle\Mapper\ArtikelFamilie as ObjectMapper;
use Publikat\Pimcore\ExportArticleFamilyBundle\Mapper\Factory as MapperFactory;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Request\Objects\Update\Factory as ObjectsUpdateFactory;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Request\Objects\Update\Item;
use Publikat\Pimcore\ExportArticleFamilyBundle\Model\Request\Objects\Update\Request;
use Publikat\Pimcore\ExportArticleFamilyBundle\ValueMapper\MapperInterface as ValueMapper;
use Publikat\Pimcore\ExportArticleFamilyBundle\ValueMapper\Factory as ValueMapperFactory;
use Publikat\Pimcore\ExportArticleFamilyBundle\Webservice\Request as WebserviceRequest;
use Publikat\Pimcore\ExportArticleFamilyBundle\Pimcore\Config as ExportArticleFamilyPimcoreConfig;
use Publikat\Pimcore\ExportArticleFamilyBundle\Translate\Translator as TranslateTranslator;
use Pimcore\Model\Object\ArtikelFamilie;
use Pimcore\Model\Object\Fieldcollection;
use Pimcore\Event\Model\DataObjectEvent;
use Pimcore\Log\ApplicationLogger;
use JMS\Serializer\Serializer;
use JMS\Serializer\SerializerBuilder;

/**
 * Class Update
 *
 * @author  Jochen Califice <jc@life-style.de>
 * @package Publikat\Pimcore\ExportArticleFamilyBundle\Worker
 */
class Update
{
    /**
     * @var ApplicationLogger
     */
    protected $logger;

    /**
     * @var MapperFactory
     */
    protected $mapperFactory;

    /**
     * @var ClientManager
     */
    protected $clientManager;

    /**
     * @var ObjectsUpdateFactory
     */
    protected $objectsUpdateFactory;

    /**
     * @var WebserviceRequest
     */
    protected $webserviceRequest;

    /**
     * @var ExportArticleFamilyPimcoreConfig
     */
    protected $exportArticleFamilyPimcoreConfig;

    /**
     * @var TranslateTranslator
     */
    protected $translateTranslator;

    /**
     * @var ValueMapperFactory
     */
    protected $valueMapperFactory;

    /**
     * @var \Publikat\Pimcore\ExportArticleFamilyBundle\Config\PluginConfig
     */
    protected $pluginConfig;

    /**
     * Update constructor.
     *
     * @param ApplicationLogger                $logger
     * @param MapperFactory                    $mapperFactory
     * @param ClientManager                    $clientManager
     * @param ObjectsUpdateFactory             $objectsUpdateFactory
     * @param WebserviceRequest                $webserviceRequest
     * @param ExportArticleFamilyPimcoreConfig $exportArticleFamilyPimcoreConfig
     * @param TranslateTranslator              $translateTranslator
     * @param ValueMapperFactory               $valueMapperFactory
     * @param string                           $configPath
     */
    public function __construct(
        ApplicationLogger $logger,
        MapperFactory $mapperFactory,
        ClientManager $clientManager,
        ObjectsUpdateFactory $objectsUpdateFactory,
        WebserviceRequest $webserviceRequest,
        ExportArticleFamilyPimcoreConfig $exportArticleFamilyPimcoreConfig,
        TranslateTranslator $translateTranslator,
        ValueMapperFactory $valueMapperFactory,
        string $configPath
    )
    {
        $this->logger = $logger;
        $this->mapperFactory = $mapperFactory;
        $this->clientManager = $clientManager;
        $this->objectsUpdateFactory = $objectsUpdateFactory;
        $this->webserviceRequest = $webserviceRequest;
        $this->exportArticleFamilyPimcoreConfig = $exportArticleFamilyPimcoreConfig;
        $this->translateTranslator = $translateTranslator;
        $this->valueMapperFactory = $valueMapperFactory;

        /*
         * Tried to auto-wire Serializer without creating instance using SerializerBuilder.
         * Didn't get it wo work, so we decided to use static way again...
         */
        $serializer = SerializerBuilder::create()->build();
        $this->pluginConfig = $serializer->deserialize(
            file_get_contents($configPath),
            'Publikat\Pimcore\ExportArticleFamilyBundle\Config\PluginConfig',
            'xml'
        );
    }

    /**
     * @param DataObjectEvent $event
     *
     * @throws \Publikat\Pimcore\ExportArticleFamilyBundle\Config\InvalidConfigException
     */
    public function run(DataObjectEvent $event)
    {
        $object = $event->getObject();

        // Object not published
        if (!$object->isPublished()) {
            $this->logger->debug(
                sprintf('Object "%s" (ID %d) not exported: not published', $object->getKey(),
                    $object->getId())
            );
            return;
        }

        // STYLEFILE-5486 : Dont' run export if only save new version is triggered within Pimcore
        if ($event->hasArgument('saveVersionOnly')) {
            return;
        }

        // Initialize object mapper
        $mapper = $this->mapperFactory->artikelFamilie($object);
        foreach ($this->clientManager->active() as $client) {

            // Prepare request
            $modelRequest = $this->objectsUpdateFactory->request();

            // Prepare data
            $this->update($client, $modelRequest, $mapper);

            $this->logger->debug(
                sprintf(
                    'Object "%s" (ID %d) added to update request',
                    $object->getKey(),
                    $object->getId()
                )
            );

            // Send request
            $service = $this->clientManager->service($client, 'Objects::Update');
            $this->webserviceRequest->send($modelRequest, $client, $service);
        }
    }

    /**
     * Update datasheet
     *
     * Prepare each attribute in each language even if it is an attribute
     * which is language independent. This makes live easier on the other site.
     *
     * @param Client       $client
     * @param Request      $request
     * @param ObjectMapper $objectMapper
     *
     * @throws \Exception
     */
    private function update(Client $client, Request $request, ObjectMapper $objectMapper)
    {
        $item = $this->objectsUpdateFactory->item($request);
        $item->setProductId($objectMapper->getObject()->getKey());

        // Cycle through configured languages
        foreach ($this->exportArticleFamilyPimcoreConfig->getLanguages() as $language) {

            // Add language
            $item->addDocumentLanguage($language);

            // Switch language
            $this->translateTranslator->setLocale($language);

            // Add properties
            foreach ($objectMapper->getObjectProperties() as $propertyName) {
                if ('localizedfields' == $propertyName) {
                    // Localized fields
                    foreach ($objectMapper->getLocalizedValues($language) as $key => $value) {
                        if (!empty($value)) {
                            $this->addLocalizedValue($objectMapper, $item, $key, $value, $language);
                        }
                    }
                } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Object\Fieldcollection')) {
                    // Field collection
                    $this->addFieldcollectionValues($objectMapper, $item, $objectMapper->getValue($propertyName),
                        $language);
                } else {
                    // Plain value
                    $this->addValue($client, $objectMapper, $item, $propertyName, $language);
                }
            }
        }
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param              $key
     * @param              $value
     * @param              $language
     *
     * @throws \Exception
     */
    private function addLocalizedValue(ObjectMapper $objectMapper, Item $item, $key, $value, $language)
    {
        $shopGroup = $this->determineShopGroup($key, 'localizedfields');

        $this->objectsUpdateFactory->nameFieldList($item)
            ->setName($key)
            ->setLanguage($language)
            ->setLabelId($objectMapper->mapNameId($key))
            ->setLabel($this->translateTranslator->trans($objectMapper->mapNameId($key)))
            ->setValue($value)
            ->setShopGroup($shopGroup);
    }

    /**
     * @param ObjectMapper    $objectMapper
     * @param Item            $item
     * @param Fieldcollection $fieldCollection
     * @param                 $language
     *
     * @throws \Exception
     */
    private function addFieldcollectionValues(ObjectMapper $objectMapper, Item $item, Fieldcollection $fieldCollection, $language) {
        $shopGroup = $this->determineShopGroup($fieldCollection->getFieldname(), 'fieldcollection');

        // Add field collection
        foreach ($fieldCollection->getItems() as $fieldCollectionData) {
            // Field collection mapper
            $fieldCollectionMapper = $this->mapperFactory->fieldCollection($fieldCollectionData);

            // Field collection data - item
            foreach ($fieldCollectionMapper->getValues() as $key => $values) {
                // prevent export of empty single item stuff
                if (!is_array($values) && strlen($values) == 0) {
                    continue;
                }
                $nameFieldList = $this->objectsUpdateFactory->nameFieldList($item)
                    ->setName($key)
                    ->setLanguage($language)
                    ->setLabelId($objectMapper->mapNameId($key))
                    ->setLabel($this->translateTranslator->trans($objectMapper->mapNameId($key)))
                    ->setShopGroup($shopGroup);
                if (is_array($values)) {
                    // Each item in field collection data may have multiple values selected
                    foreach ($values as $value) {
                        $this->objectsUpdateFactory->valueItem($nameFieldList)
                            ->setValueId($value)
                            ->setValue($this->translateTranslator->trans($value));
                    }
                } else {
                    $nameFieldList->setValue($this->translateTranslator->trans($values))
                        ->setValueId($values);
                }
            }
        }
    }

    /**
     * assign and flag shop specific properties (stylefile, bigtree)
     *
     * @param $fieldName
     * @param $type
     *
     * @return string
     */
    private function determineShopGroup($fieldName, $type)
    {
        if ($fieldName == 'datenblattAuswahllisten') {
            return 'BIGT';
        }

        if (strpos($fieldName, 'Stylefile') !== false) {
            return 'STYLEF';
        }

        if (strpos($fieldName, 'Bigtree') !== false) {
            return 'BIGT';
        }

        if (strpos($fieldName, 'BulletPoints') === 0) {
            return '';
        }

        if ($type == 'localizedfields') {
            return 'BIGT';
        }
        return '';
    }

    /**
     * @param Client       $client
     * @param ObjectMapper $objectMapper
     * @param Item         $item
     * @param              $propertyName
     * @param              $language
     *
     * @throws \Exception
     */
    private function addValue(Client $client, ObjectMapper $objectMapper, Item $item, $propertyName, $language)
    {
        $shopGroup = $this->determineShopGroup($propertyName, 'value');
        $propertyMapped = false;

        foreach ($this->pluginConfig->getObjects()->getArtikelFamilie()->getProperties() as $property) {
            // Property does not match
            if ($property->getId() != $propertyName) {
                continue;
            }

            // Ignore property
            if ($property->isDisabled()) {
                return;
            }

            // Map value
            $value = trim($this->valueMapper($client, $property->getMapper())->map($objectMapper, $property,
                $language));

            // Skip empty value
            if (!strlen($value)) {
                continue;
            }

            // Add node
            $this->objectsUpdateFactory->nameFieldList($item)
                ->setName($propertyName)
                ->setLanguage($language)
                ->setLabelId($property->getLabelId())
                ->setLabel($this->translateTranslator->trans($property->getLabel()))
                ->setValue($this->translateTranslator->trans($value))
                ->setShopGroup($shopGroup);

            // Mark property as mapped
            $propertyMapped = true;
        }

        // Passthrough not mapped value
        if (!$propertyMapped && strlen($value = trim($objectMapper->getValue($propertyName)))) {
            $this->objectsUpdateFactory->nameFieldList($item)
                ->setName($propertyName)
                ->setLanguage($language)
                ->setLabelId($propertyName)
                ->setLabel($this->translateTranslator->trans($objectMapper->mapNameId($propertyName)))
                ->setValue($this->translateTranslator->trans($value))
                ->setShopGroup($shopGroup);
        }
    }

    /**
     * @param Client $client
     * @param        $mapper
     *
     * @return ValueMapper
     */
    private function valueMapper(Client $client, $mapper)
    {
        return $this->valueMapperFactory->$mapper($client);
    }
}
