<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportArticleFamily\ValueMapper;

use ExportArticleFamily\Api\Manager as ApiManager;
use ExportArticleFamily\Config\Client;
use ExportArticleFamily\Config\Objects\Property as ConfigProperty;
use ExportArticleFamily\Mapper\ArtikelFamilie as ObjectMapper;

/**
 * Class Area unit mapper (m^2, qm)
 *
 * @package ExportArticleFamily\ValueMapper
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Area extends Base
{
    /**
     * @var string
     */
    private $baseUnit;

    /**
     * Mass constructor.
     *
     * @param ApiManager $apiM
     * @param Client     $client
     */
    public function __construct(ApiManager $apiM, Client $client)
    {

        parent::__construct($apiM, $client);
        $this->validUnits = array(
            'mm2' => 'qmm',
            'mm^2' => 'qmm',
            'mm²' => 'qmm',
            'qmm' => 'qmm',
            'm2' => 'qm',
            'm^2' => 'qm',
            'm²' => 'qm',
            'qm' => 'qm',
            'km2' => 'qkm',
            'km^2' => 'qkm',
            'km²' => 'qkm',
            'qkm' => 'qkm',
        );
        $this->baseUnit = $this->getUnit($client->getUnits()->getArea()->getBase());
    }

    /**
     * Map value
     *
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     * @param string         $language
     *
     * @return mixed
     * @throws InvalidUnitException
     */
    public function map(ObjectMapper $objectMapper, ConfigProperty $property, $language)
    {

        $value = trim($this->getObjectValue($objectMapper, $property));

        // Do not map empty values
        if (!strlen($value)) {
            return '';
        }

        $cleanValue = $this->cleanUpValue($value);
        $cleanUnit = $this->getUnit($this->getObjectUnit($objectMapper, $property));
        $baseUnit = $property->getBaseUnit() ? $this->getUnit($property->getBaseUnit()) : $this->baseUnit;

        if ($baseUnit == $cleanUnit) {
            return $cleanValue;
        }

        return $this->scaleToBase($this->scaleToDefault($cleanValue, $cleanUnit), $baseUnit);
    }

    /**
     * @param float|int $value
     * @param string    $unit
     *
     * @return float|int
     */
    private function scaleToDefault($value, $unit)
    {

        switch ($unit) {
            case 'qmm':
                $value /= 1000000;
                break;
            default:
            case 'qm':
                break;
            case 'qkm':
                $value *= 1000000;
                break;
        }

        return $value;
    }

    /**
     * @param float|int $value
     * @param string    $baseUnit
     *
     * @return float|int
     */
    private function scaleToBase($value, $baseUnit)
    {

        switch ($baseUnit) {
            case 'qmm':
                $value *= 1000000;
                break;
            default:
            case 'qm':
                break;
            case 'qkm':
                $value /= 1000000;
                break;
        }

        return $value;
    }
}
