<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportArticleFamily\ValueMapper;

use ExportArticleFamily\Api\Manager as ApiManager;
use ExportArticleFamily\Config\Client;
use ExportArticleFamily\Config\Objects\Property as ConfigProperty;
use ExportArticleFamily\Mapper\ArtikelFamilie as ObjectMapper;

/**
 * Class Voltage unit mapper (mAh, Ah)
 *
 * @package ExportArticleFamily\ValueMapper
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Voltage extends Base
{

    /**
     * @var string
     */
    private $baseUnit;

    /**
     * Power constructor.
     *
     * @param ApiManager $apiM
     * @param Client     $client
     */
    public function __construct(ApiManager $apiM, Client $client)
    {

        parent::__construct($apiM, $client);
        $this->validUnits = array(
            'mV' => 'mV',
            'V' => 'V',
            'kV' => 'kV',
        );
        $this->baseUnit = $this->getUnit($client->getUnits()->getVoltage()->getBase());
    }

    /**
     * Map value
     *
     * @param ObjectMapper   $objectMapper
     * @param ConfigProperty $property
     * @param string         $language
     *
     * @return mixed
     * @throws InvalidUnitException
     */
    public function map(ObjectMapper $objectMapper, ConfigProperty $property, $language)
    {

        $value = trim($this->getObjectValue($objectMapper, $property));

        // Do not map empty values
        if (!strlen($value)) {
            return '';
        }

        $cleanValue = $this->cleanUpValue($value);
        $cleanUnit = $this->getUnit($this->getObjectUnit($objectMapper, $property));

        if ($this->baseUnit == $cleanUnit) {
            return $cleanValue;
        }

        return $this->scaleToBase($this->scaleToDefault($cleanValue, $cleanUnit));
    }

    /**
     * @param float|int $value
     * @param string    $unit
     *
     * @return float|int
     */
    private function scaleToDefault($value, $unit)
    {

        switch ($unit) {
            case 'mV':
                $value /= 1000;
                break;
            default:
            case 'V':
                break;
            case 'kV':
                $value *= 1000;
                break;
        }

        return $value;
    }

    /**
     * @param float|int $value
     *
     * @return float|int
     */
    private function scaleToBase($value)
    {

        switch ($this->baseUnit) {
            case 'mV':
                $value *= 1000;
                break;
            default:
            case 'AV':
                break;
            case 'kV':
                $value /= 1000;
                break;
        }

        return $value;
    }
}
