<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace ExportArticleFamily\Worker;

use ExportArticleFamily\Api\Manager as ApiManager;
use ExportArticleFamily\Config\Client;
use ExportArticleFamily\Mapper\ArtikelFamilie as ObjectMapper;
use ExportArticleFamily\Model\Request\Objects\Update\Item;
use ExportArticleFamily\Model\Request\Objects\Update\Request;
use ExportArticleFamily\ValueMapper\MapperInterface as ValueMapper;
use Pimcore\Model\Object\ArtikelFamilie;
use Pimcore\Model\Object\Fieldcollection;

/**
 * Update-Worker
 *
 * @package ExportArticleFamily\Worker
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Update
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {

        $this->apiM = $apiM;
    }

    /**
     * @param \Zend_EventManager_Event $event
     *
     * @throws \ExportArticleFamily\Config\InvalidConfigException
     * @throws \ReflectionException
     */
    public function run(\Zend_EventManager_Event $event)
    {
        $object = $event->getTarget();

        // Object not published
        if (!$object->isPublished()) {
            $this->apiM->logger()->debug(sprintf('Object "%s" (ID %d) not exported: not published', $object->getKey(),
                $object->getId()));
            return;
        }

        // STYLEFILE-5486 : Dont' run export if only save new version is triggered within Pimcore
        if (true === $event->getParam('saveVersionOnly')) {
            return;
        }

        // Initialize object mapper
        $mapper = $this->apiM->mapper()->artikelFamilie($object);

        foreach ($this->apiM->clients()->active() as $client) {
            // Prepare request
            $modelRequest = $this->apiM->model()->objectsUpdate()->request();

            // Prepare data
            $this->update($client, $modelRequest, $mapper);

            $this->apiM->logger()->debug(sprintf('Object "%s" (ID %d) added to update request', $object->getKey(),
                $object->getId()));

            // Send request
            $service = $this->apiM->clients()->service($client, 'Objects::Update');

            $this->apiM->webservice()->request()->send($modelRequest, $client, $service);
        }
    }

    /**
     * Update datasheet
     *
     * Prepare each attribute in each language even if it is an attribute
     * which is language independent. This makes live easier on the other site.
     *
     * @param Client       $client
     * @param Request      $request
     * @param ObjectMapper $objectMapper
     *
     * @throws \ReflectionException
     */
    private function update(Client $client, Request $request, ObjectMapper $objectMapper)
    {
        $item = $this->apiM->model()->objectsUpdate()->item($request);
        $item->setProductId($objectMapper->getObject()->getKey());

        // Cycle through configured languages
        foreach ($this->apiM->pimcore()->config()->getLanguages() as $language) {
            // Add language
            $item->addDocumentLanguage($language);

            // Switch language
            $this->apiM->translate()->translator()->setLocale($language);

            // Add properties
            foreach ($objectMapper->getObjectProperties() as $propertyName) {
                if ('localizedfields' == $propertyName) {
                    // Localized fields
                    foreach ($objectMapper->getLocalizedValues($language) as $key => $value) {
                        if (!empty($value)) {
                            $this->addLocalizedValue($objectMapper, $item, $key, $value, $language);
                        }
                    }
                } elseif (is_a($objectMapper->getValue($propertyName), '\Pimcore\Model\Object\Fieldcollection')) {
                    // Field collection
                    $this->addFieldcollectionValues($objectMapper, $item, $objectMapper->getValue($propertyName),
                        $language);
                } else {
                    // Plain value
                    $this->addValue($client, $objectMapper, $item, $propertyName, $language);

                }
            }
        }
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item $item
     * @param string $key
     * @param string $value
     * @param string $language
     */
    private function addLocalizedValue(ObjectMapper $objectMapper, Item $item, $key, $value, $language)
    {
        $shopGroup = $this->determineShopGroup($key, 'localizedfields');

        $this->apiM->model()->objectsUpdate()->nameFieldList($item)
            ->setName($key)
            ->setLanguage($language)
            ->setLabelId($objectMapper->mapNameId($key))
            ->setLabel($this->apiM->translate()->translator()->trans($objectMapper->mapNameId($key)))
            ->setValue($value)
            ->setShopGroup($shopGroup);
    }

    /**
     * @param ObjectMapper $objectMapper
     * @param Item $item
     * @param Fieldcollection $fieldCollection
     * @param string $language
     */
    private function addFieldcollectionValues(ObjectMapper $objectMapper, Item $item, Fieldcollection $fieldCollection, $language) {
        $translator = $this->apiM->translate()->translator();

        $shopGroup = $this->determineShopGroup($fieldCollection->getFieldname(), 'fieldcollection');

        // Add field collection
        foreach ($fieldCollection->getItems() as $fieldCollectionData) {
            // Field collection mapper
            $fieldCollectionMapper = $this->apiM->mapper()->fieldCollection($fieldCollectionData);

            // Field collection data - item
            foreach ($fieldCollectionMapper->getValues() as $key => $values) {
                // prevent export of empty single item stuff
                if (!is_array($values) && strlen($values) == 0) {
                    continue;
                }
                $nameFieldList = $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                    ->setName($key)
                    ->setLanguage($language)
                    ->setLabelId($objectMapper->mapNameId($key))
                    ->setLabel($translator->trans($objectMapper->mapNameId($key)))
                    ->setShopGroup($shopGroup);
                if (is_array($values)) {
                    // Each item in field collection data may have multiple values selected
                    foreach ($values as $value) {
                        $this->apiM->model()->objectsUpdate()->valueItem($nameFieldList)
                            ->setValueId($value)
                            ->setValue($translator->trans($value));
                    }
                } else {
                    $nameFieldList->setValue($translator->trans($values))
                        ->setValueId($values);
                }
            }
        }
    }

    /**
     * assign and flag shop specific properties (stylefile, bigtree)
     *
     * @param $fieldName
     * @param $type
     *
     * @return string
     */
    private function determineShopGroup($fieldName, $type)
    {
        if ($fieldName == 'datenblattAuswahllisten') {
            return 'BIGT';
        }

        if (strpos($fieldName, 'Stylefile') !== false) {
            return 'STYLEF';
        }

        if (strpos($fieldName, 'Bigtree') !== false) {
            return 'BIGT';
        }

        if (strpos($fieldName, 'BulletPoints') === 0) {
            return '';
        }

        if ($type == 'localizedfields') {
            return 'BIGT';
        }

        return '';
    }

    /**
     * @param Client $client
     * @param ObjectMapper $objectMapper
     * @param Item $item
     * @param string $propertyName
     * @param string $language
     */
    private function addValue(Client $client, ObjectMapper $objectMapper, Item $item, $propertyName, $language)
    {

        $shopGroup = $this->determineShopGroup($propertyName, 'value');

        $translator = $this->apiM->translate()->translator();
        $propertyMapped = false;

        foreach ($this->apiM->pluginConfig()->getObjects()->getArtikelFamilie()->getProperties() as $property) {
            // Property does not match
            if ($property->getId() != $propertyName) {
                continue;
            }

            // Ignore property
            if ($property->isDisabled()) {
                return;
            }

            // Map value
            $value = trim($this->valueMapper($client, $property->getMapper())->map($objectMapper, $property,
                $language));

            // Skip empty value
            if (!strlen($value)) {
                continue;
            }

            // Add node
            $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                ->setName($propertyName)
                ->setLanguage($language)
                ->setLabelId($property->getLabelId())
                ->setLabel($translator->trans($property->getLabel()))
                ->setValue($translator->trans($value))
                ->setShopGroup($shopGroup);

            // Mark property as mapped
            $propertyMapped = true;
        }

        // Passthrough not mapped value
        if (!$propertyMapped && strlen($value = trim($objectMapper->getValue($propertyName)))) {
            $this->apiM->model()->objectsUpdate()->nameFieldList($item)
                ->setName($propertyName)
                ->setLanguage($language)
                ->setLabelId($propertyName)
                ->setLabel($translator->trans($objectMapper->mapNameId($propertyName)))
                ->setValue($translator->trans($value))
                ->setShopGroup($shopGroup);
        }
    }

    /**
     * @param Client $client
     * @param        $mapper
     *
     * @return ValueMapper
     */
    private function valueMapper(Client $client, $mapper)
    {
        return $this->apiM->valuemapper()->$mapper($client);
    }
}
