<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Tests\ExportObjectsToStepsRestApi\Database;

use ExportObjectsToStepsRestApi\Api\Manager as ApiManager;
use ExportObjectsToStepsRestApi\Database\ProductRegister;
use ExportObjectsToStepsRestApi\Entity\Factory;

/**
 * Class ConfigTest
 * @package Tests\ExportObjectsToStepsRestApi\Database
 */
class ProductRegisterTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var ApiManager
     */
    private $apiManager;

    protected function setUp()
    {
        $this->apiManager = $this
            ->getMockBuilder(ApiManager::class)
            ->setMethods(['entityFactory'])
            ->getMock();
        $this->apiManager->expects($this->any())
            ->method('entityFactory')
            ->willReturn(new Factory($this->apiManager));
    }

    public function testFindByObjectIdReturnsNull()
    {
        $dbConnectionMock = $this->getMockBuilder(\Zend_Db_Adapter_Mysqli::class)
            ->disableOriginalConstructor()
            ->setMethods(['fetchRow'])
            ->getMock();
        $dbConnectionMock
            ->expects($this->once())
            ->method('fetchRow')
            ->willReturn(false);

        $productRegister = new ProductRegister($this->apiManager, $dbConnectionMock);
        $this->assertNull($productRegister->findByClientAndObjectId(1234, 'SampleObjectID'));
    }

    public function testFindByObjectIdReturnsEntity()
    {
        $dbConnectionMock = $this->getMockBuilder(\Zend_Db_Adapter_Mysqli::class)
            ->disableOriginalConstructor()
            ->setMethods(['fetchRow'])
            ->getMock();
        $dbConnectionMock
            ->expects($this->once())
            ->method('fetchRow')
            ->willReturn([
                'objectId' => 1234,
                'clientId' => '12345',
                'published' => true,
                'deleted' => false,
                'deletedAt' => null,
            ]);

        $productRegister = new ProductRegister($this->apiManager, $dbConnectionMock);
        $object = $productRegister->findByClientAndObjectId(12345, 'SampleObjectID');
        $this->assertSame(1234, $object->getObjectId());
        $this->assertSame('12345', $object->getClientId());
        $this->assertTrue($object->isPublished());
        $this->assertFalse($object->isDeleted());
        $this->assertNull($object->getDeletedAt());
    }

    public function testAddEntityFromArrayDoNothing()
    {
        $dbConnectionMock = $this->getMockBuilder(\Zend_Db_Adapter_Mysqli::class)
            ->disableOriginalConstructor()
            ->setMethods(['insert'])
            ->getMock();
        $dbConnectionMock
            ->expects($this->never())
            ->method('insert')
            ->with(['horst']);

        $productRegister = new ProductRegister($this->apiManager, $dbConnectionMock);
        $productRegister->addEntityFromArray([]);
    }

    public function testAddEntityFromArray()
    {
        $addArray = [
            'id' => 'test',
            'name' => 'test',
        ];

        $dbConnectionMock = $this->getMockBuilder(\Zend_Db_Adapter_Mysqli::class)
            ->disableOriginalConstructor()
            ->setMethods(['insert'])
            ->getMock();
        $dbConnectionMock
            ->expects($this->once())
            ->method('insert')
            ->with('ProductRegister', $addArray);

        $productRegister = new ProductRegister($this->apiManager, $dbConnectionMock);
        $productRegister->addEntityFromArray($addArray);
    }

    public function testSaveEntityDoNothing()
    {
        $dbConnectionMock = $this->getMockBuilder(\Zend_Db_Adapter_Mysqli::class)
            ->disableOriginalConstructor()
            ->setMethods(['update'])
            ->getMock();
        $dbConnectionMock
            ->expects($this->never())
            ->method('update');

        $productRegister = new ProductRegister($this->apiManager, $dbConnectionMock);
        $productRegister->saveEntity('12345', false, []);
    }

    public function testSaveEntityNoRowFoundDoNothing()
    {
        $dbConnectionMock = $this->getMockBuilder(\Zend_Db_Adapter_Mysqli::class)
            ->disableOriginalConstructor()
            ->setMethods(['update', 'fetchRow'])
            ->getMock();
        $dbConnectionMock
            ->expects($this->never())
            ->method('update');
        $dbConnectionMock
            ->expects($this->once())
            ->method('fetchRow')
            ->willReturn(null);

        $productRegister = new ProductRegister($this->apiManager, $dbConnectionMock);
        $productRegister->saveEntity('12345', '1234', ['someData' => true]);
    }

    public function testSaveEntity()
    {
        $updateData = ['someData' => true];

        $dbConnectionMock = $this->getMockBuilder(\Zend_Db_Adapter_Mysqli::class)
            ->disableOriginalConstructor()
            ->setMethods(['update', 'fetchRow'])
            ->getMock();
        $dbConnectionMock
            ->expects($this->once())
            ->method('fetchRow')
            ->willReturn(['objectId' => '1234', 'clientId' => '12345']);
        $dbConnectionMock
            ->expects($this->once())
            ->method('update')
            ->with('ProductRegister', $updateData, array('clientId=?' => '12345', 'objectId=?' => '1234'));

        $productRegister = new ProductRegister($this->apiManager, $dbConnectionMock);
        $productRegister->saveEntity('12345', '1234', $updateData);
    }
}
