<?php

/**
 * ExportObjectsToRestApi
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Tests\ExportObjectsToStepsRestApi\Services\Product\Delete;

use ExportObjectsToStepsRestApi\Config\Client;
use ExportObjectsToStepsRestApi\Config\Service;
use ExportObjectsToStepsRestApi\Database\ProductRegister;
use ExportObjectsToStepsRestApi\Services\Product\Delete\RequestQueue\Sender;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use JMS\Serializer\Serializer;
use Pimcore\Log\ApplicationLogger as Logger;
use GuzzleHttp\Client as WebserviceClient;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\Request;
use Pimcore\Model\Element\ValidationException;
use Pimcore\Model\Object\Product;

/**
 * Class SenderTest
 * @package Tests\ExportObjectsToStepsRestApi\Services\Product\Delete
 */
class SenderTest extends \PHPUnit_Framework_TestCase
{
    public function testSendSuccess()
    {
        $guzzleMock = $this->getMockedGuzzle([
            new Response(200, [], 'SuccessMessage'),
        ]);

        $serializerMock = $this->getMockBuilder(Serializer::class)
            ->disableOriginalConstructor()
            ->getMock();

        $productRegisterRepo = $this->createProductRegisterRepoMock();
        $productRegisterRepo->expects($this->once())
            ->method('saveEntity');

        $loggerMock = $this->createLoggerMock();
        $client = $this->createClientConfig();
        $service = $this->createServiceConfig();

        $product = new Product();
        $product->setId(12345);

        $sender = new Sender($loggerMock, $serializerMock, $guzzleMock, $productRegisterRepo);
        $this->assertSame('SuccessMessage', $sender->send($client, $service, $product));
    }

    public function testSendCatchClientException()
    {
        $this->expectException(ValidationException::class);

        $guzzleMock = $this->getMockedGuzzle([
            new ClientException(
                'errorMessage',
                new Request('UNIT_TEST', 'test'),
                new Response(400, [], '{"code":400,"scriptTimeSec": 13,"trackId":"00000","errors":[{"code":"404","message":"ErrorCode: 404 Message: Resource not found for ID UNKNOWN-PRODUCT-ID!","exception":"external"}]}')
            ),
        ]);

        $productRegisterRepo = $this->createProductRegisterRepoMock();
        $loggerMock = $this->createLoggerMock();
        $client = $this->createClientConfig();
        $service = $this->createServiceConfig();

        $product = new Product();
        $product->setId(12345);

        $sender = new Sender($loggerMock, \JMS\Serializer\SerializerBuilder::create()->build(), $guzzleMock, $productRegisterRepo);
        $sender->send($client, $service, $product);
    }

    public function testSendCatchConnectException()
    {
        $excMsg = 'Default Exception';

        $this->expectException(ValidationException::class);
        $this->expectExceptionMessage($excMsg);

        $guzzleMock = $this->getMockedGuzzle([
            new ConnectException($excMsg, new Request('UNIT_TEST', 'test')),
        ]);

        $productRegisterRepo = $this->createProductRegisterRepoMock();
        $loggerMock = $this->createLoggerMock();
        $client = $this->createClientConfig();
        $service = $this->createServiceConfig();

        $product = new Product();
        $product->setId(12345);

        $sender = new Sender($loggerMock, \JMS\Serializer\SerializerBuilder::create()->build(), $guzzleMock, $productRegisterRepo);
        $sender->send($client, $service, $product);
    }

    public function testSendCatchDefaultException()
    {
        $excMsg = 'Default Exception';

        $this->expectException(ValidationException::class);
        $this->expectExceptionMessage($excMsg);

        $guzzleMock = $this->getMockedGuzzle([
            new \Exception($excMsg),
        ]);

        $productRegisterRepo = $this->createProductRegisterRepoMock();
        $loggerMock = $this->createLoggerMock();
        $client = $this->createClientConfig();
        $service = $this->createServiceConfig();

        $product = new Product();
        $product->setId(12345);

        $sender = new Sender($loggerMock, \JMS\Serializer\SerializerBuilder::create()->build(), $guzzleMock, $productRegisterRepo);
        $sender->send($client, $service, $product);
    }

    /**
     * @return Client
     */
    private function createClientConfig()
    {
        return new Client([
            'apiUserName' => 'api_unit_test_user',
            'apiKey' => 'api_unit_test_key',
        ]);
    }

    /**
     * @return Service
     */
    private function createServiceConfig()
    {
        return new Service([
            'url' => 'https://steps-rest-ws/api/products/{productId}.json',
            'method' => 'UNIT_TEST',
            'apiVersion' => 1,
        ]);
    }

    /**
     * @return \PHPUnit_Framework_MockObject_MockObject
     */
    private function createLoggerMock()
    {
        return
            $this->getMockBuilder(Logger::class)
                ->disableOriginalConstructor()
                ->getMock();

    }

    /**
     * @return \PHPUnit_Framework_MockObject_MockObject
     */
    private function createProductRegisterRepoMock()
    {
        return
            $this->getMockBuilder(ProductRegister::class)
                ->disableOriginalConstructor()
                ->setMethods(['saveEntity'])
                ->getMock();
    }

    /**
     * @param array $responses
     * @return WebserviceClient
     */
    private function getMockedGuzzle($responses)
    {
        // Create a mock and queue responses.
        $mock = new MockHandler($responses);

        $handler = HandlerStack::create($mock);
        return new WebserviceClient(['handler' => $handler]);
    }
}
