<?php

/**
 * ExportObjectsToStepsRestApi
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportObjectsToStepsRestApi\Services\Product\Add;

use ExportObjectsToStepsRestApi\Api\Manager as ApiManager;
use Pimcore\Model\Object\Product;
use ExportObjectsToStepsRestApi\Services\Product\Add\Model\Request\Request;
use ExportObjectsToStepsRestApi\Mapper\Product as ObjectMapper;
use ExportObjectsToStepsRestApi\Config\Client;
use Pimcore\Model\Object\AbstractObject;

/**
 * Class Handler
 * @package ExportObjectsToStepsRestApi\Services\Product\Add
 */
class Handler
{
    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * Handler constructor.
     * @param ApiManager $apiManager
     */
    public function __construct(ApiManager $apiManager)
    {
        $this->apiManager = $apiManager;
    }

    /**
     * @param Product $object
     */
    public function init(Product $object)
    {
        // Object not published
        if (!$object->isPublished()) {
            $this->apiManager->logger()->info(sprintf('Object "%s" (ID %d) not exported: not published', $object->getKey(), $object->getId()));
            return;
        }

        // Initialize object mapper
        $mapper = $this->apiManager->mapper()->product($object);

        foreach ($this->apiManager->clients()->active() as $client) {

            // Prepare request
            $modelRequest = $this->apiManager->services()->product()->add()->model()->request()->request();

            // Prepare data
            $this->add($client, $modelRequest, $mapper);

            $this->apiManager->logger()->info(sprintf('Object "%s" (ID %d) added to add request', $object->getKey(), $object->getId()));

            // Send request
            $service = $this->apiManager->clients()->service($client, 'Objects::Add');
            $this->apiManager->webservice()->request()->send($modelRequest, $client, $service);
            $this->apiManager->database()->productRegister()->addEntityFromArray(['clientId' => $client->getId(), 'objectId' => $object->getId(), 'published' => $object->isPublished()]);
        }
    }

    /**
     * @param Client $client
     * @param Request $request
     * @param ObjectMapper $objectMapper
     */
    private function add(Client $client, Request $request, ObjectMapper $objectMapper)
    {
        $item = $this->apiManager->services()->product()->add()->model()->request()->item();
        $item
            ->setProductId($objectMapper->getObject()->getId())
            ->setErpId($objectMapper->getValue('productId'))
            ->setCategoryName(ucfirst(strtolower($objectMapper->getValue('productGroup'))))
            ->setPrice($objectMapper->getValue('productPrice'))
            ->setVat(19);

        $brandObject = $this->getNextParentObject($objectMapper);
        if (null !== $brandObject && 'Brand' == $brandObject->getClassName()) {
            $item->setBrandName($brandObject->getBrandName());
        }

        foreach ($client->getLanguages() as $language) {
            $localizedValues = $objectMapper->getLocalizedValues($language->getId());
            if (count($localizedValues) <= 0) {
                // Required fields missing! Export not possible for this language id!
                continue;
            }
            $localizedField = $this->apiManager->services()->product()->add()->model()->request()->localizedField();
            $localizedField
                ->setLangId($language->getId())
                ->setTitle($localizedValues['productTitle'])
                ->setDescription(substr($localizedValues['productDescription'], 0, 999))
                ->setShortDescription($localizedValues['productDescriptionShort']);
            $item->addLocalizedField($localizedField);
        }

        $request->addItem($item);
    }

    /**
     * Pimcore code not worked before - I just fixed it
     *
     * @return AbstractObject|null
     */
    public function getNextParentObject(ObjectMapper $objectMapper)
    {
        if ($objectMapper->getObject()->getParent() instanceof AbstractObject) {
            $parent = $objectMapper->getObject()->getParent();
            while ($parent && $parent->getType() == "folder") {
                $parent = $parent->getParent();
            }

            if ($parent && ($parent->getType() == "object" || $parent->getType() == "variant")) {
                return $parent;
            }
        }

        return null;
    }
}
