<?php

/**
 * ExportObjectsToStepsRestApi
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportObjectsToStepsRestApi\Services\Product\Delete\RequestQueue;

use ExportObjectsToStepsRestApi\Config\Client;
use ExportObjectsToStepsRestApi\Config\Service;
use ExportObjectsToStepsRestApi\Database\ProductRegister;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use JMS\Serializer\Serializer;
use Pimcore\Log\ApplicationLogger;
use Pimcore\Model\Element\ValidationException;
use ExportObjectsToStepsRestApi\Services\Product\Delete\Model\Response\Error\Response;
use GuzzleHttp\Client as WebserviceClient;
use Pimcore\Model\Object\Product;

/**
 * Class Sender
 * @package ExportObjectsToStepsRestApi\Services\Product\Delete\RequestQueue
 */
class Sender
{
    /**
     * @var ApplicationLogger
     */
    private $logger;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var WebserviceClient
     */
    private $client;

    /**
     * Sender constructor.
     * @param ApplicationLogger $logger
     * @param Serializer $serializer
     * @param WebserviceClient $client
     * @param ProductRegister $productRegister
     */
    public function __construct(ApplicationLogger $logger, Serializer $serializer, WebserviceClient $client, ProductRegister $productRegister)
    {
        $this->logger = $logger;
        $this->serializer = $serializer;
        $this->client = $client;
        $this->productRegisterRepo = $productRegister;
    }

    /**
     * @param Client $client
     * @param Service $service
     * @return string
     * @throws ValidationException
     */
    public function send(Client $client, Service $service, Product $product)
    {
        $this->logger->info(sprintf('Object export: sending request to client "%s" url %s', $client->getId(), $service->getUrl()));

        // Send request
        try {
            $webserviceResponse = $this->client->request($service->getMethod(), $service->getUrl(), $this->getOptions($client, $service));
        } catch (ClientException $exception) {
            $errorResponse = $exception->getResponse()->getBody()->getContents();
            $this->logger->critical(sprintf('Object export: received error response for client "%s": %s', $client->getId(), $errorResponse));
            $this->handleErrorResponse($errorResponse);
        } catch (ConnectException $exception) {
            $this->logger->critical(sprintf('Connect Exception for client "%s": %s', $client->getId(), $exception->getMessage()));
            throw new ValidationException($exception->getMessage());
        } catch (\Exception $exception) {
            $this->logger->critical(sprintf('Unknown Exception occurred for client "%s": %s', $client->getId(), $exception->getMessage()));
            throw new ValidationException($exception->getMessage());
        }

        // Receive response
        $body = $webserviceResponse->getBody()->getContents();
        $this->logger->info(sprintf('Object export: received success response for client "%s"', $client->getId()));
        $this->logger->debug($body);

        $this->saveProductRegisterEntity($client->getId(), $product->getId());

        return $body;
    }

    private function saveProductRegisterEntity($clientId, $productId)
    {
        $this->productRegisterRepo->saveEntity($clientId, $productId, [
            'published' => false,
            'deleted' => true,
            'deletedAt' => date('Y-m-d H:i:s', time()),
        ]);
    }

    /**
     * @param Client $client
     * @param Service $service
     * @return array
     */
    private function getOptions(Client $client, Service $service)
    {
        return [
            'headers' => [
                'X-username' => $client->getApiUserName(),
                'X-api-key' => $client->getApiKey(),
                'X-api-version' => $service->getApiVersion(),
            ],
        ];
    }

    /**
     * @param string $errorResponse
     * @throws ValidationException
     */
    private function handleErrorResponse($errorResponse)
    {
        /** @var Response $errorResponseModel */
        $errorResponseModel = $this->serializer->deserialize(
            $errorResponse, 'ExportObjectsToStepsRestApi\Services\Product\Delete\Model\Response\Error\Response', 'json'
        );

        $baseMessage = 'Error while sending objects to steps rest webservice(' . $errorResponseModel->getCode() . ')! ErrorToken: ' . $errorResponseModel->getTrackId();

        $errorMessages = [];
        foreach ($errorResponseModel->getErrors() as $error) {
            $errorMessages[] = ' Code: ' . $error->getCode() . ' Message: ' . $error->getMessage();
        }

        throw new ValidationException($baseMessage . implode('; ', $errorMessages));
    }
}
