<?php

/**
 * ExportObjectsToStepsRestApi
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportObjectsToStepsRestApi\Services\Product\Update;

use ExportObjectsToStepsRestApi\Api\Manager as ApiManager;
use Pimcore\Model\Object\Product;
use ExportObjectsToStepsRestApi\Services\Product\Update\Model\Request\Request;
use ExportObjectsToStepsRestApi\Mapper\Product as ObjectMapper;
use ExportObjectsToStepsRestApi\Config\Client;
use Pimcore\Model\Object\AbstractObject;
use Ramsey\Uuid\Uuid;

/**
 * Class Handler
 * @package ExportObjectsToStepsRestApi\Services\Product\Update
 */
class Handler
{
    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * Handler constructor.
     * @param ApiManager $apiManager
     */
    public function __construct(ApiManager $apiManager)
    {
        $this->apiManager = $apiManager;
    }

    /**
     * @param Product $object
     */
    public function init(Product $object)
    {
        // Send update request for published and unpublished objects!
        $this->apiManager->logger()->info(sprintf('Export Object "%s" (ID %d): published: %d', $object->getKey(), $object->getId(), $object->isPublished()));

        // Initialize object mapper
        $mapper = $this->apiManager->mapper()->product($object);

        foreach ($this->apiManager->clients()->active() as $client) {

            $productRegEntity = $this->apiManager->database()->productRegister()->findByClientAndObjectId($client->getId(), $object->getId());
            if (null === $productRegEntity) {
                \Pimcore::getEventManager()->trigger('stepsbox.add', $object);
                return;
            }

            if (!$productRegEntity->isPublished() && !$object->isPublished()) {
                $this->apiManager->logger()->info(sprintf('Export skipped for Object "%s" (ID %d): It is already unpublished.', $object->getKey(), $object->getId()));
                return;
            }

            // Prepare request
            $modelRequest = $this->apiManager->services()->product()->update()->model()->request()->request();

            // Prepare data
            $this->update($client, $modelRequest, $mapper);

            $this->apiManager->logger()->info(sprintf('Object "%s" (ID %d) added to update request', $object->getKey(), $object->getId()));

            // Send request
            $service = $this->apiManager->clients()->service($client, 'Objects::Update');
            $this->apiManager->webservice()->request()->send($modelRequest, $client, $service);
            $this->apiManager->database()->productRegister()->saveEntity($client->getId(), $object->getId(), [
                'published' => $object->isPublished(),
                'updatedAt' => date('Y-m-d H:i:s', time()),
            ]);
        }
    }

    /**
     * @param Client $client
     * @param Request $request
     * @param ObjectMapper $objectMapper
     */
    private function update(Client $client, Request $request, ObjectMapper $objectMapper)
    {
        $item = $this->apiManager->services()->product()->update()->model()->request()->item();
        $item
            ->setProductId($objectMapper->getObject()->getId())
            ->setErpId($objectMapper->getValue('productId'))
            ->setCategoryName(ucfirst(strtolower($objectMapper->getValue('productGroup'))))
            ->setPrice($objectMapper->getValue('productPrice'))
            ->setVat(19)
            ->setPublished($objectMapper->getObject()->isPublished());

        $brandObject = $this->getNextParentObject($objectMapper);
        if (null !== $brandObject && 'Brand' == $brandObject->getClassName()) {
            $item->setBrandName($brandObject->getBrandName());
        }

        foreach ($client->getLanguages() as $language) {
            $localizedValues = $objectMapper->getLocalizedValues($language->getId());
            if (count($localizedValues) <= 0) {
                // Required fields missing! Export not possible for this language id - right?!
                continue;
            }
            $localizedField = $this->apiManager->services()->product()->update()->model()->request()->localizedField();
            $localizedField
                ->setLangId($language->getId())
                ->setTitle($localizedValues['productTitle'])
                ->setDescription($localizedValues['productDescription'])
                ->setShortDescription($localizedValues['productDescriptionShort']);
            $item->addLocalizedField($localizedField);
        }

        $request->addItem($item);
    }

    /**
     * Pimcore code not worked before - I just fixed it
     *
     * @return AbstractObject|null
     */
    public function getNextParentObject(ObjectMapper $objectMapper)
    {
        if ($objectMapper->getObject()->getParent() instanceof AbstractObject) {
            $parent = $objectMapper->getObject()->getParent();
            while ($parent && $parent->getType() == "folder") {
                $parent = $parent->getParent();
            }

            if ($parent && ($parent->getType() == "object" || $parent->getType() == "variant")) {
                return $parent;
            }
        }

        return null;
    }
}