<?php

/**
 * PimCore Export Objects Plugin
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace ExportObjectsToStepsRestApi\Webservice;

use ExportObjectsToStepsRestApi\Api\Manager as ApiManager;
use ExportObjectsToStepsRestApi\Config\Client as ClientConfig;
use ExportObjectsToStepsRestApi\Config\Service as ServiceConfig;
use ExportObjectsToStepsRestApi\Services\Model\Request\RequestInterface;
use ExportObjectsToStepsRestApi\Services\Model\Response\ResponseInterface;
use GuzzleHttp\Psr7\Response as WebserviceResponse;
use Pimcore\Model\Element\ValidationException;

/**
 * Webservice-Response
 *
 * @package ExportObjectsToStepsRestApi\Webservice
 * @author  Oliver Friedrich <of@life-style.de>
 */
class Response
{
    /**
     * Plugin api manager
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     * Constructor
     *
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {

        $this->apiM = $apiM;
    }

    /**
     * Receive success response
     *
     * @param ClientConfig       $client
     * @param ServiceConfig      $service
     * @param RequestInterface   $request
     * @param WebserviceResponse $webserviceResponse
     *
     * @return ResponseInterface
     * @throws InvalidResponseException
     */
    public function receive(ClientConfig $client, ServiceConfig $service, RequestInterface $request, WebserviceResponse $webserviceResponse)
    {
        // Invalid content type
        if (!$this->validateContentType($service->getContentType(), $webserviceResponse->getHeader('Content-Type')[0])) {
            $this->apiM->logger()->debug('Response ' . $webserviceResponse->getBody()->getContents());
            throw new InvalidResponseException(sprintf('Error while updating object: ' . PHP_EOL . 'Webservice-response should be type of %s but is %s', $service->getContentType(), $webserviceResponse->getHeader('Content-Type')[0]));
        }

        // Receive body
        $body = $webserviceResponse->getBody()->getContents();

        $this->apiM->logger()->info(sprintf('Object export: received success response for client "%s"', $client->getId()));
        $this->apiM->logger()->debug($body);

        $responseModel = $this->apiM->serializer()->deserialize($body, $request->getResponseType(), $service->getFormat());
        foreach ($responseModel->getItem() as $item) {
            if (200 != $item->getCode()) {
                $errorMessages = [];
                foreach ($item->getErrors()->getError() as $error) {
                    $errorMessages[] =  $error->getCode(). ' - ' . $error->getMessage() . ' ErrorToken: ' . $item->getErrors()->getTrackId();
                }
                throw new ValidationException('Error while updating object: ' . implode('; ', $errorMessages));
            }
        }

        return $responseModel;
    }

    /**
     * Returns true, if content types matches
     *
     * @param string $contentTypeA
     * @param string $contentTypeB
     *
     * @return bool
     */
    private function validateContentType($contentTypeA, $contentTypeB)
    {
        return 0 === strcasecmp(preg_replace('/;.*$/', '', $contentTypeA), preg_replace('/;.*$/', '', $contentTypeB));
    }
}
