<?php


/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2016 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */


namespace ExportObjects\Tests;


use ExportObjects\Api\Manager as ApiManager;
use ExportObjects\Model\Response\Errors\Error;
use ExportObjects\Model\Response\Errors\Errors;
use GuzzleHttp\Client;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use Pimcore\Model\Object\Datenblatt;

/**
 * Base class for unit testing
 *
 * @package ExportObjects\Tests
 */
abstract class TestBase extends \PHPUnit_Framework_TestCase
{
    /**
     * @var ApiManager
     */
    private $apiM;

    /**
     * @var MockHandler
     */
    private $webserviceHandler;

    /**
     * @return ApiManager
     */
    protected function apiM()
    {

        return $this->apiM;
    }

    /**
     * @return null|Datenblatt
     */
    protected function getDatenblattPublished()
    {

        $dataSheetObject = new Datenblatt();
        $dataSheetObject->setPublished(true);

        return $dataSheetObject;
    }

    /**
     * @return null|Datenblatt
     */
    protected function getDatenblattUnpublished()
    {

        $dataSheetObject = new Datenblatt();
        $dataSheetObject->setPublished(false);

        return $dataSheetObject;
    }

    /**
     * Sets up Test
     */
    protected function setUp()
    {

        parent::setUp();
        $this->apiM = new ApiManager();
    }

    /**
     * Add response to mocked webservice client
     *
     * @param int    $statusCode
     * @param array  $headers
     * @param string $body
     *
     * @throws \Exception
     */
    protected function webserviceAddResponse($statusCode = 200, $headers = array(), $body = null)
    {

        if (null === $this->webserviceHandler) {
            throw new \Exception('Cannot set webservice response: No webservice client initialized. Create webserviceBuildApiM first!');
        }
        $this->webserviceHandler->append(new Response($statusCode, $headers, $body));
    }

    /**
     * Create new api manager with mocked webservice-client
     *
     * @return ApiManager
     */
    protected function webserviceBuildApiM()
    {

        // Mock api manager
        $apiManagerMock = $this->getMockBuilder('\ExportObjects\Api\Manager')
            ->setMethods(array('webservice'))
            ->getMock();

        // Mock webservice manager
        $webserviceManagerMock = $this->getMockBuilder('\ExportObjects\Webservice\Manager')
            ->setConstructorArgs(array($apiManagerMock))
            ->setMethods(array('client'))
            ->getMock();

        // Create a mock
        $this->webserviceHandler = new MockHandler();
        $client = new Client(['handler' => HandlerStack::create($this->webserviceHandler)]);

        // Set mock methods
        $webserviceManagerMock->method('client')->willReturn($client);
        $apiManagerMock->method('webservice')->willReturn($webserviceManagerMock);

        return $apiManagerMock;
    }

    /**
     * Create error response body for webservice
     *
     * @param ApiManager $apiM
     *
     * @return string
     */
    protected function webserviceCreateError(ApiManager $apiM, $format = 'json')
    {

        $error = new Error();
        $error
            ->setCode(400)
            ->setDebug('Debug-Message')
            ->setException('Exception-Message');
        $errors = new Errors();
        $errors->setStatus(400);
        $errors->addError($error);

        return $apiM->serializer()->serialize($errors, $format);
    }
}
